/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5MMprivate.h" 

#define H5EA_HDR_VERSION    0 
#define H5EA_IBLOCK_VERSION 0 
#define H5EA_SBLOCK_VERSION 0 
#define H5EA_DBLOCK_VERSION 0 

static herr_t H5EA__cache_hdr_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5EA__cache_hdr_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5EA__cache_hdr_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5EA__cache_hdr_image_len(const void *thing, size_t *image_len);
static herr_t H5EA__cache_hdr_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5EA__cache_hdr_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5EA__cache_hdr_free_icr(void *thing);

static herr_t H5EA__cache_iblock_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5EA__cache_iblock_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5EA__cache_iblock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5EA__cache_iblock_image_len(const void *thing, size_t *image_len);
static herr_t H5EA__cache_iblock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5EA__cache_iblock_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5EA__cache_iblock_free_icr(void *thing);

static herr_t H5EA__cache_sblock_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5EA__cache_sblock_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5EA__cache_sblock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5EA__cache_sblock_image_len(const void *thing, size_t *image_len);
static herr_t H5EA__cache_sblock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5EA__cache_sblock_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5EA__cache_sblock_free_icr(void *thing);

static herr_t H5EA__cache_dblock_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5EA__cache_dblock_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5EA__cache_dblock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5EA__cache_dblock_image_len(const void *thing, size_t *image_len);
static herr_t H5EA__cache_dblock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5EA__cache_dblock_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5EA__cache_dblock_free_icr(void *thing);
static herr_t H5EA__cache_dblock_fsf_size(const void *thing, hsize_t *fsf_size);

static herr_t H5EA__cache_dblk_page_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5EA__cache_dblk_page_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5EA__cache_dblk_page_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5EA__cache_dblk_page_image_len(const void *thing, size_t *image_len);
static herr_t H5EA__cache_dblk_page_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5EA__cache_dblk_page_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5EA__cache_dblk_page_free_icr(void *thing);

const H5AC_class_t H5AC_EARRAY_HDR[1] = {{
    H5AC_EARRAY_HDR_ID,                    
    "Extensible Array Header",             
    H5FD_MEM_EARRAY_HDR,                   
    H5AC__CLASS_NO_FLAGS_SET,              
    H5EA__cache_hdr_get_initial_load_size, 
    NULL,                                  
    H5EA__cache_hdr_verify_chksum,         
    H5EA__cache_hdr_deserialize,           
    H5EA__cache_hdr_image_len,             
    NULL,                                  
    H5EA__cache_hdr_serialize,             
    H5EA__cache_hdr_notify,                
    H5EA__cache_hdr_free_icr,              
    NULL,                                  
}};

const H5AC_class_t H5AC_EARRAY_IBLOCK[1] = {{
    H5AC_EARRAY_IBLOCK_ID,                    
    "Extensible Array Index Block",           
    H5FD_MEM_EARRAY_IBLOCK,                   
    H5AC__CLASS_NO_FLAGS_SET,                 
    H5EA__cache_iblock_get_initial_load_size, 
    NULL,                                     
    H5EA__cache_iblock_verify_chksum,         
    H5EA__cache_iblock_deserialize,           
    H5EA__cache_iblock_image_len,             
    NULL,                                     
    H5EA__cache_iblock_serialize,             
    H5EA__cache_iblock_notify,                
    H5EA__cache_iblock_free_icr,              
    NULL,                                     
}};

const H5AC_class_t H5AC_EARRAY_SBLOCK[1] = {{
    H5AC_EARRAY_SBLOCK_ID,                    
    "Extensible Array Super Block",           
    H5FD_MEM_EARRAY_SBLOCK,                   
    H5AC__CLASS_NO_FLAGS_SET,                 
    H5EA__cache_sblock_get_initial_load_size, 
    NULL,                                     
    H5EA__cache_sblock_verify_chksum,         
    H5EA__cache_sblock_deserialize,           
    H5EA__cache_sblock_image_len,             
    NULL,                                     
    H5EA__cache_sblock_serialize,             
    H5EA__cache_sblock_notify,                
    H5EA__cache_sblock_free_icr,              
    NULL,                                     
}};

const H5AC_class_t H5AC_EARRAY_DBLOCK[1] = {{
    H5AC_EARRAY_DBLOCK_ID,                    
    "Extensible Array Data Block",            
    H5FD_MEM_EARRAY_DBLOCK,                   
    H5AC__CLASS_NO_FLAGS_SET,                 
    H5EA__cache_dblock_get_initial_load_size, 
    NULL,                                     
    H5EA__cache_dblock_verify_chksum,         
    H5EA__cache_dblock_deserialize,           
    H5EA__cache_dblock_image_len,             
    NULL,                                     
    H5EA__cache_dblock_serialize,             
    H5EA__cache_dblock_notify,                
    H5EA__cache_dblock_free_icr,              
    H5EA__cache_dblock_fsf_size,              
}};

const H5AC_class_t H5AC_EARRAY_DBLK_PAGE[1] = {{
    H5AC_EARRAY_DBLK_PAGE_ID,                    
    "Extensible Array Data Block Page",          
    H5FD_MEM_EARRAY_DBLK_PAGE,                   
    H5AC__CLASS_NO_FLAGS_SET,                    
    H5EA__cache_dblk_page_get_initial_load_size, 
    NULL,                                        
    H5EA__cache_dblk_page_verify_chksum,         
    H5EA__cache_dblk_page_deserialize,           
    H5EA__cache_dblk_page_image_len,             
    NULL,                                        
    H5EA__cache_dblk_page_serialize,             
    H5EA__cache_dblk_page_notify,                
    H5EA__cache_dblk_page_free_icr,              
    NULL,                                        
}};

static herr_t
H5EA__cache_hdr_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5EA_hdr_cache_ud_t *udata = (H5EA_hdr_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->f);
    assert(image_len);

    
    *image_len = (size_t)H5EA_HEADER_SIZE_FILE(udata->f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5EA__cache_hdr_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5EA__cache_hdr_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5EA_cls_id_t        id;           
    H5EA_hdr_t          *hdr   = NULL; 
    H5EA_hdr_cache_ud_t *udata = (H5EA_hdr_cache_ud_t *)_udata;
    const uint8_t       *image = (const uint8_t *)_image; 
    uint32_t             stored_chksum;                   
    void                *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);
    assert(udata->f);
    assert(H5_addr_defined(udata->addr));

    
    if (NULL == (hdr = H5EA__hdr_alloc(udata->f)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array shared header");

    
    hdr->addr = udata->addr;

    
    if (memcmp(image, H5EA_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array header signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5EA_HDR_VERSION)
        HGOTO_ERROR(H5E_EARRAY, H5E_VERSION, NULL, "wrong extensible array header version");

    
    id = (H5EA_cls_id_t)*image++;
    if (id >= H5EA_NUM_CLS_ID)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADTYPE, NULL, "incorrect extensible array class");
    hdr->cparam.cls = H5EA_client_class_g[id];

    
    hdr->cparam.raw_elmt_size = *image++; 
    hdr->cparam.max_nelmts_bits =
        *image++; 
    hdr->cparam.idx_blk_elmts         = *image++; 
    hdr->cparam.data_blk_min_elmts    = *image++; 
    hdr->cparam.sup_blk_min_data_ptrs = *image++; 
    hdr->cparam.max_dblk_page_nelmts_bits =
        *image++; 

    
    hdr->stats.computed.hdr_size = len;                                   
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.nsuper_blks);    
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.super_blk_size); 
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.ndata_blks);     
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.data_blk_size);  
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.max_idx_set);    
    H5F_DECODE_LENGTH(udata->f, image, hdr->stats.stored.nelmts);         

    
    H5F_addr_decode(udata->f, &image, &hdr->idx_blk_addr); 

    
    if (H5_addr_defined(hdr->idx_blk_addr)) {
        H5EA_iblock_t iblock; 

        
        hdr->stats.computed.nindex_blks = 1;

        
        iblock.hdr         = hdr;
        iblock.nsblks      = H5EA_SBLK_FIRST_IDX(hdr->cparam.sup_blk_min_data_ptrs);
        iblock.ndblk_addrs = 2 * ((size_t)hdr->cparam.sup_blk_min_data_ptrs - 1);
        iblock.nsblk_addrs = hdr->nsblks - iblock.nsblks;

        
        hdr->stats.computed.index_blk_size = H5EA_IBLOCK_SIZE(&iblock);
    } 
    else {
        hdr->stats.computed.nindex_blks    = 0; 
        hdr->stats.computed.index_blk_size = 0; 
    }                                           

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5EA_SIZEOF_CHKSUM));

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == len);

    
    if (H5EA__hdr_init(hdr, udata->ctx_udata) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, NULL, "initialization failed for extensible array header");
    assert(hdr->size == len);

    
    ret_value = hdr;

done:
    
    if (!ret_value)
        if (hdr && H5EA__hdr_dest(hdr) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_hdr_image_len(const void *_thing, size_t *image_len)
{
    const H5EA_hdr_t *hdr = (const H5EA_hdr_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(image_len);

    
    *image_len = hdr->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_hdr_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5EA_hdr_t *hdr   = (H5EA_hdr_t *)_thing; 
    uint8_t    *image = (uint8_t *)_image;    
    uint32_t    metadata_chksum;              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(hdr);

    
    H5MM_memcpy(image, H5EA_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5EA_HDR_VERSION;

    
    assert(hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)hdr->cparam.cls->id;

    
    *image++ = hdr->cparam.raw_elmt_size;      
    *image++ = hdr->cparam.max_nelmts_bits;    
    *image++ = hdr->cparam.idx_blk_elmts;      
    *image++ = hdr->cparam.data_blk_min_elmts; 
    *image++ = hdr->cparam.sup_blk_min_data_ptrs; 
    *image++ =
        hdr->cparam.max_dblk_page_nelmts_bits; 

    
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.nsuper_blks);    
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.super_blk_size); 
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.ndata_blks);     
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.data_blk_size);  
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.max_idx_set);    
    H5F_ENCODE_LENGTH(f, image, hdr->stats.stored.nelmts);         

    
    H5F_addr_encode(f, &image, hdr->idx_blk_addr); 

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_hdr_notify(H5AC_notify_action_t action, void *_thing)
{
    H5EA_hdr_t *hdr       = (H5EA_hdr_t *)_thing; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->swmr_write) {
        
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (hdr->parent) {
                    
                    assert(hdr->top_proxy);

                    
                    if (H5AC_proxy_entry_remove_child((H5AC_proxy_entry_t *)hdr->parent,
                                                      (void *)hdr->top_proxy) < 0)
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between extensible array and proxy");
                    hdr->parent = NULL;
                } 

                
                if (hdr->top_proxy) {
                    if (H5AC_proxy_entry_remove_child(hdr->top_proxy, hdr) < 0)
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                    "unable to destroy flush dependency between header and "
                                    "extensible array 'top' proxy");
                    
                } 
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     
    else
        assert(NULL == hdr->parent);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_hdr_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5EA__hdr_dest((H5EA_hdr_t *)thing) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL, "can't free extensible array header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_iblock_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5EA_hdr_t   *hdr = (H5EA_hdr_t *)_udata; 
    H5EA_iblock_t iblock;                     

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(image_len);

    
    memset(&iblock, 0, sizeof(iblock));
    iblock.hdr         = (H5EA_hdr_t *)hdr;
    iblock.nsblks      = H5EA_SBLK_FIRST_IDX(hdr->cparam.sup_blk_min_data_ptrs);
    iblock.ndblk_addrs = 2 * ((size_t)hdr->cparam.sup_blk_min_data_ptrs - 1);
    iblock.nsblk_addrs = hdr->nsblks - iblock.nsblks;

    
    *image_len = (size_t)H5EA_IBLOCK_SIZE(&iblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5EA__cache_iblock_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5EA__cache_iblock_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5EA_iblock_t *iblock = NULL;                    
    H5EA_hdr_t    *hdr    = (H5EA_hdr_t *)_udata;    
    const uint8_t *image  = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                    
    haddr_t        arr_addr;                         
    size_t         u;                                
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(hdr);

    
    if (NULL == (iblock = H5EA__iblock_alloc(hdr)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array index block");

    
    iblock->addr = hdr->idx_blk_addr;

    
    if (memcmp(image, H5EA_IBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array index block signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5EA_IBLOCK_VERSION)
        HGOTO_ERROR(H5E_EARRAY, H5E_VERSION, NULL, "wrong extensible array index block version");

    
    if (*image++ != (uint8_t)hdr->cparam.cls->id)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADTYPE, NULL, "incorrect extensible array class");

    
    H5F_addr_decode(hdr->f, &image, &arr_addr);
    if (H5_addr_ne(arr_addr, hdr->addr))
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array header address");

    

    
    if (hdr->cparam.idx_blk_elmts > 0) {
        
        if ((hdr->cparam.cls->decode)(image, iblock->elmts, (size_t)hdr->cparam.idx_blk_elmts, hdr->cb_ctx) <
            0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, NULL, "can't decode extensible array index elements");
        image += (hdr->cparam.idx_blk_elmts * hdr->cparam.raw_elmt_size);
    } 

    
    if (iblock->ndblk_addrs > 0) {
        
        for (u = 0; u < iblock->ndblk_addrs; u++)
            H5F_addr_decode(hdr->f, &image, &iblock->dblk_addrs[u]);
    } 

    
    if (iblock->nsblk_addrs > 0) {
        
        for (u = 0; u < iblock->nsblk_addrs; u++)
            H5F_addr_decode(hdr->f, &image, &iblock->sblk_addrs[u]);
    } 

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5EA_SIZEOF_CHKSUM));

    
    iblock->size = len;

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == iblock->size);

    
    ret_value = iblock;

done:
    
    if (!ret_value)
        if (iblock && H5EA__iblock_dest(iblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array index block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_iblock_image_len(const void *_thing, size_t *image_len)
{
    const H5EA_iblock_t *iblock = (const H5EA_iblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iblock);
    assert(image_len);

    
    *image_len = iblock->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_iblock_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5EA_iblock_t *iblock = (H5EA_iblock_t *)_thing; 
    uint8_t       *image  = (uint8_t *)_image;       
    uint32_t       metadata_chksum;                  
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(iblock);
    assert(iblock->hdr);

    

    
    H5MM_memcpy(image, H5EA_IBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5EA_IBLOCK_VERSION;

    
    assert(iblock->hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)iblock->hdr->cparam.cls->id;

    
    H5F_addr_encode(f, &image, iblock->hdr->addr);

    

    
    if (iblock->hdr->cparam.idx_blk_elmts > 0) {
        
        if ((iblock->hdr->cparam.cls->encode)(image, iblock->elmts, (size_t)iblock->hdr->cparam.idx_blk_elmts,
                                              iblock->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTENCODE, FAIL, "can't encode extensible array index elements");
        image += (iblock->hdr->cparam.idx_blk_elmts * iblock->hdr->cparam.raw_elmt_size);
    } 

    
    if (iblock->ndblk_addrs > 0) {
        size_t u; 

        
        for (u = 0; u < iblock->ndblk_addrs; u++)
            H5F_addr_encode(f, &image, iblock->dblk_addrs[u]);
    } 

    
    if (iblock->nsblk_addrs > 0) {
        size_t u; 

        
        for (u = 0; u < iblock->nsblk_addrs; u++)
            H5F_addr_encode(f, &image, iblock->sblk_addrs[u]);
    } 

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_iblock_notify(H5AC_notify_action_t action, void *_thing)
{
    H5EA_iblock_t *iblock    = (H5EA_iblock_t *)_thing; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            
            if (H5EA__create_flush_depend((H5AC_info_t *)iblock->hdr, (H5AC_info_t *)iblock) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                    "unable to create flush dependency between index block and header, address = %llu",
                    (unsigned long long)iblock->addr);
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            if (H5EA__destroy_flush_depend((H5AC_info_t *)iblock->hdr, (H5AC_info_t *)iblock) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                    "unable to destroy flush dependency between index block and header, address = %llu",
                    (unsigned long long)iblock->addr);

            
            if (iblock->top_proxy) {
                if (H5AC_proxy_entry_remove_child(iblock->top_proxy, iblock) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between index block and "
                                "extensible array 'top' proxy");
                iblock->top_proxy = NULL;
            } 
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_iblock_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5EA__iblock_dest((H5EA_iblock_t *)thing) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL, "can't free extensible array index block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_sblock_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5EA_sblock_cache_ud_t *udata = (H5EA_sblock_cache_ud_t *)_udata; 
    H5EA_sblock_t           sblock; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->sblk_idx > 0);
    assert(H5_addr_defined(udata->sblk_addr));
    assert(image_len);

    
    
    memset(&sblock, 0, sizeof(sblock));
    sblock.hdr         = udata->hdr;
    sblock.ndblks      = udata->hdr->sblk_info[udata->sblk_idx].ndblks;
    sblock.dblk_nelmts = udata->hdr->sblk_info[udata->sblk_idx].dblk_nelmts;

    
    if (sblock.dblk_nelmts > udata->hdr->dblk_page_nelmts) {
        
        sblock.dblk_npages = sblock.dblk_nelmts / udata->hdr->dblk_page_nelmts;

        
        assert(sblock.dblk_npages > 1);

        
        assert((sblock.dblk_npages * udata->hdr->dblk_page_nelmts) == sblock.dblk_nelmts);

        
        sblock.dblk_page_init_size = ((sblock.dblk_npages) + 7) / 8;
        assert(sblock.dblk_page_init_size > 0);
    } 

    
    *image_len = (size_t)H5EA_SBLOCK_SIZE(&sblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5EA__cache_sblock_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5EA__cache_sblock_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5EA_sblock_t          *sblock = NULL;                             
    H5EA_sblock_cache_ud_t *udata  = (H5EA_sblock_cache_ud_t *)_udata; 
    const uint8_t          *image  = (const uint8_t *)_image;          
    uint32_t                stored_chksum;                             
    haddr_t                 arr_addr; 
    size_t                  u;        
    void                   *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->parent);
    assert(udata->sblk_idx > 0);
    assert(H5_addr_defined(udata->sblk_addr));

    
    if (NULL == (sblock = H5EA__sblock_alloc(udata->hdr, udata->parent, udata->sblk_idx)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array super block");

    
    sblock->addr = udata->sblk_addr;

    
    if (memcmp(image, H5EA_SBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array super block signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5EA_SBLOCK_VERSION)
        HGOTO_ERROR(H5E_EARRAY, H5E_VERSION, NULL, "wrong extensible array super block version");

    
    if (*image++ != (uint8_t)udata->hdr->cparam.cls->id)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADTYPE, NULL, "incorrect extensible array class");

    
    H5F_addr_decode(udata->hdr->f, &image, &arr_addr);
    if (H5_addr_ne(arr_addr, udata->hdr->addr))
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array header address");

    
    UINT64DECODE_VAR(image, sblock->block_off, udata->hdr->arr_off_size);

    

    
    if (sblock->dblk_npages > 0) {
        size_t tot_page_init_size =
            sblock->ndblks * sblock->dblk_page_init_size; 

        
        H5MM_memcpy(sblock->page_init, image, tot_page_init_size);
        image += tot_page_init_size;
    } 

    
    for (u = 0; u < sblock->ndblks; u++)
        H5F_addr_decode(udata->hdr->f, &image, &sblock->dblk_addrs[u]);

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5EA_SIZEOF_CHKSUM));

    
    sblock->size = len;

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == sblock->size);

    
    ret_value = sblock;

done:
    
    if (!ret_value)
        if (sblock && H5EA__sblock_dest(sblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array super block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_sblock_image_len(const void *_thing, size_t *image_len)
{
    const H5EA_sblock_t *sblock = (const H5EA_sblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sblock);
    assert(image_len);

    
    *image_len = sblock->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_sblock_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5EA_sblock_t *sblock = (H5EA_sblock_t *)_thing; 
    uint8_t       *image  = (uint8_t *)_image;       
    uint32_t       metadata_chksum;                  
    size_t         u;                                

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(sblock);
    assert(sblock->hdr);

    
    H5MM_memcpy(image, H5EA_SBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5EA_SBLOCK_VERSION;

    
    assert(sblock->hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)sblock->hdr->cparam.cls->id;

    
    H5F_addr_encode(f, &image, sblock->hdr->addr);

    
    UINT64ENCODE_VAR(image, sblock->block_off, sblock->hdr->arr_off_size);

    

    
    if (sblock->dblk_npages > 0) {
        size_t tot_page_init_size =
            sblock->ndblks * sblock->dblk_page_init_size; 

        
        H5MM_memcpy(image, sblock->page_init, tot_page_init_size);
        image += tot_page_init_size;
    } 

    
    for (u = 0; u < sblock->ndblks; u++)
        H5F_addr_encode(f, &image, sblock->dblk_addrs[u]);

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_sblock_notify(H5AC_notify_action_t action, void *_thing)
{
    H5EA_sblock_t *sblock    = (H5EA_sblock_t *)_thing; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(sblock);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            
            if (H5EA__create_flush_depend((H5AC_info_t *)sblock->parent, (H5AC_info_t *)sblock) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                    "unable to create flush dependency between super block and index block, address = %llu",
                    (unsigned long long)sblock->addr);
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            
            if (sblock->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)sblock->hdr, (H5AC_info_t *)sblock) < 0)
                    HGOTO_ERROR(
                        H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                        "unable to destroy flush dependency between super block and header, address = %llu",
                        (unsigned long long)sblock->addr);
                sblock->has_hdr_depend = false;
            } 
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            if (H5EA__destroy_flush_depend((H5AC_info_t *)sblock->parent, (H5AC_info_t *)sblock) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                    "unable to destroy flush dependency between super block and index block, address = %llu",
                    (unsigned long long)sblock->addr);

            
            if (sblock->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)sblock->hdr, (H5AC_info_t *)sblock) < 0)
                    HGOTO_ERROR(
                        H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                        "unable to destroy flush dependency between super block and header, address = %llu",
                        (unsigned long long)sblock->addr);
                sblock->has_hdr_depend = false;
            } 

            
            if (sblock->top_proxy) {
                if (H5AC_proxy_entry_remove_child(sblock->top_proxy, sblock) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between super block and "
                                "extensible array 'top' proxy");
                sblock->top_proxy = NULL;
            } 
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_sblock_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5EA__sblock_dest((H5EA_sblock_t *)thing) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL, "can't free extensible array super block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblock_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5EA_dblock_cache_ud_t *udata = (H5EA_dblock_cache_ud_t *)_udata; 
    H5EA_dblock_t           dblock;                                   

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->nelmts > 0);
    assert(image_len);

    
    
    memset(&dblock, 0, sizeof(dblock));

    
    dblock.hdr    = udata->hdr;
    dblock.nelmts = udata->nelmts;

    if (udata->nelmts > udata->hdr->dblk_page_nelmts) {
        
        dblock.npages = udata->nelmts / udata->hdr->dblk_page_nelmts;
        assert(udata->nelmts == (dblock.npages * udata->hdr->dblk_page_nelmts));
    } 

    
    if (!dblock.npages)
        *image_len = H5EA_DBLOCK_SIZE(&dblock);
    else
        *image_len = H5EA_DBLOCK_PREFIX_SIZE(&dblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5EA__cache_dblock_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5EA__cache_dblock_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                               bool H5_ATTR_UNUSED *dirty)
{
    H5EA_dblock_t          *dblock = NULL;                             
    H5EA_dblock_cache_ud_t *udata  = (H5EA_dblock_cache_ud_t *)_udata; 
    const uint8_t          *image  = (const uint8_t *)_image;          
    uint32_t                stored_chksum;                             
    haddr_t                 arr_addr; 
    void                   *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->parent);
    assert(udata->nelmts > 0);
    assert(H5_addr_defined(udata->dblk_addr));

    
    if (NULL == (dblock = H5EA__dblock_alloc(udata->hdr, udata->parent, udata->nelmts)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array data block");

    assert(((!dblock->npages) && (len == H5EA_DBLOCK_SIZE(dblock))) ||
           (len == H5EA_DBLOCK_PREFIX_SIZE(dblock)));

    
    dblock->addr = udata->dblk_addr;

    
    if (memcmp(image, H5EA_DBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array data block signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5EA_DBLOCK_VERSION)
        HGOTO_ERROR(H5E_EARRAY, H5E_VERSION, NULL, "wrong extensible array data block version");

    
    if (*image++ != (uint8_t)udata->hdr->cparam.cls->id)
        HGOTO_ERROR(H5E_EARRAY, H5E_BADTYPE, NULL, "incorrect extensible array class");

    
    H5F_addr_decode(udata->hdr->f, &image, &arr_addr);
    if (H5_addr_ne(arr_addr, udata->hdr->addr))
        HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, NULL, "wrong extensible array header address");

    
    UINT64DECODE_VAR(image, dblock->block_off, udata->hdr->arr_off_size);

    

    
    if (!dblock->npages) {
        
        
        if ((udata->hdr->cparam.cls->decode)(image, dblock->elmts, udata->nelmts, udata->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, NULL, "can't decode extensible array data elements");
        image += (udata->nelmts * udata->hdr->cparam.raw_elmt_size);
    } 

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5EA_SIZEOF_CHKSUM));

    
    
    dblock->size = H5EA_DBLOCK_SIZE(dblock);

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == len);

    
    ret_value = dblock;

done:

    
    if (!ret_value)
        if (dblock && H5EA__dblock_dest(dblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblock_image_len(const void *_thing, size_t *image_len)
{
    const H5EA_dblock_t *dblock = (const H5EA_dblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblock);
    assert(image_len);

    
    if (!dblock->npages)
        *image_len = dblock->size;
    else
        *image_len = (size_t)H5EA_DBLOCK_PREFIX_SIZE(dblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_dblock_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5EA_dblock_t *dblock = (H5EA_dblock_t *)_thing; 
    uint8_t       *image  = (uint8_t *)_image;       
    uint32_t       metadata_chksum;                  
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(dblock);
    assert(dblock->hdr);

    
    H5MM_memcpy(image, H5EA_DBLOCK_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5EA_DBLOCK_VERSION;

    
    assert(dblock->hdr->cparam.cls->id <= 255);
    *image++ = (uint8_t)dblock->hdr->cparam.cls->id;

    
    H5F_addr_encode(f, &image, dblock->hdr->addr);

    
    UINT64ENCODE_VAR(image, dblock->block_off, dblock->hdr->arr_off_size);

    

    
    if (!dblock->npages) {
        

        
        if ((dblock->hdr->cparam.cls->encode)(image, dblock->elmts, dblock->nelmts, dblock->hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTENCODE, FAIL, "can't encode extensible array data elements");
        image += (dblock->nelmts * dblock->hdr->cparam.raw_elmt_size);
    } 

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblock_notify(H5AC_notify_action_t action, void *_thing)
{
    H5EA_dblock_t *dblock    = (H5EA_dblock_t *)_thing; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            
            if (H5EA__create_flush_depend((H5AC_info_t *)dblock->parent, (H5AC_info_t *)dblock) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                            "unable to create flush dependency between data block and parent, address = %llu",
                            (unsigned long long)dblock->addr);
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            
            if (dblock->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)dblock->hdr, (H5AC_info_t *)dblock) < 0)
                    HGOTO_ERROR(
                        H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                        "unable to destroy flush dependency between direct block and header, address = %llu",
                        (unsigned long long)dblock->addr);
                dblock->has_hdr_depend = false;
            } 
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            if (H5EA__destroy_flush_depend((H5AC_info_t *)dblock->parent, (H5AC_info_t *)dblock) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                    "unable to destroy flush dependency between data block and parent, address = %llu",
                    (unsigned long long)dblock->addr);

            
            if (dblock->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)dblock->hdr, (H5AC_info_t *)dblock) < 0)
                    HGOTO_ERROR(
                        H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                        "unable to destroy flush dependency between data block and header, address = %llu",
                        (unsigned long long)dblock->addr);
                dblock->has_hdr_depend = false;
            } 

            
            if (dblock->top_proxy) {
                if (H5AC_proxy_entry_remove_child(dblock->top_proxy, dblock) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between data block and "
                                "extensible array 'top' proxy");
                dblock->top_proxy = NULL;
            } 
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblock_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5EA__dblock_dest((H5EA_dblock_t *)thing) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL, "can't free extensible array data block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblock_fsf_size(const void *_thing, hsize_t *fsf_size)
{
    const H5EA_dblock_t *dblock = (const H5EA_dblock_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblock);
    assert(dblock->cache_info.type == H5AC_EARRAY_DBLOCK);
    assert(fsf_size);

    *fsf_size = dblock->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_dblk_page_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5EA_dblk_page_cache_ud_t *udata = (H5EA_dblk_page_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->hdr);
    assert(image_len);

    
    *image_len = (size_t)H5EA_DBLK_PAGE_SIZE(udata->hdr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5EA__cache_dblk_page_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5EA__cache_dblk_page_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5EA_dblk_page_t          *dblk_page = NULL; 
    H5EA_dblk_page_cache_ud_t *udata =
        (H5EA_dblk_page_cache_ud_t *)_udata;        
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(udata);
    assert(udata->hdr);
    assert(udata->parent);
    assert(H5_addr_defined(udata->dblk_page_addr));

    
    if (NULL == (dblk_page = H5EA__dblk_page_alloc(udata->hdr, udata->parent)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array data block page");

    
    dblk_page->addr = udata->dblk_page_addr;

    

    
    
    if ((udata->hdr->cparam.cls->decode)(image, dblk_page->elmts, udata->hdr->dblk_page_nelmts,
                                         udata->hdr->cb_ctx) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDECODE, NULL, "can't decode extensible array data elements");
    image += (udata->hdr->dblk_page_nelmts * udata->hdr->cparam.raw_elmt_size);

    
    
    assert((size_t)(image - (const uint8_t *)_image) == (len - H5EA_SIZEOF_CHKSUM));

    
    dblk_page->size = len;

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) == dblk_page->size);

    
    ret_value = dblk_page;

done:
    
    if (!ret_value)
        if (dblk_page && H5EA__dblk_page_dest(dblk_page) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array data block page");
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblk_page_image_len(const void *_thing, size_t *image_len)
{
    const H5EA_dblk_page_t *dblk_page = (const H5EA_dblk_page_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dblk_page);
    assert(image_len);

    
    *image_len = dblk_page->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__cache_dblk_page_serialize(const H5F_t H5_ATTR_NDEBUG_UNUSED *f, void *_image, size_t H5_ATTR_UNUSED len,
                                void *_thing)
{
    H5EA_dblk_page_t *dblk_page = (H5EA_dblk_page_t *)_thing; 
    uint8_t          *image     = (uint8_t *)_image;          
    uint32_t          metadata_chksum;                        
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(dblk_page);
    assert(dblk_page->hdr);

    

    

    
    if ((dblk_page->hdr->cparam.cls->encode)(image, dblk_page->elmts, dblk_page->hdr->dblk_page_nelmts,
                                             dblk_page->hdr->cb_ctx) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTENCODE, FAIL, "can't encode extensible array data elements");
    image += (dblk_page->hdr->dblk_page_nelmts * dblk_page->hdr->cparam.raw_elmt_size);

    
    metadata_chksum = H5_checksum_metadata(_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblk_page_notify(H5AC_notify_action_t action, void *_thing)
{
    H5EA_dblk_page_t *dblk_page = (H5EA_dblk_page_t *)_thing; 
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk_page);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            
            if (H5EA__create_flush_depend((H5AC_info_t *)dblk_page->parent, (H5AC_info_t *)dblk_page) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                    "unable to create flush dependency between data block page and parent, address = %llu",
                    (unsigned long long)dblk_page->addr);
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            
            if (dblk_page->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)dblk_page->hdr, (H5AC_info_t *)dblk_page) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between data block page and header, "
                                "address = %llu",
                                (unsigned long long)dblk_page->addr);
                dblk_page->has_hdr_depend = false;
            } 
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            if (H5EA__destroy_flush_depend((H5AC_info_t *)dblk_page->parent, (H5AC_info_t *)dblk_page) < 0)
                HGOTO_ERROR(
                    H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                    "unable to destroy flush dependency between data block page and parent, address = %llu",
                    (unsigned long long)dblk_page->addr);

            
            if (dblk_page->has_hdr_depend) {
                if (H5EA__destroy_flush_depend((H5AC_info_t *)dblk_page->hdr, (H5AC_info_t *)dblk_page) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between data block page and header, "
                                "address = %llu",
                                (unsigned long long)dblk_page->addr);
                dblk_page->has_hdr_depend = false;
            } 

            
            if (dblk_page->top_proxy) {
                if (H5AC_proxy_entry_remove_child(dblk_page->top_proxy, dblk_page) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy flush dependency between data block page and "
                                "extensible array 'top' proxy");
                dblk_page->top_proxy = NULL;
            } 
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__cache_dblk_page_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5EA__dblk_page_dest((H5EA_dblk_page_t *)thing) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL, "can't free extensible array data block page");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
