/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

#ifdef H5O_DEBUG

herr_t
H5O__assert(const H5O_t *oh)
{
    H5O_mesg_t *curr_msg;            
    H5O_mesg_t *tmp_msg;             
    unsigned    cont_msgs_found = 0; 
    size_t      meta_space;          
    size_t      mesg_space;          
    size_t      free_space;          
    size_t      hdr_size;            
    unsigned    u, v;                

    FUNC_ENTER_PACKAGE_NOERR

    assert(oh);
    assert(oh->chunk || oh->nchunks == 0);
    assert(oh->mesg || oh->nmesgs == 0);

    
    hdr_size   = 0;
    meta_space = (size_t)H5O_SIZEOF_HDR(oh) + (size_t)(H5O_SIZEOF_CHKHDR_OH(oh) * (oh->nchunks - 1));
    mesg_space = 0;
    free_space = 0;

    
    for (u = 0; u < oh->nchunks; u++) {
        
        hdr_size += oh->chunk[u].size;

        
        free_space += oh->chunk[u].gap;

        
        assert(oh->chunk[u].image);
        assert(oh->chunk[u].size > (size_t)H5O_SIZEOF_CHKHDR_OH(oh));

        
        assert(H5_addr_defined(oh->chunk[u].addr));

        
        if (oh->version > H5O_VERSION_1) {
            
            assert(!memcmp(oh->chunk[u].image, (u == 0 ? H5O_HDR_MAGIC : H5O_CHK_MAGIC), H5_SIZEOF_MAGIC));

            
            assert(oh->chunk[u].gap < (size_t)H5O_SIZEOF_MSGHDR_OH(oh));
        } 
        else
            
            assert(oh->chunk[u].gap == 0);
    } 

    
    if (oh->version > H5O_VERSION_1) {
        uint64_t chunk0_size = oh->chunk[0].size - (size_t)H5O_SIZEOF_HDR(oh);

        if (chunk0_size <= 255)
            assert((oh->flags & H5O_HDR_CHUNK0_SIZE) == H5O_HDR_CHUNK0_1);
        else if (chunk0_size <= 65535)
            assert((oh->flags & H5O_HDR_CHUNK0_SIZE) == H5O_HDR_CHUNK0_2);
        else if (chunk0_size <= 4294967295)
            assert((oh->flags & H5O_HDR_CHUNK0_SIZE) == H5O_HDR_CHUNK0_4);
        else
            assert((oh->flags & H5O_HDR_CHUNK0_SIZE) == H5O_HDR_CHUNK0_8);
    } 

    
    for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
        uint8_t H5_ATTR_NDEBUG_UNUSED *curr_hdr;      
        size_t                         curr_tot_size; 

        assert(curr_msg->type);

        curr_hdr      = curr_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh);
        curr_tot_size = curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);

        
        if (H5O_NULL_ID == curr_msg->type->id)
            free_space += curr_tot_size;
        else if (H5O_CONT_ID == curr_msg->type->id) {
            H5O_cont_t                *cont        = (H5O_cont_t *)curr_msg->native;
            bool H5_ATTR_NDEBUG_UNUSED found_chunk = false; 

            assert(cont);

            
            cont_msgs_found++;

            
            
            for (v = 0; v < oh->nchunks; v++) {
                if (H5_addr_eq(cont->addr, oh->chunk[v].addr) && cont->size == oh->chunk[v].size) {
                    assert(cont->chunkno == v);
                    assert(!found_chunk);
                    found_chunk = true;
                } 
            }     
            assert(found_chunk);

            meta_space += curr_tot_size;
        } 
        else {
            meta_space += (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
            mesg_space += curr_msg->raw_size;

            
            assert(curr_msg->native || !curr_msg->dirty);
        } 

        
        assert(curr_msg->chunkno < oh->nchunks);

        
        if (H5O_NULL_ID == curr_msg->type->id)
            assert(oh->chunk[curr_msg->chunkno].gap == 0);

        
        assert(curr_tot_size <= (oh->chunk[curr_msg->chunkno].size) -
                                    (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[curr_msg->chunkno].gap));
        if (curr_msg->chunkno == 0)
            assert(curr_hdr >=
                   oh->chunk[curr_msg->chunkno].image + (H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh)));
        else
            assert(curr_hdr >= oh->chunk[curr_msg->chunkno].image +
                                   (H5O_SIZEOF_CHKHDR_OH(oh) - H5O_SIZEOF_CHKSUM_OH(oh)));
        assert(curr_msg->raw + curr_msg->raw_size <=
               (oh->chunk[curr_msg->chunkno].image + oh->chunk[curr_msg->chunkno].size) -
                   (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[curr_msg->chunkno].gap));

        
        for (v = 0, tmp_msg = &oh->mesg[0]; v < oh->nmesgs; v++, tmp_msg++) {
            if (u != v)
                assert(!((tmp_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)) >= curr_hdr &&
                         (tmp_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)) < (curr_hdr + curr_tot_size)));
        } 
    }     

    
    assert(oh->nchunks == (cont_msgs_found + 1));

    
    assert(hdr_size == (free_space + meta_space + mesg_space));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
#endif 

herr_t
H5O_debug_id(unsigned type_id, H5F_t *f, const void *mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_msg_class_t *type;             
    herr_t                 ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(type_id < NELMTS(H5O_msg_class_g));
    type = H5O_msg_class_g[type_id]; 
    assert(type);
    assert(type->debug);
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if ((ret_value = (type->debug)(f, mesg, stream, indent, fwidth)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "unable to debug message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__debug_real(H5F_t *f, H5O_t *oh, haddr_t addr, FILE *stream, int indent, int fwidth)
{
    size_t    mesg_total = 0, chunk_total = 0, gap_total = 0;
    unsigned *sequence = NULL;
    unsigned  i; 
    herr_t    ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    Rfprintf(stream, "%*sObject Header...\n", indent, "");

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Dirty:", oh->cache_info.is_dirty ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Version:", oh->version);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Header size (in bytes):", (unsigned)H5O_SIZEOF_HDR(oh));
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Number of links:", oh->nlink);

    
    if (oh->version > H5O_VERSION_1) {
        
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Attribute creation order tracked:",
                (oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED) ? "Yes" : "No");
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Attribute creation order indexed:",
                (oh->flags & H5O_HDR_ATTR_CRT_ORDER_INDEXED) ? "Yes" : "No");
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Attribute storage phase change values:",
                (oh->flags & H5O_HDR_ATTR_STORE_PHASE_CHANGE) ? "Non-default" : "Default");
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
                "Timestamps:", (oh->flags & H5O_HDR_STORE_TIMES) ? "Enabled" : "Disabled");
        if (oh->flags & ~H5O_HDR_ALL_FLAGS)
            Rfprintf(stream, "*** UNKNOWN OBJECT HEADER STATUS FLAG: %02x!\n", (unsigned)oh->flags);

        
        if (oh->flags & H5O_HDR_STORE_TIMES) {
            struct tm *tm;       
            char       buf[128]; 

            
            tm = localtime(&oh->atime);
            strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", tm);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Access Time:", buf);
            tm = localtime(&oh->mtime);
            strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", tm);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Modification Time:", buf);
            tm = localtime(&oh->ctime);
            strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", tm);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Change Time:", buf);
            tm = localtime(&oh->btime);
            strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", tm);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Birth Time:", buf);
        } 

        
        if (oh->flags & H5O_HDR_ATTR_STORE_PHASE_CHANGE) {
            Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                    "Max. compact attributes:", (unsigned)oh->max_compact);
            Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                    "Min. dense attributes:", (unsigned)oh->min_dense);
        } 
    }     

    Rfprintf(stream, "%*s%-*s %llu (%llu)\n", indent, "", fwidth, "Number of messages (allocated):",
            (unsigned long long)oh->nmesgs, (unsigned long long)oh->alloc_nmesgs);
    Rfprintf(stream, "%*s%-*s %llu (%llu)\n", indent, "", fwidth, "Number of chunks (allocated):",
            (unsigned long long)oh->nchunks, (unsigned long long)oh->alloc_nchunks);

    
    for (i = 0, chunk_total = 0; i < oh->nchunks; i++) {
        size_t chunk_size;

        Rfprintf(stream, "%*sChunk %d...\n", indent, "", i);

        Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent + 3, "", MAX(0, fwidth - 3),
                "Address:", oh->chunk[i].addr);

        
        if (0 == i) {
            if (H5_addr_ne(oh->chunk[i].addr, addr))
                Rfprintf(stream, "*** WRONG ADDRESS FOR CHUNK #0!\n");
            chunk_size = oh->chunk[i].size - (size_t)H5O_SIZEOF_HDR(oh);
        } 
        else
            chunk_size = oh->chunk[i].size;

        
        chunk_total += chunk_size;
        gap_total += oh->chunk[i].gap;

        Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(0, fwidth - 3), "Size in bytes:", (unsigned long long)chunk_size);

        Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(0, fwidth - 3), "Gap:", (unsigned long long)oh->chunk[i].gap);
    } 

    
    if (NULL == (sequence = (unsigned *)H5MM_calloc(NELMTS(H5O_msg_class_g) * sizeof(unsigned))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    for (i = 0, mesg_total = 0; i < oh->nmesgs; i++) {
        const H5O_msg_class_t *debug_type; 
        unsigned               chunkno;    

        
        mesg_total += (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + oh->mesg[i].raw_size;

        
        if (oh->mesg[i].type->id == H5O_CONT_ID)
            mesg_total += H5O_SIZEOF_CHKHDR_OH(oh);

        Rfprintf(stream, "%*sMessage %d...\n", indent, "", i);

        
        if (oh->mesg[i].type->id >= (int)NELMTS(H5O_msg_class_g)) {
            Rfprintf(stream, "*** BAD MESSAGE ID 0x%04x\n", oh->mesg[i].type->id);
            continue;
        } 

        
        Rfprintf(stream, "%*s%-*s 0x%04x `%s' (%d)\n", indent + 3, "", MAX(0, fwidth - 3),
                "Message ID (sequence number):", (unsigned)(oh->mesg[i].type->id), oh->mesg[i].type->name,
                sequence[oh->mesg[i].type->id]++);
        Rfprintf(stream, "%*s%-*s %s\n", indent + 3, "", MAX(0, fwidth - 3),
                "Dirty:", oh->mesg[i].dirty ? "TRUE" : "FALSE");
        Rfprintf(stream, "%*s%-*s ", indent + 3, "", MAX(0, fwidth - 3), "Message flags:");
        if (oh->mesg[i].flags) {
            bool flag_printed = false;

            
            HDcompile_assert(H5O_MSG_FLAG_BITS ==
                             (H5O_MSG_FLAG_CONSTANT | H5O_MSG_FLAG_SHARED | H5O_MSG_FLAG_DONTSHARE |
                              H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE | H5O_MSG_FLAG_MARK_IF_UNKNOWN |
                              H5O_MSG_FLAG_WAS_UNKNOWN | H5O_MSG_FLAG_SHAREABLE |
                              H5O_MSG_FLAG_FAIL_IF_UNKNOWN_ALWAYS));

            if (oh->mesg[i].flags & H5O_MSG_FLAG_CONSTANT) {
                Rfprintf(stream, "%sC", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_SHARED) {
                Rfprintf(stream, "%sS", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_DONTSHARE) {
                Rfprintf(stream, "%sDS", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE) {
                Rfprintf(stream, "%sFIUW", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_MARK_IF_UNKNOWN) {
                Rfprintf(stream, "%sMIU", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_WAS_UNKNOWN) {
                assert(oh->mesg[i].flags & H5O_MSG_FLAG_MARK_IF_UNKNOWN);
                Rfprintf(stream, "%sWU", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_SHAREABLE) {
                Rfprintf(stream, "%sSA", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (oh->mesg[i].flags & H5O_MSG_FLAG_FAIL_IF_UNKNOWN_ALWAYS) {
                Rfprintf(stream, "%sFIUA", (flag_printed ? ", " : "<"));
                flag_printed = true;
            } 
            if (!flag_printed)
                Rfprintf(stream, "-");
            Rfprintf(stream, ">\n");
            if (oh->mesg[i].flags & ~H5O_MSG_FLAG_BITS)
                Rfprintf(stream, "%*s%-*s 0x%02x\n", indent + 3, "", MAX(0, fwidth - 3),
                        "*** ADDITIONAL UNKNOWN FLAGS --->", oh->mesg[i].flags & ~H5O_MSG_FLAG_BITS);
        } 
        else
            Rfprintf(stream, "<none>\n");

        Rfprintf(stream, "%*s%-*s %u\n", indent + 3, "", MAX(0, fwidth - 3),
                "Chunk number:", oh->mesg[i].chunkno);
        chunkno = oh->mesg[i].chunkno;
        if (chunkno >= oh->nchunks)
            Rfprintf(stream, "*** BAD CHUNK NUMBER\n");
        Rfprintf(stream, "%*s%-*s (%llu, %llu) bytes\n", indent + 3, "", MAX(0, fwidth - 3),
                "Raw message data (offset, size) in chunk:",
                (unsigned long long)(oh->mesg[i].raw - oh->chunk[chunkno].image), 
                (unsigned long long)oh->mesg[i].raw_size);

        
        if ((oh->mesg[i].raw + oh->mesg[i].raw_size > oh->chunk[chunkno].image + oh->chunk[chunkno].size) ||
            (oh->mesg[i].raw < oh->chunk[chunkno].image))
            Rfprintf(stream, "*** BAD MESSAGE RAW ADDRESS\n");

        
        debug_type = oh->mesg[i].type;
        if (NULL == oh->mesg[i].native && debug_type->decode)
            H5O_LOAD_NATIVE(f, H5O_DECODEIO_NOCHANGE, oh, &oh->mesg[i], FAIL)

        
        Rfprintf(stream, "%*s%-*s\n", indent + 3, "", MAX(0, fwidth - 3), "Message Information:");
        if (debug_type->debug && oh->mesg[i].native != NULL)
            (debug_type->debug)(f, oh->mesg[i].native, stream, indent + 6, MAX(0, fwidth - 6));
        else
            Rfprintf(stream, "%*s<No info for this message>\n", indent + 6, "");
    } 

    if ((mesg_total + gap_total) != chunk_total)
        Rfprintf(stream, "*** TOTAL SIZE DOES NOT MATCH ALLOCATED SIZE!\n");

done:
    
    if (sequence)
        sequence = (unsigned *)H5MM_xfree(sequence);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth)
{
    H5O_t    *oh = NULL;           
    H5O_loc_t loc;                 
    herr_t    ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    loc.file         = f;
    loc.addr         = addr;
    loc.holding_file = false;

    if (NULL == (oh = H5O_protect(&loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    if (H5O__debug_real(f, oh, addr, stream, indent, fwidth) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_SYSTEM, FAIL, "debug dump call failed");

done:
    if (oh && H5O_unprotect(&loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
