/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5VL_FRIEND 

#include "H5private.h"  
#include "H5Aprivate.h" 
#include "H5Dprivate.h" 
#include "H5Eprivate.h" 
#include "H5Fprivate.h" 
#include "H5Gprivate.h" 
#include "H5Iprivate.h" 
#include "H5Oprivate.h" 
#include "H5Pprivate.h" 
#include "H5Tprivate.h" 
#include "H5VLpkg.h"    

#include "H5VLnative_private.h" 

hid_t             H5VL_NATIVE_g      = H5I_INVALID_HID;
H5VL_connector_t *H5VL_NATIVE_conn_g = NULL;

#define H5VL_NATIVE_CAP_FLAGS                                                                                \
    (H5VL_CAP_FLAG_NATIVE_FILES | H5VL_CAP_FLAG_ATTR_BASIC | H5VL_CAP_FLAG_ATTR_MORE |                       \
     H5VL_CAP_FLAG_DATASET_BASIC | H5VL_CAP_FLAG_DATASET_MORE | H5VL_CAP_FLAG_FILE_BASIC |                   \
     H5VL_CAP_FLAG_FILE_MORE | H5VL_CAP_FLAG_GROUP_BASIC | H5VL_CAP_FLAG_GROUP_MORE |                        \
     H5VL_CAP_FLAG_LINK_BASIC | H5VL_CAP_FLAG_LINK_MORE | H5VL_CAP_FLAG_OBJECT_BASIC |                       \
     H5VL_CAP_FLAG_OBJECT_MORE | H5VL_CAP_FLAG_REF_BASIC | H5VL_CAP_FLAG_REF_MORE | H5VL_CAP_FLAG_OBJ_REF |  \
     H5VL_CAP_FLAG_REG_REF | H5VL_CAP_FLAG_ATTR_REF | H5VL_CAP_FLAG_STORED_DATATYPES |                       \
     H5VL_CAP_FLAG_CREATION_ORDER | H5VL_CAP_FLAG_ITERATE | H5VL_CAP_FLAG_STORAGE_SIZE |                     \
     H5VL_CAP_FLAG_BY_IDX | H5VL_CAP_FLAG_GET_PLIST | H5VL_CAP_FLAG_FLUSH_REFRESH |                          \
     H5VL_CAP_FLAG_EXTERNAL_LINKS | H5VL_CAP_FLAG_HARD_LINKS | H5VL_CAP_FLAG_SOFT_LINKS |                    \
     H5VL_CAP_FLAG_UD_LINKS | H5VL_CAP_FLAG_TRACK_TIMES | H5VL_CAP_FLAG_MOUNT | H5VL_CAP_FLAG_FILTERS |      \
     H5VL_CAP_FLAG_FILL_VALUES)

static const H5VL_class_t H5VL_native_cls_g = {
    H5VL_VERSION,          
    H5VL_NATIVE_VALUE,     
    H5VL_NATIVE_NAME,      
    H5VL_NATIVE_VERSION,   
    H5VL_NATIVE_CAP_FLAGS, 
    NULL,                  
    NULL,                  
    {
        
        (size_t)0, 
        NULL,      
        NULL,      
        NULL,      
        NULL,      
        NULL       
    },
    {
        
        NULL, 
        NULL, 
        NULL, 
        NULL, 
        NULL  
    },
    {
        
        H5VL__native_attr_create,   
        H5VL__native_attr_open,     
        H5VL__native_attr_read,     
        H5VL__native_attr_write,    
        H5VL__native_attr_get,      
        H5VL__native_attr_specific, 
        H5VL__native_attr_optional, 
        H5VL__native_attr_close     
    },
    {
        
        H5VL__native_dataset_create,   
        H5VL__native_dataset_open,     
        H5VL__native_dataset_read,     
        H5VL__native_dataset_write,    
        H5VL__native_dataset_get,      
        H5VL__native_dataset_specific, 
        H5VL__native_dataset_optional, 
        H5VL__native_dataset_close     
    },
    {
        
        H5VL__native_datatype_commit,   
        H5VL__native_datatype_open,     
        H5VL__native_datatype_get,      
        H5VL__native_datatype_specific, 
        NULL,                           
        H5VL__native_datatype_close     
    },
    {
        
        H5VL__native_file_create,   
        H5VL__native_file_open,     
        H5VL__native_file_get,      
        H5VL__native_file_specific, 
        H5VL__native_file_optional, 
        H5VL__native_file_close     
    },
    {
        
        H5VL__native_group_create,   
        H5VL__native_group_open,     
        H5VL__native_group_get,      
        H5VL__native_group_specific, 
        H5VL__native_group_optional, 
        H5VL__native_group_close     
    },
    {
        
        H5VL__native_link_create,   
        H5VL__native_link_copy,     
        H5VL__native_link_move,     
        H5VL__native_link_get,      
        H5VL__native_link_specific, 
        NULL                        
    },
    {
        
        H5VL__native_object_open,     
        H5VL__native_object_copy,     
        H5VL__native_object_get,      
        H5VL__native_object_specific, 
        H5VL__native_object_optional  
    },
    {
        
        H5VL__native_introspect_get_conn_cls,  
        H5VL__native_introspect_get_cap_flags, 
        H5VL__native_introspect_opt_query,     
    },
    {
        
        NULL, 
        NULL, 
        NULL, 
        NULL, 
        NULL, 
        NULL  
    },
    {
        
        H5VL__native_blob_put,      
        H5VL__native_blob_get,      
        H5VL__native_blob_specific, 
        NULL                        
    },
    {
        
        H5VL__native_token_cmp,    
        H5VL__native_token_to_str, 
        H5VL__native_str_to_token  
    },
    NULL 
};

herr_t
H5VL__native_register(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == H5VL_NATIVE_conn_g)
        if (NULL ==
            (H5VL_NATIVE_conn_g = H5VL__register_connector(&H5VL_native_cls_g, H5P_VOL_INITIALIZE_DEFAULT)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, FAIL, "can't register native VOL connector");

    
    if (H5I_VOL != H5I_get_type(H5VL_NATIVE_g)) {
        if ((H5VL_NATIVE_g = H5I_register(H5I_VOL, H5VL_NATIVE_conn_g, false)) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, FAIL, "can't create ID for native VOL connector");

        
        H5VL_conn_inc_rc(H5VL_NATIVE_conn_g);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_unregister(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5VL_NATIVE_g      = H5I_INVALID_HID;
    H5VL_NATIVE_conn_g = NULL;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VL__native_introspect_get_conn_cls(void H5_ATTR_UNUSED *obj, H5VL_get_conn_lvl_t H5_ATTR_UNUSED lvl,
                                     const H5VL_class_t **conn_cls)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(conn_cls);

    
    *conn_cls = &H5VL_native_cls_g;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VL__native_introspect_get_cap_flags(const void H5_ATTR_UNUSED *info, uint64_t *cap_flags)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cap_flags);

    
    *cap_flags = H5VL_native_cls_g.cap_flags;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VL_native_get_file_addr_len(hid_t loc_id, size_t *addr_len)
{
    H5I_type_t vol_obj_type = H5I_BADID; 
    void      *vol_obj      = NULL;      
    herr_t     ret_value    = SUCCEED;   

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(addr_len);

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (NULL == (vol_obj = H5VL_object(loc_id)))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5VL__native_get_file_addr_len(vol_obj, vol_obj_type, addr_len) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get file address length");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_get_file_addr_len(void *obj, H5I_type_t obj_type, size_t *addr_len)
{
    H5F_t *file      = NULL; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(obj);
    assert(addr_len);

    
    if (H5VL_native_get_file_struct(obj, obj_type, &file) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "couldn't get file from VOL object");

    
    *addr_len = H5F_SIZEOF_ADDR(file);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VLnative_addr_to_token(hid_t loc_id, haddr_t addr, H5O_token_t *token)
{
    H5I_type_t vol_obj_type = H5I_BADID; 
    void      *vol_obj      = NULL;      
    herr_t     ret_value    = SUCCEED;   

    FUNC_ENTER_API(FAIL)

    
    if (NULL == token)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "token pointer can't be NULL");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (NULL == (vol_obj = H5VL_object(loc_id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get underlying VOL object");

#ifndef NDEBUG
    {
        H5VL_object_t *vol_obj_container;
        bool           is_native_vol_obj;

        
        if (NULL == (vol_obj_container = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        if (H5VL_object_is_native(vol_obj_container, &is_native_vol_obj) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL,
                        "can't determine if VOL object is native connector object");

        assert(is_native_vol_obj && "not a native VOL connector object");
    }
#endif

    
    if (H5VL_native_addr_to_token(vol_obj, vol_obj_type, addr, token) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSERIALIZE, FAIL, "couldn't serialize haddr_t into object token");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VL_native_addr_to_token(void *obj, H5I_type_t obj_type, haddr_t addr, H5O_token_t *token)
{
    uint8_t *p;
    size_t   addr_len  = 0; 
    herr_t   ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(obj);
    assert(token);

    
    if (H5VL__native_get_file_addr_len(obj, obj_type, &addr_len) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "couldn't get length of haddr_t from VOL object");

    
    memset(token, 0, sizeof(H5O_token_t));

    
    p = (uint8_t *)token;
    H5F_addr_encode_len(addr_len, &p, addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VLnative_token_to_addr(hid_t loc_id, H5O_token_t token, haddr_t *addr)
{
    H5I_type_t vol_obj_type = H5I_BADID; 
    void      *vol_obj      = NULL;      
    herr_t     ret_value    = SUCCEED;   

    FUNC_ENTER_API(FAIL)

    
    if (NULL == addr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addr pointer can't be NULL");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (NULL == (vol_obj = H5VL_object(loc_id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get underlying VOL object");

#ifndef NDEBUG
    {
        H5VL_object_t *vol_obj_container;
        bool           is_native_vol_obj;

        
        if (NULL == (vol_obj_container = H5VL_vol_object(loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

        
        if (H5VL_object_is_native(vol_obj_container, &is_native_vol_obj) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL,
                        "can't determine if VOL object is native connector object");

        assert(is_native_vol_obj && "not a native VOL connector object");
    }
#endif

    
    if (H5VL_native_token_to_addr(vol_obj, vol_obj_type, token, addr) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTUNSERIALIZE, FAIL, "couldn't deserialize object token into haddr_t");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VL_native_token_to_addr(void *obj, H5I_type_t obj_type, H5O_token_t token, haddr_t *addr)
{
    const uint8_t *p;
    size_t         addr_len  = 0; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(obj);
    assert(addr);

    
    if (H5VL__native_get_file_addr_len(obj, obj_type, &addr_len) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "couldn't get length of haddr_t from VOL object");

    
    p = (const uint8_t *)&token;
    H5F_addr_decode_len(addr_len, &p, addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_native_get_file_struct(void *obj, H5I_type_t type, H5F_t **file)
{
    H5O_loc_t *oloc      = NULL;    
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    *file = NULL;

    switch (type) {
        case H5I_FILE:
            *file = (H5F_t *)obj;
            break;

        case H5I_GROUP:
            oloc = H5G_oloc((H5G_t *)obj);
            break;

        case H5I_DATATYPE:
            oloc = H5T_oloc((H5T_t *)obj);
            break;

        case H5I_DATASET:
            oloc = H5D_oloc((H5D_t *)obj);
            break;

        case H5I_ATTR:
            oloc = H5A_oloc((H5A_t *)obj);
            break;

        case H5I_MAP:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "maps not supported in native VOL connector");

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_DATASPACE:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");
    } 

    
    if (oloc)
        *file = oloc->file;

    
    if (!*file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "object is not associated with a file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
