/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HGmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HGpkg.h"     
#include "H5MMprivate.h" 

static herr_t H5HG__cache_heap_get_initial_load_size(void *udata, size_t *image_len);
static herr_t H5HG__cache_heap_get_final_load_size(const void *_image, size_t image_len, void *udata,
                                                   size_t *actual_len);
static void  *H5HG__cache_heap_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5HG__cache_heap_image_len(const void *thing, size_t *image_len);
static herr_t H5HG__cache_heap_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5HG__cache_heap_free_icr(void *thing);

static herr_t H5HG__hdr_deserialize(H5HG_heap_t *heap, const uint8_t *image, size_t len, const H5F_t *f);

const H5AC_class_t H5AC_GHEAP[1] = {{
    H5AC_GHEAP_ID,                          
    "global heap",                          
    H5FD_MEM_GHEAP,                         
    H5AC__CLASS_SPECULATIVE_LOAD_FLAG,      
    H5HG__cache_heap_get_initial_load_size, 
    H5HG__cache_heap_get_final_load_size,   
    NULL,                                   
    H5HG__cache_heap_deserialize,           
    H5HG__cache_heap_image_len,             
    NULL,                                   
    H5HG__cache_heap_serialize,             
    NULL,                                   
    H5HG__cache_heap_free_icr,              
    NULL,                                   
}};

static herr_t
H5HG__hdr_deserialize(H5HG_heap_t *heap, const uint8_t *image, size_t len, const H5F_t *f)
{
    const uint8_t *p_end     = image + len - 1; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(heap);
    assert(image);
    assert(f);

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5_SIZEOF_MAGIC, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    if (memcmp(image, H5HG_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "bad global heap collection signature");
    image += H5_SIZEOF_MAGIC;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    if (H5HG_VERSION != *image++)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "wrong version number in global heap");

    
    if (H5_IS_BUFFER_OVERFLOW(image, 3, p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    image += 3;

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5F_sizeof_size(f), p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    H5F_DECODE_LENGTH(f, image, heap->size);
    if (heap->size < H5HG_MINSIZE)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, FAIL, "global heap size is too small");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HG__cache_heap_get_initial_load_size(void H5_ATTR_UNUSED *_udata, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(image_len);

    *image_len = H5HG_MINSIZE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HG__cache_heap_get_final_load_size(const void *image, size_t image_len, void *udata, size_t *actual_len)
{
    H5HG_heap_t heap;
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);
    assert(actual_len);
    assert(*actual_len == image_len);
    assert(image_len == H5HG_MINSIZE);

    
    heap.size = 0;
    if (H5HG__hdr_deserialize(&heap, (const uint8_t *)image, image_len, (const H5F_t *)udata) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDECODE, FAIL, "can't decode global heap prefix");

    
    *actual_len = heap.size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5HG__cache_heap_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5F_t         *f         = (H5F_t *)_udata; 
    H5HG_heap_t   *heap      = NULL;            
    uint8_t       *p         = NULL;            
    const uint8_t *p_end     = NULL;            
    size_t         max_idx   = 0;               
    size_t         nalloc    = 0;               
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(_image);
    assert(len >= (size_t)H5HG_MINSIZE);
    assert(f);
    assert(dirty);

    
    if (NULL == (heap = H5FL_CALLOC(H5HG_heap_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    heap->shared = H5F_SHARED(f);
    if (NULL == (heap->chunk = H5FL_BLK_MALLOC(gheap_chunk, len)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    H5MM_memcpy(heap->chunk, _image, len);

    
    p_end = heap->chunk + len - 1;

    
    if (H5_IS_BUFFER_OVERFLOW(heap->chunk, H5HG_SIZEOF_HDR(f), p_end))
        HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (H5HG__hdr_deserialize(heap, (const uint8_t *)heap->chunk, len, f) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDECODE, NULL, "can't decode global heap header");

    

    
    p = heap->chunk + H5HG_SIZEOF_HDR(f);

    
    nalloc = H5HG_NOBJS(f, heap->size);

    
    if (NULL == (heap->obj = H5FL_SEQ_CALLOC(H5HG_obj_t, nalloc)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    heap->nalloc = nalloc;

    while (p < (heap->chunk + heap->size)) {

        if ((p + H5HG_SIZEOF_OBJHDR(f)) > (heap->chunk + heap->size)) {

            
            if (NULL != heap->obj[0].begin)
                HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "object 0 should not be set");
            heap->obj[0].size  = (size_t)(((const uint8_t *)heap->chunk + heap->size) - p);
            heap->obj[0].begin = p;

            
            p += heap->obj[0].size;
        }
        else {
            size_t   need = 0;  
            unsigned idx;       
            uint8_t *begin = p; 

            

            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, idx);

            
            if (idx >= heap->nalloc) {
                size_t      new_alloc; 
                H5HG_obj_t *new_obj;   

                
                new_alloc = MAX(heap->nalloc * 2, (idx + 1));
                if (idx >= new_alloc)
                    HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "inappropriate heap index");

                
                if (NULL == (new_obj = H5FL_SEQ_REALLOC(H5HG_obj_t, heap->obj, new_alloc)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

                
                memset(&new_obj[heap->nalloc], 0, (new_alloc - heap->nalloc) * sizeof(heap->obj[0]));

                
                heap->nalloc = new_alloc;
                heap->obj    = new_obj;
                if (heap->nalloc <= heap->nused)
                    HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "inappropriate # allocated slots");
            }

            
            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, heap->obj[idx].nrefs);

            
            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            p += 4;

            
            if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5F_DECODE_LENGTH(f, p, heap->obj[idx].size);

            
            heap->obj[idx].begin = begin;

            
            if (idx > 0) {
                need = H5HG_SIZEOF_OBJHDR(f) + H5HG_ALIGN(heap->obj[idx].size);
                if (idx > max_idx)
                    max_idx = idx;
            }
            else
                need = heap->obj[idx].size;

            
            if (H5_IS_BUFFER_OVERFLOW(begin, need, p_end))
                HGOTO_ERROR(H5E_HEAP, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            p = begin + need;
        }
    }

    
    if (p != heap->chunk + heap->size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "partially decoded global heap");
    if (false == H5HG_ISALIGNED(heap->obj[0].size))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "decoded global heap is not aligned");

    
    if (max_idx > 0)
        heap->nused = max_idx + 1;
    else
        heap->nused = 1;

    if (max_idx >= heap->nused)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, NULL, "bad `next unused` heap index value");

    
    if (H5F_cwfs_add(f, heap) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "unable to add global heap collection to file's CWFS");

    ret_value = heap;

done:
    if (!ret_value && heap)
        if (H5HG__free(heap) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, NULL, "unable to destroy global heap collection");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HG__cache_heap_image_len(const void *_thing, size_t *image_len)
{
    const H5HG_heap_t *heap = (const H5HG_heap_t *)_thing;

    FUNC_ENTER_PACKAGE_NOERR

    assert(heap);
    assert(heap->cache_info.type == H5AC_GHEAP);
    assert(heap->size >= H5HG_MINSIZE);
    assert(image_len);

    *image_len = heap->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HG__cache_heap_serialize(const H5F_t H5_ATTR_NDEBUG_UNUSED *f, void *image, size_t len, void *_thing)
{
    H5HG_heap_t *heap = (H5HG_heap_t *)_thing;

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(image);
    assert(heap);
    assert(heap->cache_info.type == H5AC_GHEAP);
    assert(heap->size == len);
    assert(heap->chunk);

    
    H5MM_memcpy(image, heap->chunk, len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HG__cache_heap_free_icr(void *_thing)
{
    H5HG_heap_t *heap      = (H5HG_heap_t *)_thing;
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(heap);
    assert(heap->cache_info.type == H5AC_GHEAP);

    
    if (H5HG__free(heap) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy global heap collection");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
