/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 

#define H5HF_TINY_LEN_SHORT  16     
#define H5HF_TINY_MASK_SHORT 0x0F   
#define H5HF_TINY_MASK_EXT   0x0FFF 
#define H5HF_TINY_MASK_EXT_1 0x0F00 
#define H5HF_TINY_MASK_EXT_2 0x00FF 

static herr_t H5HF__tiny_op_real(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data);

herr_t
H5HF__tiny_init(H5HF_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    

    
    
    if ((hdr->id_len - 1) <= H5HF_TINY_LEN_SHORT) {
        hdr->tiny_max_len      = hdr->id_len - 1;
        hdr->tiny_len_extended = false;
    } 
    else if ((hdr->id_len - 1) == (H5HF_TINY_LEN_SHORT + 1)) {
        hdr->tiny_max_len      = H5HF_TINY_LEN_SHORT;
        hdr->tiny_len_extended = false;
    } 
    else {
        hdr->tiny_max_len      = hdr->id_len - 2;
        hdr->tiny_len_extended = true;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__tiny_insert(H5HF_hdr_t *hdr, size_t obj_size, const void *obj, void *_id)
{
    uint8_t *id = (uint8_t *)_id; 
    size_t   enc_obj_size;        
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(obj_size <= hdr->tiny_max_len);
    assert(obj_size <= (H5HF_TINY_MASK_EXT + 1));
    assert(obj);
    assert(id);

    
    enc_obj_size = obj_size - 1;

    
    if (!hdr->tiny_len_extended) {
        *id++ = (uint8_t)(H5HF_ID_VERS_CURR | H5HF_ID_TYPE_TINY | (enc_obj_size & H5HF_TINY_MASK_SHORT));
    } 
    else {
        *id++ =
            (uint8_t)(H5HF_ID_VERS_CURR | H5HF_ID_TYPE_TINY | ((enc_obj_size & H5HF_TINY_MASK_EXT_1) >> 8));
        *id++ = enc_obj_size & H5HF_TINY_MASK_EXT_2;
    } 

    H5MM_memcpy(id, obj, obj_size);
    memset(id + obj_size, 0, (hdr->id_len - ((size_t)1 + (size_t)hdr->tiny_len_extended + obj_size)));

    
    hdr->tiny_size += obj_size;
    hdr->tiny_nobjs++;

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__tiny_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p)
{
    size_t enc_obj_size; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(id);
    assert(obj_len_p);

    
    if (!hdr->tiny_len_extended)
        enc_obj_size = *id & H5HF_TINY_MASK_SHORT;
    else
        
        enc_obj_size = (size_t) * (id + 1) | ((size_t)(*id & H5HF_TINY_MASK_EXT_1) << 8);

    
    *obj_len_p = enc_obj_size + 1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__tiny_op_real(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data)
{
    size_t enc_obj_size;        
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(op);

    
    
    ret_value = H5HF__tiny_get_obj_len(hdr, id, &enc_obj_size);

    
    if (!hdr->tiny_len_extended)
        id++;
    else {
        
        id++;
        id++;
    }

    
    if (op(id, enc_obj_size, op_data) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "application's callback failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__tiny_read(H5HF_hdr_t *hdr, const uint8_t *id, void *obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(obj);

    
    if (H5HF__tiny_op_real(hdr, id, H5HF__op_read, obj) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__tiny_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(op);

    
    if (H5HF__tiny_op_real(hdr, id, op, op_data) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__tiny_remove(H5HF_hdr_t *hdr, const uint8_t *id)
{
    size_t enc_obj_size;        
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);

    
    
    ret_value = H5HF__tiny_get_obj_len(hdr, id, &enc_obj_size);

    
    hdr->tiny_size -= enc_obj_size;
    hdr->tiny_nobjs--;

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
