/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 

#define H5HF_SECT_INDIRECT_SERIAL_SIZE(h)                                                                    \
    ((unsigned)(h)->heap_off_size                               \
     + (unsigned)2                                                                                  \
     + (unsigned)2                                                                               \
     + (unsigned)2                                                                         \
    )

typedef struct {
    H5HF_hdr_t *hdr; 
} H5HF_sect_private_t;

static herr_t               H5FS__sect_init_cls(H5FS_section_class_t *cls, H5HF_hdr_t *hdr);
static herr_t               H5FS__sect_term_cls(H5FS_section_class_t *cls);
static H5HF_free_section_t *H5FS__sect_node_new(unsigned sect_type, haddr_t sect_addr, hsize_t sect_size,
                                                H5FS_section_state_t state);
static herr_t               H5HF__sect_node_free(H5HF_free_section_t *sect, H5HF_indirect_t *parent);

static herr_t H5HF__sect_single_locate_parent(H5HF_hdr_t *hdr, bool refresh, H5HF_free_section_t *sect);
static herr_t H5HF__sect_single_full_dblock(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);

static herr_t               H5HF__sect_single_add(H5FS_section_info_t **sect, unsigned *flags, void *udata);
static H5FS_section_info_t *H5HF__sect_single_deserialize(const H5FS_section_class_t *cls, const uint8_t *buf,
                                                          haddr_t sect_addr, hsize_t sect_size,
                                                          unsigned *des_flags);
static htri_t H5HF__sect_single_can_merge(const H5FS_section_info_t *sect1, const H5FS_section_info_t *sect2,
                                          void *udata);
static herr_t H5HF__sect_single_merge(H5FS_section_info_t **sect1, H5FS_section_info_t *sect2, void *udata);
static htri_t H5HF__sect_single_can_shrink(const H5FS_section_info_t *sect, void *udata);
static herr_t H5HF__sect_single_shrink(H5FS_section_info_t **_sect, void *udata);
static herr_t H5HF__sect_single_valid(const H5FS_section_class_t *cls, const H5FS_section_info_t *sect);

static H5HF_free_section_t *H5HF__sect_row_create(haddr_t sect_off, hsize_t sect_size, bool is_first,
                                                  unsigned row, unsigned col, unsigned nentries,
                                                  H5HF_free_section_t *under_sect);
static herr_t               H5HF__sect_row_first(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
static herr_t               H5HF__sect_row_parent_removed(H5HF_free_section_t *sect);
static herr_t H5HF__sect_row_from_single(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, H5HF_direct_t *dblock);
static herr_t H5HF__sect_row_free_real(H5HF_free_section_t *sect);

static herr_t H5HF__sect_row_init_cls(H5FS_section_class_t *cls, void *udata);
static herr_t H5HF__sect_row_term_cls(H5FS_section_class_t *cls);
static herr_t H5HF__sect_row_serialize(const H5FS_section_class_t *cls, const H5FS_section_info_t *sect,
                                       uint8_t *buf);
static H5FS_section_info_t *H5HF__sect_row_deserialize(const H5FS_section_class_t *cls, const uint8_t *buf,
                                                       haddr_t sect_addr, hsize_t sect_size,
                                                       unsigned *des_flags);
static htri_t H5HF__sect_row_can_merge(const H5FS_section_info_t *sect1, const H5FS_section_info_t *sect2,
                                       void *udata);
static herr_t H5HF__sect_row_merge(H5FS_section_info_t **sect1, H5FS_section_info_t *sect2, void *udata);
static htri_t H5HF__sect_row_can_shrink(const H5FS_section_info_t *sect, void *udata);
static herr_t H5HF__sect_row_shrink(H5FS_section_info_t **sect, void *udata);
static herr_t H5HF__sect_row_free(H5FS_section_info_t *sect);
static herr_t H5HF__sect_row_valid(const H5FS_section_class_t *cls, const H5FS_section_info_t *sect);
static herr_t H5HF__sect_row_debug(const H5FS_section_info_t *sect, FILE *stream, int indent, int fwidth);

static H5HF_free_section_t *H5HF__sect_indirect_new(H5HF_hdr_t *hdr, haddr_t sect_off, hsize_t sect_size,
                                                    H5HF_indirect_t *iblock, hsize_t iblock_off, unsigned row,
                                                    unsigned col, unsigned nentries);
static herr_t H5HF__sect_indirect_init_rows(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, bool first_child,
                                            H5HF_free_section_t **first_row_sect, unsigned space_flags,
                                            unsigned start_row, unsigned start_col, unsigned end_row,
                                            unsigned end_col);
static H5HF_free_section_t *H5HF__sect_indirect_for_row(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock,
                                                        H5HF_free_section_t *row_sect);
static herr_t               H5HF__sect_indirect_decr(H5HF_free_section_t *sect);
static herr_t               H5HF__sect_indirect_revive_row(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
static herr_t               H5HF__sect_indirect_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect,
                                                       H5HF_indirect_t *sect_iblock);
static herr_t               H5HF__sect_indirect_reduce_row(H5HF_hdr_t *hdr, H5HF_free_section_t *row_sect,
                                                           bool *alloc_from_start);
static herr_t H5HF__sect_indirect_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, unsigned child_entry);
static herr_t H5HF__sect_indirect_first(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
static bool   H5HF__sect_indirect_is_first(H5HF_free_section_t *sect);
static H5HF_indirect_t     *H5HF__sect_indirect_get_iblock(H5HF_free_section_t *sect);
static hsize_t              H5HF__sect_indirect_iblock_off(const H5HF_free_section_t *sect);
static H5HF_free_section_t *H5HF__sect_indirect_top(H5HF_free_section_t *sect);
static herr_t               H5HF__sect_indirect_merge_row(H5HF_hdr_t *hdr, H5HF_free_section_t *sect1,
                                                          H5HF_free_section_t *sect2);
static herr_t               H5HF__sect_indirect_build_parent(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
static herr_t               H5HF__sect_indirect_shrink(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
static herr_t H5HF__sect_indirect_serialize(H5HF_hdr_t *hdr, const H5HF_free_section_t *sect, uint8_t *buf);
static H5FS_section_info_t *H5HF__sect_indirect_deserialize(H5HF_hdr_t *hdr, const uint8_t *buf,
                                                            haddr_t sect_addr, hsize_t sect_size,
                                                            unsigned *des_flags);
static herr_t               H5HF__sect_indirect_free(H5HF_free_section_t *sect);
static herr_t               H5HF__sect_indirect_valid(const H5HF_hdr_t *hdr, const H5HF_free_section_t *sect);
static herr_t H5HF__sect_indirect_debug(const H5HF_free_section_t *sect, FILE *stream, int indent,
                                        int fwidth);

static herr_t H5HF__sect_indirect_init_cls(H5FS_section_class_t *cls, void *udata);
static herr_t H5HF__sect_indirect_term_cls(H5FS_section_class_t *cls);

H5FS_section_class_t H5HF_FSPACE_SECT_CLS_SINGLE[1] = {{
    
    H5HF_FSPACE_SECT_SINGLE, 
    0,                       
    H5FS_CLS_MERGE_SYM,      
    NULL,                    

    
    NULL, 
    NULL, 

    
    H5HF__sect_single_add,         
    NULL,                          
    H5HF__sect_single_deserialize, 
    H5HF__sect_single_can_merge,   
    H5HF__sect_single_merge,       
    H5HF__sect_single_can_shrink,  
    H5HF__sect_single_shrink,      
    H5HF__sect_single_free,        
    H5HF__sect_single_valid,       
    NULL,                          
    NULL,                          
}};

H5FS_section_class_t H5HF_FSPACE_SECT_CLS_FIRST_ROW[1] = {{
    
    H5HF_FSPACE_SECT_FIRST_ROW, 
    0,                          
    H5FS_CLS_MERGE_SYM,         
    NULL,                       

    
    H5HF__sect_row_init_cls, 
    H5HF__sect_row_term_cls, 

    
    NULL,                       
    H5HF__sect_row_serialize,   
    H5HF__sect_row_deserialize, 
    H5HF__sect_row_can_merge,   
    H5HF__sect_row_merge,       
    H5HF__sect_row_can_shrink,  
    H5HF__sect_row_shrink,      
    H5HF__sect_row_free,        
    H5HF__sect_row_valid,       
    NULL,                       
    H5HF__sect_row_debug,       
}};

H5FS_section_class_t H5HF_FSPACE_SECT_CLS_NORMAL_ROW[1] = {{
    
    H5HF_FSPACE_SECT_NORMAL_ROW,                                  
    0,                                                            
    H5FS_CLS_MERGE_SYM | H5FS_CLS_SEPAR_OBJ | H5FS_CLS_GHOST_OBJ, 
    NULL,                                                         

    
    H5HF__sect_row_init_cls, 
    H5HF__sect_row_term_cls, 

    
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    H5HF__sect_row_free,  
    H5HF__sect_row_valid, 
    NULL,                 
    H5HF__sect_row_debug, 
}};

H5FS_section_class_t H5HF_FSPACE_SECT_CLS_INDIRECT[1] = {{
    
    H5HF_FSPACE_SECT_INDIRECT,               
    0,                                       
    H5FS_CLS_MERGE_SYM | H5FS_CLS_GHOST_OBJ, 
    NULL,                                    

    
    H5HF__sect_indirect_init_cls, 
    H5HF__sect_indirect_term_cls, 

    
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
}};

H5FL_DEFINE_STATIC(H5HF_free_section_t);

static herr_t
H5FS__sect_init_cls(H5FS_section_class_t *cls, H5HF_hdr_t *hdr)
{
    H5HF_sect_private_t *cls_prvt;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(!cls->cls_private);

    
    if (NULL == (cls_prvt = (H5HF_sect_private_t *)H5MM_malloc(sizeof(H5HF_sect_private_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    cls_prvt->hdr    = hdr;
    cls->cls_private = cls_prvt;

    
    if (H5HF__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared heap header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_term_cls(H5FS_section_class_t *cls)
{
    H5HF_sect_private_t *cls_prvt;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    cls_prvt = (H5HF_sect_private_t *)cls->cls_private;

    
    if (H5HF__hdr_decr(cls_prvt->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared heap header");

    
    cls->cls_private = H5MM_xfree(cls_prvt);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_free_section_t *
H5FS__sect_node_new(unsigned sect_type, haddr_t sect_addr, hsize_t sect_size, H5FS_section_state_t sect_state)
{
    H5HF_free_section_t *new_sect;         
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(H5_addr_defined(sect_addr));

    
    if (NULL == (new_sect = H5FL_MALLOC(H5HF_free_section_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for direct block free list section");

    
    new_sect->sect_info.addr = sect_addr;
    new_sect->sect_info.size = sect_size;

    
    new_sect->sect_info.type  = sect_type;
    new_sect->sect_info.state = sect_state;

    
    ret_value = new_sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_node_free(H5HF_free_section_t *sect, H5HF_indirect_t *iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(sect);

    
    if (iblock)
        if (H5HF__iblock_decr(iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on section's indirect block");

    
    sect = H5FL_FREE(H5HF_free_section_t, sect);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_free_section_t *
H5HF__sect_single_new(hsize_t sect_off, size_t sect_size, H5HF_indirect_t *parent, unsigned par_entry)
{
    H5HF_free_section_t *sect      = NULL; 
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(sect_size);

    
    if (NULL ==
        (sect = H5FS__sect_node_new(H5HF_FSPACE_SECT_SINGLE, sect_off, (hsize_t)sect_size, H5FS_SECT_LIVE)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for single section");

    
    sect->u.single.parent = parent;
    if (sect->u.single.parent) {
        if (H5HF__iblock_incr(sect->u.single.parent) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL,
                        "can't increment reference count on shared indirect block");
    } 
    sect->u.single.par_entry = par_entry;

    
    ret_value = sect;

done:
    if (!ret_value && sect) {
        
        sect = H5FL_FREE(H5HF_free_section_t, sect);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_locate_parent(H5HF_hdr_t *hdr, bool refresh, H5HF_free_section_t *sect)
{
    H5HF_indirect_t *sec_iblock;          
    unsigned         sec_entry;           
    bool             did_protect;         
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->man_dtable.curr_root_rows > 0);
    assert(sect);

    
    if (H5HF__man_dblock_locate(hdr, sect->sect_info.addr, &sec_iblock, &sec_entry, &did_protect,
                                H5AC__READ_ONLY_FLAG) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of section");

    
    if (H5HF__iblock_incr(sec_iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared indirect block");

    
    if (refresh) {
        if (sect->u.single.parent) {
            
            if (H5HF__iblock_decr(sect->u.single.parent) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                            "can't decrement reference count on section's indirect block");
        } 
    }     

    
    sect->u.single.parent    = sec_iblock;
    sect->u.single.par_entry = sec_entry;

    
    if (H5HF__man_iblock_unprotect(sec_iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");
    sec_iblock = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_single_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.state == H5FS_SECT_SERIALIZED);

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        
        assert(H5_addr_defined(hdr->man_dtable.table_addr));
        sect->u.single.parent    = NULL;
        sect->u.single.par_entry = 0;
    } 
    else {
        
        if (H5HF__sect_single_locate_parent(hdr, false, sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get section's parent info");
    } 

    
    sect->sect_info.state = H5FS_SECT_LIVE;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_single_dblock_info(H5HF_hdr_t *hdr, const H5HF_free_section_t *sect, haddr_t *dblock_addr,
                              size_t *dblock_size)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_SINGLE);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);
    assert(dblock_addr);
    assert(dblock_size);

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        
        assert(H5_addr_defined(hdr->man_dtable.table_addr));
        *dblock_addr = hdr->man_dtable.table_addr;
        *dblock_size = hdr->man_dtable.cparam.start_block_size;
    } 
    else {
        
        *dblock_addr = sect->u.single.parent->ents[sect->u.single.par_entry].addr;
        *dblock_size =
            hdr->man_dtable.row_block_size[sect->u.single.par_entry / hdr->man_dtable.cparam.width];
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__sect_single_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, size_t amt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_SINGLE);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);

    
    if (sect->sect_info.size == amt) {
        
        if (H5HF__sect_single_free((H5FS_section_info_t *)sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free single section node");
    } 
    else {
        
        sect->sect_info.addr += amt;
        sect->sect_info.size -= amt;

        
        if (H5HF__space_add(hdr, sect, 0) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't re-add single section to free space manager");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_full_dblock(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    haddr_t dblock_addr = HADDR_UNDEF; 
    size_t  dblock_size = SIZE_MAX;    
    size_t  dblock_overhead;           
    herr_t  ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);
    assert(hdr);

    
    if (H5HF__sect_single_dblock_info(hdr, sect, &dblock_addr, &dblock_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve direct block information");
    if (!H5_addr_defined(dblock_addr))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block address");
    if (SIZE_MAX == dblock_size)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block size");

    
    
    dblock_overhead = H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
    if ((dblock_size - dblock_overhead) == sect->sect_info.size && hdr->man_dtable.curr_root_rows > 0) {
        H5HF_direct_t *dblock;         
        bool           parent_removed; 

        if (NULL == (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, sect->u.single.parent,
                                                       sect->u.single.par_entry, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load fractal heap direct block");
        assert(H5_addr_eq(dblock->block_off + dblock_overhead, sect->sect_info.addr));

        
        if (H5HF__sect_row_from_single(hdr, sect, dblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCONVERT, FAIL, "can't convert single section into row section");

        
        if (H5HF__man_dblock_destroy(hdr, dblock, dblock_addr, &parent_removed) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release direct block");
        dblock = NULL;

        
        if (parent_removed && H5FS_SECT_LIVE == sect->u.row.under->sect_info.state)
            if (H5HF__sect_row_parent_removed(sect) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUPDATE, FAIL, "can't update section info");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_add(H5FS_section_info_t **_sect, unsigned *flags, void *_udata)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (!(*flags & H5FS_ADD_DESERIALIZING)) {
        H5HF_free_section_t **sect  = (H5HF_free_section_t **)_sect; 
        H5HF_sect_add_ud_t   *udata = (H5HF_sect_add_ud_t *)_udata;  
        H5HF_hdr_t           *hdr   = udata->hdr;                    

        
        assert(sect);
        assert(hdr);

        
        
        if (H5HF__sect_single_full_dblock(hdr, (*sect)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCONVERT, FAIL, "can't check/convert single section");

        
        if ((*sect)->sect_info.type != H5HF_FSPACE_SECT_SINGLE)
            *flags |= H5FS_ADD_RETURNED_SPACE;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5FS_section_info_t *
H5HF__sect_single_deserialize(const H5FS_section_class_t H5_ATTR_UNUSED *cls,
                              const uint8_t H5_ATTR_UNUSED *buf, haddr_t sect_addr, hsize_t sect_size,
                              unsigned H5_ATTR_UNUSED *des_flags)
{
    H5HF_free_section_t *new_sect;         
    H5FS_section_info_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(H5_addr_defined(sect_addr));
    assert(sect_size);

    
    if (NULL ==
        (new_sect = H5FS__sect_node_new(H5HF_FSPACE_SECT_SINGLE, sect_addr, sect_size, H5FS_SECT_SERIALIZED)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "allocation failed for direct block free list section");

    
    ret_value = (H5FS_section_info_t *)new_sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5HF__sect_single_can_merge(const H5FS_section_info_t *_sect1, const H5FS_section_info_t *_sect2,
                            void H5_ATTR_UNUSED *_udata)
{
    const H5HF_free_section_t *sect1 = (const H5HF_free_section_t *)_sect1; 
    const H5HF_free_section_t *sect2 = (const H5HF_free_section_t *)_sect2; 
    htri_t                     ret_value = false;                           

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect1);
    assert(sect2);
    assert(sect1->sect_info.type == sect2->sect_info.type); 
    assert(H5_addr_lt(sect1->sect_info.addr, sect2->sect_info.addr));

    
    
    if (H5_addr_eq(sect1->sect_info.addr + sect1->sect_info.size, sect2->sect_info.addr))
        HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_merge(H5FS_section_info_t **_sect1, H5FS_section_info_t *_sect2, void *_udata)
{
    H5HF_free_section_t **sect1     = (H5HF_free_section_t **)_sect1; 
    H5HF_free_section_t  *sect2     = (H5HF_free_section_t *)_sect2;  
    H5HF_sect_add_ud_t   *udata     = (H5HF_sect_add_ud_t *)_udata;   
    H5HF_hdr_t           *hdr       = udata->hdr;                     
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect1);
    assert((*sect1)->sect_info.type == H5HF_FSPACE_SECT_SINGLE);
    assert(sect2);
    assert(sect2->sect_info.type == H5HF_FSPACE_SECT_SINGLE);
    assert(H5_addr_eq((*sect1)->sect_info.addr + (*sect1)->sect_info.size, sect2->sect_info.addr));

    
    (*sect1)->sect_info.size += sect2->sect_info.size;

    
    if (H5HF__sect_single_free((H5FS_section_info_t *)sect2) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

    
    if ((*sect1)->sect_info.state != H5FS_SECT_LIVE)
        if (H5HF__sect_single_revive(hdr, (*sect1)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't revive single free section");

    
    
    if (H5HF__sect_single_full_dblock(hdr, (*sect1)) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCONVERT, FAIL, "can't check/convert single section");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5HF__sect_single_can_shrink(const H5FS_section_info_t *_sect, void *_udata)
{
    const H5HF_free_section_t *sect      = (const H5HF_free_section_t *)_sect; 
    H5HF_sect_add_ud_t        *udata     = (H5HF_sect_add_ud_t *)_udata;       
    H5HF_hdr_t                *hdr       = udata->hdr;                         
    htri_t                     ret_value = false;                              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    
    
    if (hdr->man_dtable.curr_root_rows == 0) {
        size_t dblock_size;     
        size_t dblock_overhead; 

        dblock_size     = hdr->man_dtable.cparam.start_block_size;
        dblock_overhead = H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
        if ((dblock_size - dblock_overhead) == sect->sect_info.size)
            HGOTO_DONE(true);
    } 
    else {
        
        assert(hdr->man_iter_off > sect->sect_info.addr);
        HGOTO_DONE(false);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_shrink(H5FS_section_info_t **_sect, void *_udata)
{
    H5HF_free_section_t **sect  = (H5HF_free_section_t **)_sect; 
    H5HF_sect_add_ud_t   *udata = (H5HF_sect_add_ud_t *)_udata;  
    H5HF_hdr_t           *hdr   = udata->hdr;                    
    H5HF_direct_t        *dblock;                                
    haddr_t               dblock_addr = HADDR_UNDEF;             
    size_t                dblock_size = SIZE_MAX;                
    herr_t                ret_value   = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(*sect);
    assert((*sect)->sect_info.type == H5HF_FSPACE_SECT_SINGLE);

    
    if ((*sect)->sect_info.state != H5FS_SECT_LIVE)
        if (H5HF__sect_single_revive(hdr, (*sect)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't revive single free section");

    
    if (H5HF__sect_single_dblock_info(hdr, (*sect), &dblock_addr, &dblock_size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve direct block information");
    if (!H5_addr_defined(dblock_addr))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block address");
    if (SIZE_MAX == dblock_size)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block size");

    
    
    assert(dblock_addr == hdr->man_dtable.table_addr);
    if (NULL == (dblock = H5HF__man_dblock_protect(hdr, dblock_addr, dblock_size, (*sect)->u.single.parent,
                                                   (*sect)->u.single.par_entry, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to load fractal heap direct block");
    assert(H5_addr_eq(dblock->block_off + dblock_size, (*sect)->sect_info.addr + (*sect)->sect_info.size));

    
    if (H5HF__sect_single_free((H5FS_section_info_t *)*sect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

    
    if (H5HF__man_dblock_destroy(hdr, dblock, dblock_addr, NULL) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release direct block");
    dblock = NULL;

    
    *sect = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_single_free(H5FS_section_info_t *_sect)
{
    H5HF_free_section_t *sect      = (H5HF_free_section_t *)_sect; 
    H5HF_indirect_t     *parent    = NULL;                         
    herr_t               ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect);

    
    if (sect->sect_info.state == H5FS_SECT_LIVE)
        
        if (sect->u.single.parent)
            parent = sect->u.single.parent;

    
    if (H5HF__sect_node_free(sect, parent) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_single_valid(const H5FS_section_class_t H5_ATTR_UNUSED *cls, const H5FS_section_info_t *_sect)
{
    const H5HF_free_section_t *sect = (const H5HF_free_section_t *)_sect; 
    herr_t                     ret_value = SUCCEED;                       

    FUNC_ENTER_PACKAGE

    
    assert(sect);

    if (sect->sect_info.state == H5FS_SECT_LIVE) {
        
        
        if (sect->u.single.parent != NULL) {
            H5HF_indirect_t             *iblock; 
            haddr_t                      dblock_addr   = HADDR_UNDEF; 
            size_t                       dblock_size   = SIZE_MAX;    
            unsigned                     dblock_status = 0; 
            size_t H5_ATTR_NDEBUG_UNUSED dblock_overhead;   
            herr_t H5_ATTR_NDEBUG_UNUSED status;            

            
            iblock = sect->u.single.parent;
            assert(H5_addr_defined(iblock->ents[sect->u.single.par_entry].addr));

            
            if (H5HF__sect_single_dblock_info(iblock->hdr, (const H5HF_free_section_t *)sect, &dblock_addr,
                                              &dblock_size) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve direct block information");
            if (!H5_addr_defined(dblock_addr))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block address");
            if (SIZE_MAX == dblock_size)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve data block size");
            assert(H5_addr_eq(iblock->ents[sect->u.single.par_entry].addr, dblock_addr));

            
            assert(sect->sect_info.addr < iblock->hdr->man_iter_off);

            
            dblock_overhead = H5HF_MAN_ABS_DIRECT_OVERHEAD(iblock->hdr);
            assert((sect->sect_info.size + dblock_overhead) < dblock_size);

            
            status = H5AC_get_entry_status(iblock->hdr->f, dblock_addr, &dblock_status);
            assert(status >= 0);

            
            if (!(dblock_status & H5AC_ES__IS_PROTECTED)) {
                H5HF_direct_t *dblock; 

                
                dblock = H5HF__man_dblock_protect(iblock->hdr, dblock_addr, dblock_size, iblock,
                                                  sect->u.single.par_entry, H5AC__READ_ONLY_FLAG);
                assert(dblock);

                
                assert(dblock_size == dblock->size);
                assert(dblock->size > sect->sect_info.size);
                assert(H5_addr_lt(dblock->block_off, sect->sect_info.addr));
                assert(H5_addr_ge((dblock->block_off + dblock->size),
                                  (sect->sect_info.addr + sect->sect_info.size)));

                
                status = H5AC_unprotect(iblock->hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock,
                                        H5AC__NO_FLAGS_SET);
                assert(status >= 0);
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_free_section_t *
H5HF__sect_row_create(haddr_t sect_off, hsize_t sect_size, bool is_first, unsigned row, unsigned col,
                      unsigned nentries, H5HF_free_section_t *under_sect)
{
    H5HF_free_section_t *sect      = NULL; 
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(sect_size);
    assert(nentries);
    assert(under_sect);

    
    
    if (NULL == (sect = H5FS__sect_node_new(
                     (unsigned)(is_first ? H5HF_FSPACE_SECT_FIRST_ROW : H5HF_FSPACE_SECT_NORMAL_ROW),
                     sect_off, sect_size, under_sect->sect_info.state)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for row section");

    
    sect->u.row.under       = under_sect;
    sect->u.row.row         = row;
    sect->u.row.col         = col;
    sect->u.row.num_entries = nentries;
    sect->u.row.checked_out = false;

    
    ret_value = sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_from_single(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, H5HF_direct_t *dblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(dblock);

    
    sect->sect_info.addr    = dblock->block_off;
    sect->sect_info.type    = H5HF_FSPACE_SECT_FIRST_ROW;
    sect->u.row.row         = dblock->par_entry / hdr->man_dtable.cparam.width;
    sect->u.row.col         = dblock->par_entry % hdr->man_dtable.cparam.width;
    sect->u.row.num_entries = 1;
    sect->u.row.checked_out = false;

    
    if (NULL == (sect->u.row.under = H5HF__sect_indirect_for_row(hdr, dblock->parent, sect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL, "serializing row section not supported yet");

    
    if (H5HF__iblock_decr(dblock->parent) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared indirect block");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_row_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->u.row.under);

    
    if ((H5FS_SECT_LIVE == sect->u.row.under->sect_info.state) &&
        (true == sect->u.row.under->u.indirect.u.iblock->removed_from_cache))
        if (H5HF__sect_row_parent_removed(sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUPDATE, FAIL, "can't update section info");

    
    
    if (H5HF__sect_indirect_revive_row(hdr, sect->u.row.under) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTREVIVE, FAIL, "can't revive indirect section");
    assert(sect->sect_info.state == H5FS_SECT_LIVE);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_row_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, unsigned *entry_p)
{
    bool   alloc_from_start;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW ||
           sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);
    assert(entry_p);

    
    assert(sect->u.row.checked_out == false);
    sect->u.row.checked_out = true;

    
    alloc_from_start = false;
    if (H5HF__sect_indirect_reduce_row(hdr, sect, &alloc_from_start) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce underlying section");

    
    *entry_p = (sect->u.row.row * hdr->man_dtable.cparam.width) + sect->u.row.col;
    if (!alloc_from_start)
        *entry_p += (sect->u.row.num_entries - 1);

    
    if (sect->u.row.num_entries == 1) {
        
        if (H5HF__sect_row_free((H5FS_section_info_t *)sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free row section node");
    } 
    else {
        
        if (alloc_from_start) {
            
            sect->sect_info.addr += hdr->man_dtable.row_block_size[sect->u.row.row];
            sect->u.row.col++;
        } 

        
        sect->u.row.num_entries--;

        
        sect->u.row.checked_out = false;

        
        if (H5HF__space_add(hdr, sect, 0) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't re-add indirect section to free space manager");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_first(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);

    
    if (sect->u.row.checked_out)
        sect->sect_info.type = H5HF_FSPACE_SECT_FIRST_ROW;
    else
        
        if (H5HF__space_sect_change_class(hdr, sect, H5HF_FSPACE_SECT_FIRST_ROW) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL, "can't set row section to be first row");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_indirect_t *
H5HF__sect_row_get_iblock(H5HF_free_section_t *sect)
{
    H5HF_indirect_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW ||
           sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);

    ret_value = H5HF__sect_indirect_get_iblock(sect->u.row.under);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_parent_removed(H5HF_free_section_t *sect)
{
    hsize_t  tmp_iblock_off;      
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sect);

    
    tmp_iblock_off = sect->u.row.under->u.indirect.u.iblock->block_off;

    
    if (H5HF__iblock_decr(sect->u.row.under->u.indirect.u.iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared indirect block");

    
    
    sect->u.row.under->u.indirect.u.iblock_off   = tmp_iblock_off;
    sect->u.row.under->u.indirect.iblock_entries = 0;

    
    for (u = 0; u < sect->u.row.under->u.indirect.dir_nrows; u++)
        sect->u.row.under->u.indirect.dir_rows[u]->sect_info.state = H5FS_SECT_SERIALIZED;

    
    sect->u.row.under->sect_info.state = H5FS_SECT_SERIALIZED;

    
    sect->sect_info.state = H5FS_SECT_SERIALIZED;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_init_cls(H5FS_section_class_t *cls, void *_udata)
{
    H5HF_hdr_t *hdr       = (H5HF_hdr_t *)_udata; 
    herr_t      ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(hdr);

    
    if (H5FS__sect_init_cls(cls, hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize common section class");

    
    if (cls->type == H5HF_FSPACE_SECT_FIRST_ROW)
        cls->serial_size = H5HF_SECT_INDIRECT_SERIAL_SIZE(hdr);
    else
        cls->serial_size = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_term_cls(H5FS_section_class_t *cls)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (H5FS__sect_term_cls(cls) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't terminate common section class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_serialize(const H5FS_section_class_t *cls, const H5FS_section_info_t *_sect, uint8_t *buf)
{
    H5HF_hdr_t                *hdr; 
    const H5HF_free_section_t *sect      = (const H5HF_free_section_t *)_sect;
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(buf);
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);
    assert(sect->sect_info.addr == sect->u.row.under->sect_info.addr);

    
    hdr = ((H5HF_sect_private_t *)(cls->cls_private))->hdr;
    if (H5HF__sect_indirect_serialize(hdr, sect->u.row.under, buf) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSERIALIZE, FAIL,
                    "can't serialize row section's underlying indirect section");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5FS_section_info_t *
H5HF__sect_row_deserialize(const H5FS_section_class_t *cls, const uint8_t *buf, haddr_t sect_addr,
                           hsize_t sect_size, unsigned *des_flags)
{
    H5HF_hdr_t          *hdr;              
    H5FS_section_info_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(buf);
    assert(H5_addr_defined(sect_addr));
    assert(sect_size);

    
    hdr = ((H5HF_sect_private_t *)(cls->cls_private))->hdr;
    if (NULL == (ret_value = H5HF__sect_indirect_deserialize(hdr, buf, sect_addr, sect_size, des_flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDECODE, NULL,
                    "can't deserialize row section's underlying indirect section");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5HF__sect_row_can_merge(const H5FS_section_info_t *_sect1, const H5FS_section_info_t *_sect2,
                         void H5_ATTR_UNUSED *_udata)
{
    const H5HF_free_section_t *sect1 = (const H5HF_free_section_t *)_sect1; 
    const H5HF_free_section_t *sect2 = (const H5HF_free_section_t *)_sect2; 
    H5HF_free_section_t       *top_indir_sect1, *top_indir_sect2; 
    htri_t                     ret_value = false;                 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect1);
    assert(sect1->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);
    assert(sect2);
    assert(sect1->sect_info.type == sect2->sect_info.type); 
    assert(H5_addr_lt(sect1->sect_info.addr, sect2->sect_info.addr));

    
    top_indir_sect1 = H5HF__sect_indirect_top(sect1->u.row.under);
    assert(top_indir_sect1);
    top_indir_sect2 = H5HF__sect_indirect_top(sect2->u.row.under);
    assert(top_indir_sect2);

    
    if (top_indir_sect1 != top_indir_sect2)
        if (H5HF__sect_indirect_iblock_off(sect1->u.row.under) ==
            H5HF__sect_indirect_iblock_off(sect2->u.row.under))
            
            if (H5_addr_eq((top_indir_sect1->sect_info.addr + top_indir_sect1->u.indirect.span_size),
                           top_indir_sect2->sect_info.addr))
                HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_merge(H5FS_section_info_t **_sect1, H5FS_section_info_t *_sect2, void *_udata)
{
    H5HF_free_section_t **sect1     = (H5HF_free_section_t **)_sect1; 
    H5HF_free_section_t  *sect2     = (H5HF_free_section_t *)_sect2;  
    H5HF_sect_add_ud_t   *udata     = (H5HF_sect_add_ud_t *)_udata;   
    H5HF_hdr_t           *hdr       = udata->hdr;                     
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect1);
    assert((*sect1)->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);
    assert(sect2);
    assert(sect2->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);

    
    if (sect2->sect_info.addr >= hdr->man_iter_off) {
        H5HF_free_section_t *top_indir_sect; 

        
        top_indir_sect = H5HF__sect_indirect_top(sect2->u.row.under);

        
        if (H5HF__sect_indirect_shrink(hdr, top_indir_sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't shrink underlying indirect section");
    } 
    else
        
        if (H5HF__sect_indirect_merge_row(hdr, (*sect1), sect2) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTMERGE, FAIL, "can't merge underlying indirect sections");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5HF__sect_row_can_shrink(const H5FS_section_info_t *_sect, void H5_ATTR_UNUSED *_udata)
{
    const H5HF_free_section_t *sect      = (const H5HF_free_section_t *)_sect; 
    H5HF_sect_add_ud_t        *udata     = (H5HF_sect_add_ud_t *)_udata;       
    H5HF_hdr_t                *hdr       = udata->hdr;                         
    htri_t                     ret_value = false;                              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);

    
    if (sect->sect_info.addr >= hdr->man_iter_off)
        HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_shrink(H5FS_section_info_t **_sect, void *_udata)
{
    H5HF_free_section_t **sect = (H5HF_free_section_t **)_sect;     
    H5HF_free_section_t  *top_indir_sect;                           
    H5HF_sect_add_ud_t   *udata     = (H5HF_sect_add_ud_t *)_udata; 
    H5HF_hdr_t           *hdr       = udata->hdr;                   
    herr_t                ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(*sect);
    assert((*sect)->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);

    
    top_indir_sect = H5HF__sect_indirect_top((*sect)->u.row.under);

    
    if (H5HF__sect_indirect_shrink(hdr, top_indir_sect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't shrink underlying indirect section");

    
    *sect = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_free_real(H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(sect);

    
    if (H5HF__sect_node_free(sect, NULL) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_free(H5FS_section_info_t *_sect)
{
    H5HF_free_section_t *sect      = (H5HF_free_section_t *)_sect; 
    herr_t               ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    assert(sect);
    assert(sect->u.row.under);

    
    if (H5HF__sect_indirect_decr(sect->u.row.under) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't detach section node");

    
    if (H5HF__sect_row_free_real(sect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_row_valid(const H5FS_section_class_t *cls, const H5FS_section_info_t *_sect)
{
    H5HF_sect_private_t       *cls_prvt;                                  
    const H5HF_hdr_t          *hdr;                                       
    const H5HF_free_section_t *sect = (const H5HF_free_section_t *)_sect; 
    const H5HF_free_section_t *indir_sect;    
    unsigned H5_ATTR_NDEBUG_UNUSED indir_idx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cls);
    assert(sect);

    
    cls_prvt = (H5HF_sect_private_t *)cls->cls_private;
    hdr      = cls_prvt->hdr;

    
    assert(sect->u.row.under);
    assert(sect->u.row.num_entries);
    assert(sect->u.row.checked_out == false);
    indir_sect = sect->u.row.under;
    indir_idx  = sect->u.row.row - indir_sect->u.indirect.row;
    assert(indir_sect->u.indirect.dir_rows[indir_idx] == sect);

    
    assert(sect->sect_info.addr < hdr->man_iter_off);

    
    if (sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW) {
        H5HF_free_section_t *top_indir_sect; 

        
        assert(sect->u.row.row == indir_sect->u.indirect.row);

        
        top_indir_sect = H5HF__sect_indirect_top(sect->u.row.under);

        
        H5HF__sect_indirect_valid(hdr, top_indir_sect);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__sect_row_debug(const H5FS_section_info_t *_sect, FILE *stream, int indent, int fwidth)
{
    const H5HF_free_section_t *sect = (const H5HF_free_section_t *)_sect; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Row:", sect->u.row.row);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Column:", sect->u.row.col);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Number of entries:", sect->u.row.num_entries);

    
    if (sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW) {
        
        Rfprintf(stream, "%*s%-*s\n", indent, "", fwidth, "Underlying indirect section:");

        H5HF__sect_indirect_debug(sect->u.row.under, stream, indent + 3, MAX(0, fwidth - 3));
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static hsize_t
H5HF__sect_indirect_iblock_off(const H5HF_free_section_t *sect)
{
    hsize_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    ret_value = sect->sect_info.state == H5FS_SECT_LIVE ? sect->u.indirect.u.iblock->block_off
                                                        : sect->u.indirect.u.iblock_off;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_free_section_t *
H5HF__sect_indirect_top(H5HF_free_section_t *sect)
{
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    if (sect->u.indirect.parent)
        ret_value = H5HF__sect_indirect_top(sect->u.indirect.parent);
    else
        ret_value = sect;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_init_cls(H5FS_section_class_t *cls, void *_udata)
{
    H5HF_hdr_t *hdr       = (H5HF_hdr_t *)_udata; 
    herr_t      ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(hdr);

    
    if (H5FS__sect_init_cls(cls, hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize common section class");

    
    cls->serial_size = H5HF_SECT_INDIRECT_SERIAL_SIZE(hdr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_term_cls(H5FS_section_class_t *cls)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (H5FS__sect_term_cls(cls) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't terminate common section class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_free_section_t *
H5HF__sect_indirect_new(H5HF_hdr_t *hdr, haddr_t sect_off, hsize_t sect_size, H5HF_indirect_t *iblock,
                        hsize_t iblock_off, unsigned row, unsigned col, unsigned nentries)
{
    H5HF_free_section_t *sect      = NULL; 
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(nentries);

    
    if (NULL == (sect = H5FS__sect_node_new(H5HF_FSPACE_SECT_INDIRECT, sect_off, sect_size,
                                            (iblock ? H5FS_SECT_LIVE : H5FS_SECT_SERIALIZED))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for indirect section");

    
    if (iblock) {
        sect->u.indirect.u.iblock       = iblock;
        sect->u.indirect.iblock_entries = hdr->man_dtable.cparam.width * sect->u.indirect.u.iblock->max_rows;
        if (H5HF__iblock_incr(sect->u.indirect.u.iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL,
                        "can't increment reference count on shared indirect block");
    } 
    else {
        sect->u.indirect.u.iblock_off   = iblock_off;
        sect->u.indirect.iblock_entries = 0;
    } 
    sect->u.indirect.row         = row;
    sect->u.indirect.col         = col;
    sect->u.indirect.num_entries = nentries;

    
    sect->u.indirect.span_size = H5HF__dtable_span_size(&hdr->man_dtable, row, col, nentries);
    assert(sect->u.indirect.span_size > 0);

    
    sect->u.indirect.parent    = NULL;
    sect->u.indirect.par_entry = 0;

    
    ret_value = sect;

done:
    if (!ret_value && sect) {
        
        sect = H5FL_FREE(H5HF_free_section_t, sect);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_free_section_t *
H5HF__sect_indirect_for_row(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, H5HF_free_section_t *row_sect)
{
    H5HF_free_section_t *sect      = NULL; 
    H5HF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(iblock);
    assert(row_sect);
    assert(row_sect->u.row.row < hdr->man_dtable.max_direct_rows);

    
    if (NULL == (sect = H5HF__sect_indirect_new(hdr, row_sect->sect_info.addr, row_sect->sect_info.size,
                                                iblock, iblock->block_off, row_sect->u.row.row,
                                                row_sect->u.row.col, row_sect->u.row.num_entries)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't create indirect section");

    
    sect->u.indirect.dir_nrows = 1;

    
    if (NULL ==
        (sect->u.indirect.dir_rows = (H5HF_free_section_t **)H5MM_malloc(sizeof(H5HF_free_section_t *))))
        HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, NULL, "allocation failed for row section pointer array");

    
    sect->u.indirect.dir_rows[0] = row_sect;
    sect->u.indirect.rc          = 1;

    
    sect->u.indirect.indir_nents = 0;
    sect->u.indirect.indir_ents  = NULL;

    
    ret_value = sect;

done:
    if (!ret_value && sect)
        if (H5HF__sect_indirect_free(sect) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, NULL, "can't free indirect section node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_init_rows(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, bool first_child,
                              H5HF_free_section_t **first_row_sect, unsigned space_flags, unsigned start_row,
                              unsigned start_col, unsigned end_row, unsigned end_col)
{
    hsize_t  curr_off;            
    size_t   dblock_overhead;     
    unsigned row_entries;         
    unsigned row_col;             
    unsigned curr_entry;          
    unsigned curr_indir_entry;    
    unsigned curr_row;            
    unsigned dir_nrows;           
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(sect->u.indirect.span_size > 0);

    
    
    sect->u.indirect.rc         = 0;
    sect->u.indirect.dir_rows   = NULL;
    sect->u.indirect.indir_ents = NULL;

    
    if (start_row < hdr->man_dtable.max_direct_rows) {
        unsigned max_direct_row; 

        
        max_direct_row = MIN(end_row, (hdr->man_dtable.max_direct_rows - 1));

        
        dir_nrows = (max_direct_row - start_row) + 1;

        
        sect->u.indirect.dir_nrows = 0;

        
        if (NULL == (sect->u.indirect.dir_rows =
                         (H5HF_free_section_t **)H5MM_malloc(sizeof(H5HF_free_section_t *) * dir_nrows)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "allocation failed for row section pointer array");
    } 
    else {
        
        dir_nrows                  = 0;
        sect->u.indirect.dir_nrows = 0;
    } 

    
    if (end_row >= hdr->man_dtable.max_direct_rows) {
        unsigned indirect_start_row;   
        unsigned indirect_start_col;   
        unsigned indirect_start_entry; 
        unsigned indirect_end_entry;   

        
        if (start_row < hdr->man_dtable.max_direct_rows) {
            indirect_start_row = hdr->man_dtable.max_direct_rows;
            indirect_start_col = 0;
        } 
        else {
            indirect_start_row = start_row;
            indirect_start_col = start_col;
        } 
        indirect_start_entry = (indirect_start_row * hdr->man_dtable.cparam.width) + indirect_start_col;

        
        indirect_end_entry = (end_row * hdr->man_dtable.cparam.width) + end_col;

        
        sect->u.indirect.indir_nents = (indirect_end_entry - indirect_start_entry) + 1;

        
        if (NULL == (sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_malloc(
                         sizeof(H5HF_free_section_t *) * sect->u.indirect.indir_nents)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "allocation failed for indirect section pointer array");
    } 
    else {
        
        sect->u.indirect.indir_nents = 0;
    } 

    
    if (start_row == end_row)
        row_entries = (end_col - start_col) + 1;
    else
        row_entries = hdr->man_dtable.cparam.width - start_col;
    row_col = start_col;

    
    curr_off         = sect->sect_info.addr;
    curr_entry       = (start_row * hdr->man_dtable.cparam.width) + start_col;
    curr_row         = 0;
    curr_indir_entry = 0;
    dblock_overhead  = H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
    for (u = start_row; u <= end_row; u++, curr_row++) {
        if (u < hdr->man_dtable.max_direct_rows) {
            H5HF_free_section_t *row_sect = NULL; 

            
            if (NULL == (row_sect = H5HF__sect_row_create(
                             curr_off, (hdr->man_dtable.row_block_size[u] - dblock_overhead), first_child, u,
                             row_col, row_entries, sect)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL, "creation failed for child row section");

            
            sect->u.indirect.dir_rows[curr_row] = row_sect;

            
            if (first_row_sect)
                *first_row_sect = row_sect;
            else
                
                if (H5HF__space_add(hdr, row_sect, space_flags) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add row section to free space");

            
            sect->u.indirect.rc++;

            
            curr_off += row_entries * hdr->man_dtable.row_block_size[u];

            
            curr_entry += row_entries;

            
            first_child    = false;
            first_row_sect = NULL;
        } 
        else {
            H5HF_indirect_t     *child_iblock;   
            H5HF_free_section_t *child_sect;     
            unsigned             child_nrows;    
            unsigned             child_nentries; 
            unsigned             v;              

            
            child_nrows    = H5HF__dtable_size_to_rows(&hdr->man_dtable, hdr->man_dtable.row_block_size[u]);
            child_nentries = child_nrows * hdr->man_dtable.cparam.width;

            
            for (v = 0; v < row_entries; v++) {
                bool did_protect = false; 

                
                if (sect->sect_info.state == H5FS_SECT_LIVE) {
                    haddr_t child_iblock_addr; 

                    
                    if (H5HF__man_iblock_entry_addr(sect->u.indirect.u.iblock, curr_entry,
                                                    &child_iblock_addr) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                                    "unable to retrieve child indirect block's address");

                    
                    if (H5_addr_defined(child_iblock_addr)) {
                        if (NULL == (child_iblock = H5HF__man_iblock_protect(
                                         hdr, child_iblock_addr, child_nrows, sect->u.indirect.u.iblock,
                                         curr_entry, false, H5AC__NO_FLAGS_SET, &did_protect)))
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL,
                                        "unable to protect fractal heap indirect block");
                    } 
                    else
                        child_iblock = NULL;
                } 
                else
                    child_iblock = NULL;

                
                if (NULL == (child_sect = H5HF__sect_indirect_new(hdr, curr_off, (hsize_t)0, child_iblock,
                                                                  curr_off, 0, 0, child_nentries)))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create indirect section");

                
                if (H5HF__sect_indirect_init_rows(hdr, child_sect, first_child, first_row_sect, space_flags,
                                                  0, 0, (child_nrows - 1),
                                                  (hdr->man_dtable.cparam.width - 1)) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize indirect section");

                
                
                if (child_iblock)
                    if (H5HF__man_iblock_unprotect(child_iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL,
                                    "unable to release fractal heap indirect block");

                
                child_sect->u.indirect.parent                 = sect;
                child_sect->u.indirect.par_entry              = curr_entry;
                sect->u.indirect.indir_ents[curr_indir_entry] = child_sect;
                sect->u.indirect.rc++;

                
                curr_off += hdr->man_dtable.row_block_size[u];

                
                curr_entry++;
                curr_indir_entry++;

                
                first_child    = false;
                first_row_sect = NULL;
            } 
        }     

        
        if (u < (end_row - 1))
            row_entries = hdr->man_dtable.cparam.width;
        else
            row_entries = end_col + 1;

        
        row_col = 0;
    } 

    
    sect->u.indirect.dir_nrows = dir_nrows;

    
    assert(sect->u.indirect.rc == (sect->u.indirect.indir_nents + sect->u.indirect.dir_nrows));

done:
    if (ret_value < 0) {
        if (sect->u.indirect.indir_ents)
            H5MM_xfree(sect->u.indirect.indir_ents);
        if (sect->u.indirect.dir_rows)
            H5MM_xfree(sect->u.indirect.dir_rows);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__sect_indirect_add(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, unsigned start_entry, unsigned nentries)
{
    H5HF_free_section_t *sect           = NULL; 
    H5HF_free_section_t *first_row_sect = NULL; 
    hsize_t              sect_off;              
    unsigned             start_row;             
    unsigned             start_col;             
    unsigned             end_entry;             
    unsigned             end_row;               
    unsigned             end_col;               
    unsigned             u;                     
    herr_t               ret_value = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(iblock);
    assert(nentries);

    
    start_row = start_entry / hdr->man_dtable.cparam.width;
    start_col = start_entry % hdr->man_dtable.cparam.width;

    
    end_entry = (start_entry + nentries) - 1;
    end_row   = end_entry / hdr->man_dtable.cparam.width;
    end_col   = end_entry % hdr->man_dtable.cparam.width;

    
    sect_off = iblock->block_off;
    for (u = 0; u < start_row; u++)
        sect_off += hdr->man_dtable.row_block_size[u] * hdr->man_dtable.cparam.width;
    sect_off += hdr->man_dtable.row_block_size[start_row] * start_col;

    
    if (NULL == (sect = H5HF__sect_indirect_new(hdr, sect_off, (hsize_t)0, iblock, iblock->block_off,
                                                start_row, start_col, nentries)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create indirect section");

    
    if (H5HF__sect_indirect_init_rows(hdr, sect, true, &first_row_sect, H5FS_ADD_SKIP_VALID, start_row,
                                      start_col, end_row, end_col) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize indirect section");
    assert(first_row_sect);

    
    if (H5HF__space_add(hdr, first_row_sect, H5FS_ADD_RETURNED_SPACE) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add row section to free space");

done:
    if (ret_value < 0 && sect)
        if (H5HF__sect_indirect_free(sect) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_decr(H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(sect->u.indirect.rc);

    
    sect->u.indirect.rc--;

    
    if (sect->u.indirect.rc == 0) {
        H5HF_free_section_t *par_sect; 

        
        par_sect = sect->u.indirect.parent;

        
        if (H5HF__sect_indirect_free(sect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");

        
        if (par_sect)
            if (H5HF__sect_indirect_decr(par_sect) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL,
                            "can't decrement ref. count on parent indirect section");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_revive_row(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    H5HF_indirect_t *sec_iblock;          
    bool             did_protect;         
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.state == H5FS_SECT_SERIALIZED);

    
    if (H5HF__man_dblock_locate(hdr, sect->sect_info.addr, &sec_iblock, NULL, &did_protect,
                                H5AC__READ_ONLY_FLAG) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of section");

    
    if (H5HF__sect_indirect_revive(hdr, sect, sec_iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTREVIVE, FAIL, "can't revive indirect section");

done:
    
    if (sec_iblock && H5HF__man_iblock_unprotect(sec_iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, H5HF_indirect_t *sect_iblock)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->sect_info.state == H5FS_SECT_SERIALIZED);
    assert(sect_iblock);

    
    if (H5HF__iblock_incr(sect_iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared indirect block");

    
    sect->u.indirect.u.iblock = sect_iblock;

    
    sect->u.indirect.iblock_entries = hdr->man_dtable.cparam.width * sect->u.indirect.u.iblock->max_rows;

    
    sect->sect_info.state = H5FS_SECT_LIVE;

    
    for (u = 0; u < sect->u.indirect.dir_nrows; u++)
        sect->u.indirect.dir_rows[u]->sect_info.state = H5FS_SECT_LIVE;

    
    if (sect->u.indirect.parent && sect->u.indirect.parent->sect_info.state == H5FS_SECT_SERIALIZED)
        if (H5HF__sect_indirect_revive(hdr, sect->u.indirect.parent, sect->u.indirect.u.iblock->parent) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREVIVE, FAIL, "can't revive indirect section");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_reduce_row(H5HF_hdr_t *hdr, H5HF_free_section_t *row_sect, bool *alloc_from_start)
{
    H5HF_free_section_t *sect;                
    unsigned             row_start_entry;     
    unsigned             row_end_entry;       
    unsigned             row_entry;           
    unsigned             start_entry;         
    unsigned             start_row;           
    unsigned             start_col;           
    unsigned             end_entry;           
    unsigned             end_row;             
    H5HF_free_section_t *peer_sect = NULL;    
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(row_sect);

    
    row_start_entry = (row_sect->u.row.row * hdr->man_dtable.cparam.width) + row_sect->u.row.col;
    row_end_entry   = (row_start_entry + row_sect->u.row.num_entries) - 1;

    
    sect        = row_sect->u.row.under;
    start_row   = sect->u.indirect.row;
    start_col   = sect->u.indirect.col;
    start_entry = (start_row * hdr->man_dtable.cparam.width) + start_col;
    end_entry   = (start_entry + sect->u.indirect.num_entries) - 1;
    end_row     = end_entry / hdr->man_dtable.cparam.width;

    
    assert(sect->u.indirect.span_size > 0);
    assert(sect->u.indirect.iblock_entries > 0);
    assert(sect->u.indirect.dir_nrows > 0);
    assert(sect->u.indirect.dir_rows);
    assert(sect->u.indirect.dir_rows[(row_sect->u.row.row - start_row)] == row_sect);

    
    if (row_end_entry == end_entry && start_row != end_row) {
        *alloc_from_start = false;
        row_entry         = row_end_entry;
    } 
    else {
        *alloc_from_start = true;
        row_entry         = row_start_entry;
    } 

    
    if (sect->u.indirect.parent) {
        bool is_first; 

        
        is_first = H5HF__sect_indirect_is_first(sect);

        
        if (H5HF__sect_indirect_reduce(hdr, sect->u.indirect.parent, sect->u.indirect.par_entry) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce parent indirect section");
        sect->u.indirect.parent    = NULL;
        sect->u.indirect.par_entry = 0;

        
        if (!is_first)
            if (H5HF__sect_indirect_first(hdr, sect) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't make new 'first row' for indirect section");
    } 

    
    sect->u.indirect.span_size -= row_sect->sect_info.size;

    
    if (sect->u.indirect.num_entries > 1) {
        if (row_entry == start_entry) {
            
            sect->sect_info.addr += hdr->man_dtable.row_block_size[sect->u.indirect.row];

            
            sect->u.indirect.col++;
            if (sect->u.indirect.col == hdr->man_dtable.cparam.width) {
                assert(row_sect->u.row.num_entries == 1);

                
                sect->u.indirect.row++;
                sect->u.indirect.col = 0;

                
                sect->u.indirect.dir_nrows--;

                
                if (sect->u.indirect.dir_nrows > 0) {
                    assert(sect->u.indirect.dir_rows);
                    memmove(&sect->u.indirect.dir_rows[0], &sect->u.indirect.dir_rows[1],
                            sect->u.indirect.dir_nrows * sizeof(H5HF_free_section_t *));
                    assert(sect->u.indirect.dir_rows[0]);

                    
                    if (row_sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW)
                        if (H5HF__sect_row_first(hdr, sect->u.indirect.dir_rows[0]) < 0)
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                                        "can't make new 'first row' for indirect section");
                } 
                else {
                    
                    assert(sect->u.indirect.indir_nents > 0);
                    assert(sect->u.indirect.indir_ents);

                    
                    sect->u.indirect.dir_rows = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.dir_rows);

                    
                    if (row_sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW)
                        if (H5HF__sect_indirect_first(hdr, sect->u.indirect.indir_ents[0]) < 0)
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                                        "can't make new 'first row' for child indirect section");
                } 
            }     

            
            sect->u.indirect.num_entries--;
        } 
        else if (row_entry == end_entry) {
            unsigned new_end_row; 

            
            assert(sect->u.indirect.indir_nents == 0);
            assert(sect->u.indirect.indir_ents == NULL);

            
            sect->u.indirect.num_entries--;

            
            new_end_row = ((start_entry + sect->u.indirect.num_entries) - 1) / hdr->man_dtable.cparam.width;
            assert(new_end_row <= end_row);
            if (new_end_row < end_row) {
                assert(new_end_row == (end_row - 1));
                sect->u.indirect.dir_nrows--;
            } 
        }     
        else {
            H5HF_indirect_t *iblock;         
            hsize_t          iblock_off;     
            unsigned         peer_nentries;  
            unsigned         peer_dir_nrows; 
            unsigned         new_start_row;  
            unsigned         u;              

            
            assert(row_sect->u.row.col == 0);
            assert(row_sect->u.row.row > 0);
            assert(row_sect->u.row.row < hdr->man_dtable.max_direct_rows);
            assert(row_sect->u.row.num_entries == hdr->man_dtable.cparam.width);
            assert(row_sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);

            
            new_start_row  = row_sect->u.row.row;
            peer_nentries  = row_entry - start_entry;
            peer_dir_nrows = new_start_row - start_row;

            
            if (sect->sect_info.state == H5FS_SECT_LIVE) {
                iblock     = sect->u.indirect.u.iblock;
                iblock_off = sect->u.indirect.u.iblock->block_off;
            } 
            else {
                iblock     = NULL;
                iblock_off = sect->u.indirect.u.iblock_off;
            } 

            
            if (NULL ==
                (peer_sect = H5HF__sect_indirect_new(hdr, sect->sect_info.addr, sect->sect_info.size, iblock,
                                                     iblock_off, start_row, start_col, peer_nentries)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create indirect section");

            
            peer_sect->u.indirect.indir_nents = 0;
            peer_sect->u.indirect.indir_ents  = NULL;
            peer_sect->u.indirect.dir_nrows   = peer_dir_nrows;
            if (NULL == (peer_sect->u.indirect.dir_rows = (H5HF_free_section_t **)H5MM_malloc(
                             sizeof(H5HF_free_section_t *) * peer_dir_nrows)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "allocation failed for row section pointer array");

            
            H5MM_memcpy(&peer_sect->u.indirect.dir_rows[0], &sect->u.indirect.dir_rows[0],
                        (sizeof(H5HF_free_section_t *) * peer_dir_nrows));
            memmove(&sect->u.indirect.dir_rows[0], &sect->u.indirect.dir_rows[peer_dir_nrows],
                    (sizeof(H5HF_free_section_t *) * (sect->u.indirect.dir_nrows - peer_dir_nrows)));
            sect->u.indirect.dir_nrows -= peer_dir_nrows;
            assert(row_sect == sect->u.indirect.dir_rows[0]);

            
            for (u = 0; u < peer_dir_nrows; u++)
                peer_sect->u.indirect.dir_rows[u]->u.row.under = peer_sect;

            
            
            row_sect->sect_info.type = H5HF_FSPACE_SECT_FIRST_ROW;

            
            peer_sect->u.indirect.rc = peer_dir_nrows;
            sect->u.indirect.rc -= peer_dir_nrows;

            
            peer_sect->u.indirect.iblock_entries = sect->u.indirect.iblock_entries;
            peer_sect->u.indirect.span_size      = row_sect->sect_info.addr - peer_sect->sect_info.addr;

            
            sect->sect_info.addr = row_sect->sect_info.addr + hdr->man_dtable.row_block_size[new_start_row];
            sect->u.indirect.span_size -=
                peer_sect->u.indirect.span_size; 
            sect->u.indirect.row = new_start_row;
            sect->u.indirect.col = row_sect->u.row.col + 1;
            sect->u.indirect.num_entries -=
                (peer_nentries + 1); 

            
            assert(sect->u.indirect.rc == (sect->u.indirect.indir_nents + sect->u.indirect.dir_nrows));
            assert(peer_sect->u.indirect.rc ==
                   (peer_sect->u.indirect.indir_nents + peer_sect->u.indirect.dir_nrows));

            
            peer_sect = NULL;
        } 
    }     
    else {
        
        sect->u.indirect.num_entries--;
        sect->u.indirect.dir_nrows--;
        assert(sect->u.indirect.dir_nrows == 0);

        
        sect->u.indirect.dir_rows = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.dir_rows);
    } 

done:
    
    if (peer_sect) {
        
        assert(ret_value < 0);

        if (H5HF__sect_indirect_free(peer_sect) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, unsigned child_entry)
{
    unsigned             start_entry;         
    unsigned             start_row;           
    unsigned             start_col;           
    unsigned             end_entry;           
    unsigned             end_row;             
    H5HF_free_section_t *peer_sect = NULL;    
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(sect->u.indirect.span_size > 0);
    assert(sect->u.indirect.iblock_entries > 0);

    
    start_row   = sect->u.indirect.row;
    start_col   = sect->u.indirect.col;
    start_entry = (start_row * hdr->man_dtable.cparam.width) + start_col;
    end_entry   = (start_entry + sect->u.indirect.num_entries) - 1;
    end_row     = end_entry / hdr->man_dtable.cparam.width;

    
    if (sect->u.indirect.num_entries > 1) {
        
        if (sect->u.indirect.parent) {
            bool is_first; 

            
            is_first = H5HF__sect_indirect_is_first(sect);

            
            if (H5HF__sect_indirect_reduce(hdr, sect->u.indirect.parent, sect->u.indirect.par_entry) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't reduce parent indirect section");
            sect->u.indirect.parent    = NULL;
            sect->u.indirect.par_entry = 0;

            
            if (!is_first)
                if (H5HF__sect_indirect_first(hdr, sect) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                                "can't make new 'first row' for indirect section");
        } 

        
        if (child_entry == start_entry) {
            
            assert(sect->u.indirect.dir_nrows == 0);
            assert(sect->u.indirect.dir_rows == NULL);
            assert(sect->u.indirect.indir_nents > 0);
            assert(sect->u.indirect.indir_ents);

            
            sect->sect_info.addr += hdr->man_dtable.row_block_size[start_row];

            
            sect->u.indirect.col++;
            if (sect->u.indirect.col == hdr->man_dtable.cparam.width) {
                sect->u.indirect.row++;
                sect->u.indirect.col = 0;
            } 
            sect->u.indirect.num_entries--;
            sect->u.indirect.span_size -= hdr->man_dtable.row_block_size[start_row];

            
            sect->u.indirect.indir_nents--;
            memmove(&sect->u.indirect.indir_ents[0], &sect->u.indirect.indir_ents[1],
                    sect->u.indirect.indir_nents * sizeof(H5HF_free_section_t *));
            assert(sect->u.indirect.indir_ents[0]);

            
            if (H5HF__sect_indirect_first(hdr, sect->u.indirect.indir_ents[0]) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                            "can't make new 'first row' for child indirect section");
        } 
        else if (child_entry == end_entry) {
            
            assert(sect->u.indirect.indir_nents > 0);
            assert(sect->u.indirect.indir_ents);

            
            sect->u.indirect.num_entries--;
            sect->u.indirect.span_size -= hdr->man_dtable.row_block_size[end_row];

            
            sect->u.indirect.indir_nents--;
            if (sect->u.indirect.indir_nents == 0)
                sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.indir_ents);
        } 
        else {
            H5HF_indirect_t *iblock;         
            hsize_t          iblock_off;     
            haddr_t          peer_sect_addr; 
            unsigned         peer_nentries;  
            unsigned         peer_start_row; 
            unsigned         peer_start_col; 
            unsigned         child_row;      
            unsigned         new_nentries;   
            unsigned         u;              

            
            assert(sect->u.indirect.indir_nents > 0);
            assert(sect->u.indirect.indir_ents);

            
            peer_nentries  = end_entry - child_entry;
            peer_start_row = (child_entry + 1) / hdr->man_dtable.cparam.width;
            peer_start_col = (child_entry + 1) % hdr->man_dtable.cparam.width;
            child_row      = child_entry / hdr->man_dtable.cparam.width;
            new_nentries   = sect->u.indirect.num_entries - (peer_nentries + 1);
            assert(child_row >= hdr->man_dtable.max_direct_rows);

            
            if (sect->sect_info.state == H5FS_SECT_LIVE) {
                iblock     = sect->u.indirect.u.iblock;
                iblock_off = sect->u.indirect.u.iblock->block_off;
            } 
            else {
                iblock     = NULL;
                iblock_off = sect->u.indirect.u.iblock_off;
            } 

            
            
            sect->u.indirect.num_entries = new_nentries;
            sect->u.indirect.span_size   = H5HF__dtable_span_size(&hdr->man_dtable, sect->u.indirect.row,
                                                                  sect->u.indirect.col, new_nentries);
            assert(sect->u.indirect.span_size > 0);

            
            peer_sect_addr = sect->sect_info.addr;
            peer_sect_addr += sect->u.indirect.span_size;
            peer_sect_addr += hdr->man_dtable.row_block_size[child_row];

            
            if (NULL == (peer_sect = H5HF__sect_indirect_new(hdr, peer_sect_addr, sect->sect_info.size,
                                                             iblock, iblock_off, peer_start_row,
                                                             peer_start_col, peer_nentries)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create indirect section");

            
            peer_sect->u.indirect.dir_nrows   = 0;
            peer_sect->u.indirect.dir_rows    = NULL;
            peer_sect->u.indirect.indir_nents = peer_nentries;
            if (NULL == (peer_sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_malloc(
                             sizeof(H5HF_free_section_t *) * peer_nentries)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL,
                            "allocation failed for indirect section pointer array");

            
            H5MM_memcpy(&peer_sect->u.indirect.indir_ents[0],
                        &sect->u.indirect.indir_ents[sect->u.indirect.indir_nents - peer_nentries],
                        (sizeof(H5HF_free_section_t *) * peer_nentries));
            sect->u.indirect.indir_nents -= (peer_nentries + 1); 

            
            if (sect->u.indirect.indir_nents == 0)
                sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.indir_ents);

            
            for (u = 0; u < peer_nentries; u++)
                peer_sect->u.indirect.indir_ents[u]->u.indirect.parent = peer_sect;

            
            peer_sect->u.indirect.rc = peer_nentries;
            sect->u.indirect.rc -= peer_nentries;

            
            peer_sect->u.indirect.iblock_entries = sect->u.indirect.iblock_entries;

            
            
            assert((sect->u.indirect.rc - 1) == (sect->u.indirect.indir_nents + sect->u.indirect.dir_nrows));
            assert(peer_sect->u.indirect.rc ==
                   (peer_sect->u.indirect.indir_nents + peer_sect->u.indirect.dir_nrows));

            
            if (H5HF__sect_indirect_first(hdr, peer_sect->u.indirect.indir_ents[0]) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                            "can't make new 'first row' for peer indirect section");

            
            peer_sect = NULL;
        } 
    }     
    else {
        
        sect->u.indirect.num_entries--;
        sect->u.indirect.indir_nents--;
        assert(sect->u.indirect.indir_nents == 0);

        
        sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.indir_ents);
    } 

    
    
    if (H5HF__sect_indirect_decr(sect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't decrement section's ref. count ");

done:
    
    if (peer_sect) {
        
        assert(ret_value < 0);

        if (H5HF__sect_indirect_free(peer_sect) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5HF__sect_indirect_is_first(H5HF_free_section_t *sect)
{
    bool ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    
    if (sect->u.indirect.parent) {
        if (sect->sect_info.addr == sect->u.indirect.parent->sect_info.addr)
            ret_value = H5HF__sect_indirect_is_first(sect->u.indirect.parent);
    } 
    else
        ret_value = true;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_first(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);

    
    if (sect->u.indirect.dir_nrows > 0) {
        
        assert(sect->u.indirect.row == 0);
        assert(sect->u.indirect.col == 0);
        assert(sect->u.indirect.dir_rows);
        assert(sect->u.indirect.dir_rows[0]);

        
        if (H5HF__sect_row_first(hdr, sect->u.indirect.dir_rows[0]) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL, "can't set row section to be first row");
    } 
    else {
        
        assert(sect->u.indirect.indir_nents > 0);
        assert(sect->u.indirect.indir_ents);
        assert(sect->u.indirect.indir_ents[0]);

        
        if (H5HF__sect_indirect_first(hdr, sect->u.indirect.indir_ents[0]) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL, "can't set child indirect section to be first row");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5HF_indirect_t *
H5HF__sect_indirect_get_iblock(H5HF_free_section_t *sect)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_INDIRECT);
    assert(sect->sect_info.state == H5FS_SECT_LIVE);

    FUNC_LEAVE_NOAPI(sect->u.indirect.u.iblock)
} 

static herr_t
H5HF__sect_indirect_merge_row(H5HF_hdr_t *hdr, H5HF_free_section_t *row_sect1, H5HF_free_section_t *row_sect2)
{
    H5HF_free_section_t *sect1 = NULL, *sect2 = NULL; 
    unsigned             start_entry1;                
    unsigned             start_row1, start_col1;      
    unsigned             end_entry1;                  
    unsigned             end_row1;                    
    unsigned             start_row2;                  
    bool                 merged_rows;                 
    unsigned             u;                           
    herr_t               ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(row_sect1);
    assert(row_sect1->u.row.under);
    assert(row_sect2);
    assert(row_sect2->u.row.under);
    assert(row_sect2->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW);

    
    if (NULL == (sect1 = H5HF__sect_indirect_top(row_sect1->u.row.under)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve pointer to sections");
    if (NULL == (sect2 = H5HF__sect_indirect_top(row_sect2->u.row.under)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve pointer to sections");

    
    assert(sect1->u.indirect.span_size > 0);
    assert(sect2->u.indirect.span_size > 0);

    
    start_row1   = sect1->u.indirect.row;
    start_col1   = sect1->u.indirect.col;
    start_entry1 = (start_row1 * hdr->man_dtable.cparam.width) + start_col1;
    end_entry1   = (start_entry1 + sect1->u.indirect.num_entries) - 1;
    end_row1     = end_entry1 / hdr->man_dtable.cparam.width;
    start_row2   = sect2->u.indirect.row;

    
    
    if (sect2->u.indirect.dir_nrows > 0) {
        hsize_t  sect1_iblock_off, sect2_iblock_off; 
        unsigned new_dir_nrows1; 
        unsigned src_row2;       
        unsigned nrows_moved2;   

        
        
        assert(sect1->u.indirect.dir_nrows > 0);
        assert(sect1->u.indirect.dir_rows);

        
        if (H5FS_SECT_LIVE == row_sect1->u.row.under->sect_info.state)
            sect1_iblock_off = row_sect1->u.row.under->u.indirect.u.iblock->block_off;
        else
            sect1_iblock_off = row_sect1->u.row.under->u.indirect.u.iblock_off;
        if (H5FS_SECT_LIVE == row_sect2->u.row.under->sect_info.state)
            sect2_iblock_off = row_sect2->u.row.under->u.indirect.u.iblock->block_off;
        else
            sect2_iblock_off = row_sect2->u.row.under->u.indirect.u.iblock_off;

        
        if (sect1_iblock_off == sect2_iblock_off && end_row1 == start_row2) {
            H5HF_free_section_t *last_row_sect1; 

            
            if (row_sect1->u.row.row != end_row1)
                last_row_sect1 = sect1->u.indirect.dir_rows[sect1->u.indirect.dir_nrows - 1];
            else
                last_row_sect1 = row_sect1;
            assert(last_row_sect1);
            assert(last_row_sect1->u.row.row == end_row1);

            
            assert((last_row_sect1->u.row.col + last_row_sect1->u.row.num_entries) == row_sect2->u.row.col);
            last_row_sect1->u.row.num_entries += row_sect2->u.row.num_entries;

            
            src_row2       = 1;
            nrows_moved2   = sect2->u.indirect.dir_nrows - 1;
            new_dir_nrows1 = (sect1->u.indirect.dir_nrows + sect2->u.indirect.dir_nrows) - 1;

            
            merged_rows = true;
        } 
        else {

            
            src_row2       = 0;
            nrows_moved2   = sect2->u.indirect.dir_nrows;
            new_dir_nrows1 = sect1->u.indirect.dir_nrows + sect2->u.indirect.dir_nrows;

            
            merged_rows = false;
        } 

        
        if (nrows_moved2 > 0) {
            H5HF_free_section_t **new_dir_rows; 

            
            if (NULL == (new_dir_rows = (H5HF_free_section_t **)H5MM_realloc(
                             sect1->u.indirect.dir_rows, sizeof(H5HF_free_section_t *) * new_dir_nrows1)))
                HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "allocation failed for row section pointer array");
            sect1->u.indirect.dir_rows = new_dir_rows;

            
            H5MM_memcpy(&sect1->u.indirect.dir_rows[sect1->u.indirect.dir_nrows],
                        &sect2->u.indirect.dir_rows[src_row2],
                        (sizeof(H5HF_free_section_t *) * nrows_moved2));

            
            for (u = sect1->u.indirect.dir_nrows; u < new_dir_nrows1; u++)
                sect1->u.indirect.dir_rows[u]->u.row.under = sect1;

            
            sect1->u.indirect.rc += nrows_moved2;
            sect2->u.indirect.rc -= nrows_moved2;

            
            sect1->u.indirect.dir_nrows = new_dir_nrows1;
        } 
    }     
    else
        
        merged_rows = false;

    
    if (sect2->u.indirect.indir_nents > 0) {
        unsigned new_indir_nents1; 

        
        assert(sect2->u.indirect.rc > 0);
        assert(sect2->u.indirect.indir_nents > 0);
        assert(sect2->u.indirect.indir_ents);

        
        new_indir_nents1 = sect1->u.indirect.indir_nents + sect2->u.indirect.indir_nents;

        
        if (sect1->u.indirect.indir_ents == NULL) {
            sect1->u.indirect.indir_ents = sect2->u.indirect.indir_ents;
            sect2->u.indirect.indir_ents = NULL;
        } 
        else {
            H5HF_free_section_t **new_indir_ents; 

            
            if (NULL == (new_indir_ents = (H5HF_free_section_t **)H5MM_realloc(
                             sect1->u.indirect.indir_ents, sizeof(H5HF_free_section_t *) * new_indir_nents1)))
                HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "allocation failed for row section pointer array");
            sect1->u.indirect.indir_ents = new_indir_ents;

            
            H5MM_memcpy(&sect1->u.indirect.indir_ents[sect1->u.indirect.indir_nents],
                        &sect2->u.indirect.indir_ents[0],
                        (sizeof(H5HF_free_section_t *) * sect2->u.indirect.indir_nents));
        } 

        
        for (u = sect1->u.indirect.indir_nents; u < new_indir_nents1; u++)
            sect1->u.indirect.indir_ents[u]->u.indirect.parent = sect1;

        
        sect1->u.indirect.rc += sect2->u.indirect.indir_nents;
        sect2->u.indirect.rc -= sect2->u.indirect.indir_nents;

        
        sect1->u.indirect.indir_nents = new_indir_nents1;
    } 

    
    sect1->u.indirect.num_entries += sect2->u.indirect.num_entries;
    sect1->u.indirect.span_size += sect2->u.indirect.span_size;

    
    assert(sect1->u.indirect.rc == (sect1->u.indirect.indir_nents + sect1->u.indirect.dir_nrows));

    
    
    if (merged_rows) {
        
        
        assert(sect2->u.indirect.rc == 1);
        if (H5HF__sect_row_free((H5FS_section_info_t *)row_sect2) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free row section");
    } 
    else {
        
        assert(sect2->u.indirect.rc == 0);
        if (sect2->u.indirect.parent)
            if (H5HF__sect_indirect_decr(sect2->u.indirect.parent) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL,
                            "can't decrement ref. count on parent indirect section");

        
        if (H5HF__sect_indirect_free(sect2) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");

        
        
        row_sect2->sect_info.type = H5HF_FSPACE_SECT_NORMAL_ROW;
        if (H5HF__space_add(hdr, row_sect2, H5FS_ADD_SKIP_VALID) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't re-add second row section to free space");
    } 

    
    
    if (sect1->u.indirect.iblock_entries == sect1->u.indirect.num_entries) {
        
        assert(sect1->u.indirect.parent == NULL);
        if (H5HF__sect_indirect_build_parent(hdr, sect1) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL, "can't create parent for full indirect section");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_build_parent(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    H5HF_indirect_t     *par_iblock;          
    H5HF_free_section_t *par_sect = NULL;     
    hsize_t              par_block_off;       
    unsigned             par_row, par_col;    
    unsigned             par_entry;           
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(H5FS_SECT_LIVE == sect->sect_info.state);
    assert(sect->u.indirect.span_size > 0);
    assert(sect->u.indirect.iblock_entries > 0);
    assert(sect->u.indirect.iblock_entries == sect->u.indirect.num_entries);
    assert(sect->u.indirect.u.iblock);
    assert(sect->u.indirect.parent == NULL);

    
    if (sect->u.indirect.u.iblock->parent) {
        par_entry     = sect->u.indirect.u.iblock->par_entry;
        par_iblock    = sect->u.indirect.u.iblock->parent;
        par_block_off = par_iblock->block_off;
    } 
    else {
        
        if (H5HF__man_iblock_parent_info(hdr, sect->sect_info.addr, &par_block_off, &par_entry) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't get block entry");
        par_iblock = NULL;
    } 

    
    par_row = par_entry / hdr->man_dtable.cparam.width;
    par_col = par_entry % hdr->man_dtable.cparam.width;
    assert(par_row >= hdr->man_dtable.max_direct_rows);

    
    if (NULL == (par_sect = H5HF__sect_indirect_new(hdr, sect->sect_info.addr, sect->sect_info.size,
                                                    par_iblock, par_block_off, par_row, par_col, 1)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create indirect section");

    
    par_sect->u.indirect.dir_nrows = 0;
    par_sect->u.indirect.dir_rows  = NULL;

    
    par_sect->u.indirect.indir_nents = 1;
    if (NULL == (par_sect->u.indirect.indir_ents =
                     (H5HF_free_section_t **)H5MM_malloc(sizeof(H5HF_free_section_t *))))
        HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "allocation failed for indirect section pointer array");

    
    sect->u.indirect.parent            = par_sect;
    sect->u.indirect.par_entry         = par_entry;
    par_sect->u.indirect.indir_ents[0] = sect;
    par_sect->u.indirect.rc            = 1;

done:
    if (ret_value < 0)
        if (par_sect && H5HF__sect_indirect_free(par_sect) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_shrink(H5HF_hdr_t *hdr, H5HF_free_section_t *sect)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);

    
    assert(sect->u.indirect.dir_nrows > 0 || sect->u.indirect.indir_nents > 0);

    
    for (u = 0; u < sect->u.indirect.dir_nrows; u++) {
        
        if (sect->u.indirect.dir_rows[u]->sect_info.type != H5HF_FSPACE_SECT_FIRST_ROW) {
            assert(sect->u.indirect.dir_rows[u]->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);
            if (H5HF__space_remove(hdr, sect->u.indirect.dir_rows[u]) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove section from heap free space");
        } 

        
        if (H5HF__sect_row_free_real(sect->u.indirect.dir_rows[u]) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free child section node");
    } 

    
    for (u = 0; u < sect->u.indirect.indir_nents; u++)
        if (H5HF__sect_indirect_shrink(hdr, sect->u.indirect.indir_ents[u]) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free child section node");

    
    if (H5HF__sect_indirect_free(sect) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free indirect section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_serialize(H5HF_hdr_t *hdr, const H5HF_free_section_t *sect, uint8_t *buf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(sect);
    assert(buf);

    
    if (sect->u.indirect.parent) {
        if (sect->sect_info.addr == sect->u.indirect.parent->sect_info.addr)
            if (H5HF__sect_indirect_serialize(hdr, sect->u.indirect.parent, buf) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTSERIALIZE, FAIL,
                            "can't serialize indirect section's parent indirect section");
    } 
    else {
        
        if (sect->sect_info.state == H5FS_SECT_LIVE) {
            assert(sect->u.indirect.u.iblock);
            UINT64ENCODE_VAR(buf, sect->u.indirect.u.iblock->block_off, hdr->heap_off_size);
        } 
        else
            UINT64ENCODE_VAR(buf, sect->u.indirect.u.iblock_off, hdr->heap_off_size);

        
        UINT16ENCODE(buf, sect->u.indirect.row);

        
        UINT16ENCODE(buf, sect->u.indirect.col);

        
        UINT16ENCODE(buf, sect->u.indirect.num_entries);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5FS_section_info_t *
H5HF__sect_indirect_deserialize(H5HF_hdr_t *hdr, const uint8_t *buf, haddr_t sect_addr, hsize_t sect_size,
                                unsigned *des_flags)
{
    H5HF_free_section_t *new_sect;         
    hsize_t              iblock_off;       
    unsigned             start_row;        
    unsigned             start_col;        
    unsigned             nentries;         
    unsigned             start_entry;      
    unsigned             end_entry;        
    unsigned             end_row;          
    unsigned             end_col;          
    H5FS_section_info_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(buf);
    assert(H5_addr_defined(sect_addr));
    assert(sect_size);

    
    UINT64DECODE_VAR(buf, iblock_off, hdr->heap_off_size);

    
    UINT16DECODE(buf, start_row);

    
    UINT16DECODE(buf, start_col);

    
    UINT16DECODE(buf, nentries);

    
    if (NULL == (new_sect = H5HF__sect_indirect_new(hdr, sect_addr, sect_size, NULL, iblock_off, start_row,
                                                    start_col, nentries)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't create indirect section");

    
    start_entry = (start_row * hdr->man_dtable.cparam.width) + start_col;

    
    end_entry = (start_entry + nentries) - 1;
    end_row   = end_entry / hdr->man_dtable.cparam.width;
    end_col   = end_entry % hdr->man_dtable.cparam.width;

    
    if (H5HF__sect_indirect_init_rows(hdr, new_sect, true, NULL, H5FS_ADD_DESERIALIZING,
                                      new_sect->u.indirect.row, new_sect->u.indirect.col, end_row,
                                      end_col) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, NULL, "can't initialize indirect section");

    
    *des_flags |= H5FS_DESERIALIZE_NO_ADD;

    
    ret_value = (H5FS_section_info_t *)new_sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_free(H5HF_free_section_t *sect)
{
    H5HF_indirect_t *iblock    = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(sect);

    
    sect->u.indirect.dir_rows = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.dir_rows);

    
    sect->u.indirect.indir_ents = (H5HF_free_section_t **)H5MM_xfree(sect->u.indirect.indir_ents);

    
    if (sect->sect_info.state == H5FS_SECT_LIVE)
        
        if (sect->u.indirect.u.iblock)
            iblock = sect->u.indirect.u.iblock;

    
    if (H5HF__sect_node_free(sect, iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__sect_indirect_valid(const H5HF_hdr_t *hdr, const H5HF_free_section_t *sect)
{
    unsigned start_row;   
    unsigned start_col;   
    unsigned start_entry; 
    unsigned end_row;     
    unsigned end_entry;   
    unsigned u;           

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(sect);

    
    start_row   = sect->u.indirect.row;
    start_col   = sect->u.indirect.col;
    start_entry = (start_row * hdr->man_dtable.cparam.width) + start_col;

    
    end_entry = (start_entry + sect->u.indirect.num_entries) - 1;
    end_row   = end_entry / hdr->man_dtable.cparam.width;

    
    if (sect->u.indirect.dir_nrows > 0) {
        unsigned dir_nrows;   
        unsigned max_dir_row; 

        
        if (end_row >= hdr->man_dtable.max_direct_rows)
            max_dir_row = hdr->man_dtable.max_direct_rows - 1;
        else
            max_dir_row = end_row;

        
        dir_nrows = (max_dir_row - start_row) + 1;
        assert(dir_nrows == sect->u.indirect.dir_nrows);
        for (u = 0; u < dir_nrows; u++) {
            const H5HF_free_section_t H5_ATTR_NDEBUG_UNUSED *tmp_row_sect; 

            tmp_row_sect = sect->u.indirect.dir_rows[u];
            assert(tmp_row_sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW ||
                   tmp_row_sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW);
            assert(tmp_row_sect->u.row.under == sect);
            assert(tmp_row_sect->u.row.row == (start_row + u));
            if (u > 0) {
                const H5HF_free_section_t H5_ATTR_NDEBUG_UNUSED *tmp_row_sect2; 

                tmp_row_sect2 = sect->u.indirect.dir_rows[u - 1];
                assert(tmp_row_sect2->u.row.row < tmp_row_sect->u.row.row);
                assert(H5_addr_lt(tmp_row_sect2->sect_info.addr, tmp_row_sect->sect_info.addr));
                assert(tmp_row_sect2->sect_info.size <= tmp_row_sect->sect_info.size);
            } 
        }     
    }         

    
    if (sect->u.indirect.indir_nents > 0) {
        
        if (sect->sect_info.state == H5FS_SECT_LIVE) {
            assert(sect->u.indirect.iblock_entries);
            assert(sect->u.indirect.indir_nents <= sect->u.indirect.iblock_entries);
        } 
        assert(sect->u.indirect.indir_ents);

        
        for (u = 0; u < sect->u.indirect.indir_nents; u++) {
            const H5HF_free_section_t *tmp_child_sect; 

            tmp_child_sect = sect->u.indirect.indir_ents[u];
            assert(tmp_child_sect->sect_info.type == H5HF_FSPACE_SECT_INDIRECT);
            assert(tmp_child_sect->u.indirect.parent == sect);
            if (u > 0) {
                const H5HF_free_section_t H5_ATTR_NDEBUG_UNUSED
                    *tmp_child_sect2; 

                tmp_child_sect2 = sect->u.indirect.indir_ents[u - 1];
                assert(H5_addr_lt(tmp_child_sect2->sect_info.addr, tmp_child_sect->sect_info.addr));
            } 

            
            H5HF__sect_indirect_valid(hdr, tmp_child_sect);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__sect_indirect_debug(const H5HF_free_section_t *sect, FILE *stream, int indent, int fwidth)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Row:", sect->u.indirect.row);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Column:", sect->u.indirect.col);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Number of entries:", sect->u.indirect.num_entries);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
