/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"  
#include "H5Aprivate.h" 
#include "H5Dprivate.h" 
#include "H5Eprivate.h" 
#include "H5Gpkg.h"     
#include "H5Iprivate.h" 
#include "H5Lprivate.h" 

typedef struct {
    
    H5G_loc_t *loc; 
} H5G_loc_fnd_t;

typedef struct {
    
    H5_index_t      idx_type; 
    H5_iter_order_t order;    
    hsize_t         n;        

    
    H5G_loc_t *loc; 
} H5G_loc_fbi_t;

typedef struct {
    
    unsigned fields; 

    
    H5O_info2_t *oinfo; 
} H5G_loc_info_t;

typedef struct {
    
    unsigned fields; 

    
    H5O_native_info_t *oinfo; 
} H5G_loc_native_info_t;

typedef struct {
    
    const char *comment; 

    
} H5G_loc_sc_t;

typedef struct {
    
    char  *comment; 
    size_t bufsize; 

    
    size_t comment_size; 
} H5G_loc_gc_t;

static herr_t H5G__loc_find_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                               H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_find_by_idx_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                                      H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_addr_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                               H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_info_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                               H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_native_info_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                                      H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_set_comment_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                                      H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);
static herr_t H5G__loc_get_comment_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                                      H5G_loc_t *obj_loc, void *_udata, H5G_own_loc_t *own_loc);

herr_t
H5G_loc_real(void *obj, H5I_type_t type, H5G_loc_t *loc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    switch (type) {
        case H5I_FILE: {
            H5F_t *f = (H5F_t *)obj;

            
            if (H5G_root_loc(f, loc) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unable to create location for file");
            break;
        }

        case H5I_GROUP: {
            H5G_t *group = (H5G_t *)obj;

            if (NULL == (loc->oloc = H5G_oloc(group)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location of group");
            if (NULL == (loc->path = H5G_nameof(group)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path of group");
            break;
        }

        case H5I_DATATYPE: {
            H5T_t *dt = NULL;

            
            dt = H5T_get_actual_type((H5T_t *)obj);

            if (NULL == (loc->oloc = H5T_oloc(dt)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location of datatype");
            if (NULL == (loc->path = H5T_nameof(dt)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path of datatype");
            break;
        }

        case H5I_DATASET: {
            H5D_t *dset = (H5D_t *)obj;

            if (NULL == (loc->oloc = H5D_oloc(dset)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location of dataset");
            if (NULL == (loc->path = H5D_nameof(dset)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path of dataset");
            break;
        }

        case H5I_ATTR: {
            H5A_t *attr = (H5A_t *)obj;

            if (NULL == (loc->oloc = H5A_oloc(attr)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location of attribute");
            if (NULL == (loc->path = H5A_nameof(attr)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path of attribute");
            break;
        }

        case H5I_DATASPACE:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get group location of dataspace");

        case H5I_MAP:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "maps not supported in native VOL connector");

        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get group location of property list");

        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "unable to get group location of error class, message or stack");

        case H5I_VFL:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "unable to get group location of a virtual file driver (VFD)");

        case H5I_VOL:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "unable to get group location of a virtual object layer (VOL) connector");

        case H5I_SPACE_SEL_ITER:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "unable to get group location of a dataspace selection iterator");

        case H5I_EVENTSET:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get group location of a event set");

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid location type");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc(hid_t loc_id, H5G_loc_t *loc)
{
    void  *obj       = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (obj = H5VL_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5G_loc_real(obj, H5I_get_type(loc_id), loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unable to fill in location struct");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_copy(H5G_loc_t *dst, const H5G_loc_t *src, H5_copy_depth_t depth)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dst);
    assert(src);

    
    if (H5O_loc_copy(dst->oloc, src->oloc, depth) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to copy entry");
    if (H5G_name_copy(dst->path, src->path, depth) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to copy path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_reset(H5G_loc_t *loc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);

    
    if (H5O_loc_reset(loc->oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to reset entry");
    if (H5G_name_reset(loc->path) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to reset path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_free(H5G_loc_t *loc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);

    
    if (H5G_name_free(loc->path) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to free path");
    if (H5O_loc_free(loc->oloc) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "unable to free object header location");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_find_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char *name,
                 const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                 H5G_own_loc_t *own_loc )
{
    H5G_loc_fnd_t *udata     = (H5G_loc_fnd_t *)_udata; 
    herr_t         ret_value = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "object '%s' doesn't exist", name);

    
    
    H5G_loc_copy(udata->loc, obj_loc, H5_COPY_SHALLOW);
    *own_loc = H5G_OWN_OBJ_LOC;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_find(const H5G_loc_t *loc, const char *name, H5G_loc_t *obj_loc )
{
    H5G_loc_fnd_t udata;               
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name);
    assert(obj_loc);

    if (!*name)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "invalid object name");

    
    udata.loc = obj_loc;

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_find_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_find_by_idx_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                        const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                        H5G_own_loc_t *own_loc )
{
    H5G_loc_fbi_t *udata = (H5G_loc_fbi_t *)_udata; 
    H5O_link_t     fnd_lnk;                         
    bool           lnk_copied    = false;           
    bool           obj_loc_valid = false;           
    bool           obj_exists    = false;           
    herr_t         ret_value     = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "group doesn't exist");

    
    if (H5G_obj_lookup_by_idx(obj_loc->oloc, udata->idx_type, udata->order, udata->n, &fnd_lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "link not found");
    lnk_copied = true;

    
    if (H5G__link_to_loc(obj_loc, &fnd_lnk, udata->loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "cannot initialize object location");
    obj_loc_valid = true;

    
    
    
    if (H5G__traverse_special(obj_loc, &fnd_lnk, H5G_TARGET_NORMAL, true, udata->loc, &obj_exists) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_TRAVERSE, FAIL, "special link traversal failed");

done:
    
    if (lnk_copied)
        H5O_msg_reset(H5O_LINK_ID, &fnd_lnk);

    
    if (ret_value < 0 && obj_loc_valid)
        if (H5G_loc_free(udata->loc) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't free location");

    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_find_by_idx(const H5G_loc_t *loc, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                    hsize_t n, H5G_loc_t *obj_loc )
{
    H5G_loc_fbi_t udata;               
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(group_name && *group_name);
    assert(obj_loc);

    
    udata.idx_type = idx_type;
    udata.order    = order;
    udata.n        = n;
    udata.loc      = obj_loc;

    
    if (H5G_traverse(loc, group_name, H5G_TARGET_NORMAL, H5G__loc_find_by_idx_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__loc_insert(H5G_loc_t *grp_loc, char *name, H5G_loc_t *obj_loc, H5O_type_t obj_type, const void *crt_info)
{
    H5O_link_t lnk;                 
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_loc);
    assert(name && *name);
    assert(obj_loc);

    
    lnk.type         = H5L_TYPE_HARD;
    lnk.cset         = H5F_DEFAULT_CSET;
    lnk.corder       = 0;     
    lnk.corder_valid = false; 
    lnk.name         = name;
    lnk.u.hard.addr  = obj_loc->oloc->addr;

    
    if (H5G_obj_insert(grp_loc->oloc, &lnk, true, obj_type, crt_info) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert object");

    
    if (H5G_name_set(grp_loc->path, obj_loc->path, name) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "cannot set name");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_exists_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                   const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                   H5G_own_loc_t *own_loc )
{
    bool *exists = (bool *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    if (obj_loc == NULL)
        *exists = false;
    else
        *exists = true;

    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_loc_exists(const H5G_loc_t *loc, const char *name, bool *exists)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name && *name);
    assert(exists);

    
    if (H5G_traverse(loc, name, H5G_TARGET_EXISTS, H5G__loc_exists_cb, exists) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't check if object exists");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_addr_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                 const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                 H5G_own_loc_t *own_loc )
{
    haddr_t *udata = (haddr_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    if (obj_loc == NULL)
        *udata = HADDR_UNDEF; 
    else
        *udata = obj_loc->oloc->addr; 

    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G__loc_addr(const H5G_loc_t *loc, const char *name, haddr_t *addr )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name && *name);
    assert(addr);

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_addr_cb, addr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_info_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                 const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                 H5G_own_loc_t *own_loc )
{
    H5G_loc_info_t *udata     = (H5G_loc_info_t *)_udata; 
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "name doesn't exist");

    
    if (H5O_get_info(obj_loc->oloc, udata->oinfo, udata->fields) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get object info");

done:
    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_info(const H5G_loc_t *loc, const char *name, H5O_info2_t *oinfo , unsigned fields)
{
    H5G_loc_info_t udata;               
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name && *name);
    assert(oinfo);

    
    udata.fields = fields;
    udata.oinfo  = oinfo;

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_info_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_native_info_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                        const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                        H5G_own_loc_t *own_loc )
{
    H5G_loc_native_info_t *udata     = (H5G_loc_native_info_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "name doesn't exist");

    
    if (H5O_get_native_info(obj_loc->oloc, udata->oinfo, udata->fields) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get object info");

done:
    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_native_info(const H5G_loc_t *loc, const char *name, H5O_native_info_t *oinfo , unsigned fields)
{
    H5G_loc_native_info_t udata;               
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name && *name);
    assert(oinfo);

    
    udata.fields = fields;
    udata.oinfo  = oinfo;

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_native_info_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_set_comment_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                        const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                        H5G_own_loc_t *own_loc )
{
    H5G_loc_sc_t *udata   = (H5G_loc_sc_t *)_udata; 
    H5O_name_t    comment = {0};                    
    htri_t        exists;                           
    herr_t        ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "name doesn't exist");

    
    if ((exists = H5O_msg_exists(obj_loc->oloc, H5O_NAME_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to read object header");

    
    if (exists)
        if (H5O_msg_remove(obj_loc->oloc, H5O_NAME_ID, 0, true) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL,
                        "unable to delete existing comment object header message");

    
    if (udata->comment && *udata->comment) {
        if (NULL == (comment.s = strdup(udata->comment)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't copy group comment");
        if (H5O_msg_create(obj_loc->oloc, H5O_NAME_ID, 0, H5O_UPDATE_TIME, &comment) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to set comment object header message");
    } 

done:
    free(comment.s);

    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_set_comment(const H5G_loc_t *loc, const char *name, const char *comment)
{
    H5G_loc_sc_t udata;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name && *name);

    
    udata.comment = comment;

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_set_comment_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__loc_get_comment_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc , const char H5_ATTR_UNUSED *name,
                        const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                        H5G_own_loc_t *own_loc )
{
    H5G_loc_gc_t *udata = (H5G_loc_gc_t *)_udata; 
    H5O_name_t    comment;                        
    herr_t        ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "name doesn't exist");

    
    comment.s = NULL;
    if (NULL == H5O_msg_read(obj_loc->oloc, H5O_NAME_ID, &comment)) {
        if (udata->comment && udata->bufsize > 0)
            udata->comment[0] = '\0';
        udata->comment_size = 0;
    }
    else {
        if (udata->comment && udata->bufsize)
            strncpy(udata->comment, comment.s, udata->bufsize);
        udata->comment_size = strlen(comment.s);
        H5O_msg_reset(H5O_NAME_ID, &comment);
    }

done:
    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_loc_get_comment(const H5G_loc_t *loc, const char *name, char *comment , size_t bufsize,
                    size_t *comment_len)
{
    H5G_loc_gc_t udata;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);
    assert(name && *name);

    
    udata.comment      = comment;
    udata.bufsize      = bufsize;
    udata.comment_size = 0;

    
    if (H5G_traverse(loc, name, H5G_TARGET_NORMAL, H5G__loc_get_comment_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't find object");

    
    if (comment_len)
        *comment_len = udata.comment_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
