/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5MFprivate.h" 

static herr_t H5D__single_idx_init(const H5D_chk_idx_info_t *idx_info, const H5S_t *space,
                                   haddr_t dset_ohdr_addr);
static herr_t H5D__single_idx_create(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__single_idx_open(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__single_idx_close(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__single_idx_is_open(const H5D_chk_idx_info_t *idx_info, bool *is_open);
static bool   H5D__single_idx_is_space_alloc(const H5O_storage_chunk_t *storage);
static herr_t H5D__single_idx_insert(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata,
                                     const H5D_t *dset);
static herr_t H5D__single_idx_get_addr(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata);
static herr_t H5D__single_idx_load_metadata(const H5D_chk_idx_info_t *idx_info);
static int    H5D__single_idx_iterate(const H5D_chk_idx_info_t *idx_info, H5D_chunk_cb_func_t chunk_cb,
                                      void *chunk_udata);
static herr_t H5D__single_idx_remove(const H5D_chk_idx_info_t *idx_info, H5D_chunk_common_ud_t *udata);
static herr_t H5D__single_idx_delete(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__single_idx_copy_setup(const H5D_chk_idx_info_t *idx_info_src,
                                         const H5D_chk_idx_info_t *idx_info_dst);
static herr_t H5D__single_idx_size(const H5D_chk_idx_info_t *idx_info, hsize_t *size);
static herr_t H5D__single_idx_reset(H5O_storage_chunk_t *storage, bool reset_addr);
static herr_t H5D__single_idx_dump(const H5O_storage_chunk_t *storage, FILE *stream);

const H5D_chunk_ops_t H5D_COPS_SINGLE[1] = {{
    false,                          
    H5D__single_idx_init,           
    H5D__single_idx_create,         
    H5D__single_idx_open,           
    H5D__single_idx_close,          
    H5D__single_idx_is_open,        
    H5D__single_idx_is_space_alloc, 
    H5D__single_idx_insert,         
    H5D__single_idx_get_addr,       
    H5D__single_idx_load_metadata,  
    NULL,                           
    H5D__single_idx_iterate,        
    H5D__single_idx_remove,         
    H5D__single_idx_delete,         
    H5D__single_idx_copy_setup,     
    NULL,                           
    H5D__single_idx_size,           
    H5D__single_idx_reset,          
    H5D__single_idx_dump,           
    NULL                            
}};

static herr_t
H5D__single_idx_init(const H5D_chk_idx_info_t *idx_info, const H5S_t H5_ATTR_UNUSED *space,
                     haddr_t H5_ATTR_UNUSED dset_ohdr_addr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);

    if (idx_info->pline->nused) {
        idx_info->layout->u.chunk.flags |= H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER;

        if (!H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr)) {
            idx_info->layout->storage.u.chunk.u.single.nbytes      = 0;
            idx_info->layout->storage.u.chunk.u.single.filter_mask = 0;
        }
    }
    else
        idx_info->layout->u.chunk.flags = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_create(const H5D_chk_idx_info_t *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(idx_info->layout->u.chunk.max_nchunks == idx_info->layout->u.chunk.nchunks);
    assert(idx_info->layout->u.chunk.nchunks == 1);
    assert(!H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    if (idx_info->pline->nused)
        assert(idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER);
    else
        assert(!(idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_open(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_close(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_is_open(const H5D_chk_idx_info_t H5_ATTR_NDEBUG_UNUSED *idx_info, bool *is_open)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(idx_info);
    assert(idx_info->layout);
    assert(H5D_CHUNK_IDX_SINGLE == idx_info->layout->storage.u.chunk.idx_type);
    assert(is_open);

    *is_open = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static bool
H5D__single_idx_is_space_alloc(const H5O_storage_chunk_t *storage)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    FUNC_LEAVE_NOAPI((bool)H5_addr_defined(storage->idx_addr))
} 

static herr_t
H5D__single_idx_insert(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata, const H5D_t *dset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(idx_info->layout->u.chunk.nchunks == 1);
    assert(idx_info->layout->u.chunk.max_nchunks == 1);
    assert(udata);

    
    assert(H5_addr_defined(udata->chunk_block.offset));
    idx_info->layout->storage.u.chunk.idx_addr = udata->chunk_block.offset;

    if (idx_info->pline->nused > 0) {
        idx_info->layout->storage.u.chunk.u.single.nbytes      = udata->chunk_block.length;
        idx_info->layout->storage.u.chunk.u.single.filter_mask = udata->filter_mask;
    } 

    if (dset)
        if (dset->shared->dcpl_cache.fill.alloc_time != H5D_ALLOC_TIME_EARLY || idx_info->pline->nused > 0)
            
            if (H5D__mark(dset, H5D_MARK_LAYOUT) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to mark layout as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__single_idx_get_addr(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(idx_info->layout->u.chunk.nchunks == 1);
    assert(idx_info->layout->u.chunk.max_nchunks == 1);
    assert(udata);

    udata->chunk_block.offset = idx_info->layout->storage.u.chunk.idx_addr;
    if (idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER) {
        udata->chunk_block.length = idx_info->layout->storage.u.chunk.u.single.nbytes;
        udata->filter_mask        = idx_info->layout->storage.u.chunk.u.single.filter_mask;
    } 
    else {
        udata->chunk_block.length = idx_info->layout->u.chunk.size;
        udata->filter_mask        = 0;
    } 
    if (!H5_addr_defined(udata->chunk_block.offset))
        udata->chunk_block.length = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_load_metadata(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5D__single_idx_iterate(const H5D_chk_idx_info_t *idx_info, H5D_chunk_cb_func_t chunk_cb, void *chunk_udata)
{
    H5D_chunk_rec_t chunk_rec;      
    int             ret_value = -1; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(chunk_cb);
    assert(chunk_udata);
    assert(H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    
    memset(&chunk_rec, 0, sizeof(chunk_rec));
    chunk_rec.chunk_addr = idx_info->layout->storage.u.chunk.idx_addr;

    if (idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER) {
        chunk_rec.nbytes      = idx_info->layout->storage.u.chunk.u.single.nbytes;
        chunk_rec.filter_mask = idx_info->layout->storage.u.chunk.u.single.filter_mask;
    } 
    else {
        chunk_rec.nbytes      = idx_info->layout->u.chunk.size;
        chunk_rec.filter_mask = 0;
    } 

    
    if ((ret_value = (*chunk_cb)(&chunk_rec, chunk_udata)) < 0)
        HERROR(H5E_DATASET, H5E_CALLBACK, "failure in generic chunk iterator callback");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__single_idx_remove(const H5D_chk_idx_info_t *idx_info, H5D_chunk_common_ud_t H5_ATTR_UNUSED *udata)
{
    hsize_t nbytes;              
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    if (idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER)
        nbytes = idx_info->layout->storage.u.chunk.u.single.nbytes;
    else
        nbytes = idx_info->layout->u.chunk.size;

    if (H5MF_xfree(idx_info->f, H5FD_MEM_DRAW, idx_info->layout->storage.u.chunk.idx_addr, nbytes) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, H5_ITER_ERROR, "unable to free dataset chunks");

    idx_info->layout->storage.u.chunk.idx_addr = HADDR_UNDEF;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__single_idx_delete(const H5D_chk_idx_info_t *idx_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);

    if (H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr))
        ret_value = H5D__single_idx_remove(idx_info, NULL);
    else
        assert(!H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__single_idx_copy_setup(const H5D_chk_idx_info_t H5_ATTR_NDEBUG_UNUSED *idx_info_src,
                           const H5D_chk_idx_info_t                       *idx_info_dst)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info_src);
    assert(idx_info_src->f);
    assert(idx_info_src->pline);
    assert(idx_info_src->layout);
    assert(H5_addr_defined(idx_info_src->layout->storage.u.chunk.idx_addr));

    assert(idx_info_dst);
    assert(idx_info_dst->f);
    assert(idx_info_dst->pline);
    assert(idx_info_dst->layout);

    
    H5_BEGIN_TAG(H5AC__COPIED_TAG)

    
    if (H5D__single_idx_create(idx_info_dst) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize chunked storage");

    
    H5_END_TAG

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__single_idx_size(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info, hsize_t *index_size)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(index_size);

    *index_size = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_reset(H5O_storage_chunk_t *storage, bool reset_addr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    
    if (reset_addr)
        storage->idx_addr = HADDR_UNDEF;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__single_idx_dump(const H5O_storage_chunk_t *storage, FILE *stream)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);
    assert(stream);

    Rfprintf(stream, "    Address: %" PRIuHADDR "\n", storage->idx_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
