/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 
#define H5O_FRIEND     

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Opkg.h"      
#include "H5VLprivate.h" 

static void      *H5O__dset_get_copy_file_udata(void);
static void       H5O__dset_free_copy_file_udata(void *);
static htri_t     H5O__dset_isa(const H5O_t *loc);
static void      *H5O__dset_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type);
static void      *H5O__dset_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc);
static H5O_loc_t *H5O__dset_get_oloc(hid_t obj_id);
static herr_t     H5O__dset_bh_info(const H5O_loc_t *loc, H5O_t *oh, H5_ih_info_t *bh_info);
static herr_t     H5O__dset_flush(void *_obj_ptr);

const H5O_obj_class_t H5O_OBJ_DATASET[1] = {{
    H5O_TYPE_DATASET,               
    "dataset",                      
    H5O__dset_get_copy_file_udata,  
    H5O__dset_free_copy_file_udata, 
    H5O__dset_isa,                  
    H5O__dset_open,                 
    H5O__dset_create,               
    H5O__dset_get_oloc,             
    H5O__dset_bh_info,              
    H5O__dset_flush                 
}};

H5FL_DEFINE(H5D_copy_file_ud_t);

static void *
H5O__dset_get_copy_file_udata(void)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5FL_CALLOC(H5D_copy_file_ud_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5O__dset_free_copy_file_udata(void *_udata)
{
    H5D_copy_file_ud_t *udata = (H5D_copy_file_ud_t *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);

    
    if (udata->src_space_extent)
        H5O_msg_free(H5O_SDSPACE_ID, udata->src_space_extent);

    
    if (udata->src_dtype)
        H5T_close_real(udata->src_dtype);

    
    if (udata->common.src_pline)
        H5O_msg_free(H5O_PLINE_ID, udata->common.src_pline);

    
    udata = H5FL_FREE(H5D_copy_file_ud_t, udata);

    FUNC_LEAVE_NOAPI_VOID
} 

static htri_t
H5O__dset_isa(const H5O_t *oh)
{
    htri_t exists;           
    htri_t ret_value = true; 

    FUNC_ENTER_PACKAGE

    assert(oh);

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_DTYPE_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to read object header");
    else if (!exists)
        HGOTO_DONE(false);

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_SDSPACE_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to read object header");
    else if (!exists)
        HGOTO_DONE(false);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dset_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type)
{
    H5D_t *dset = NULL;      
    hid_t  dapl_id;          
    void  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(obj_loc);

    *opened_type = H5I_DATASET;

    
    dapl_id = H5CX_get_lapl();
    if (dapl_id == H5P_LINK_ACCESS_DEFAULT)
        dapl_id = H5P_DATASET_ACCESS_DEFAULT;
    else {
        htri_t is_lapl, is_dapl; 

        
        if ((is_lapl = H5P_isa_class(dapl_id, H5P_LINK_ACCESS)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "unable to get LAPL status");
        if ((is_dapl = H5P_isa_class(dapl_id, H5P_DATASET_ACCESS)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "unable to get DAPL status");

        
        if (!is_dapl && is_lapl)
            dapl_id = H5P_DATASET_ACCESS_DEFAULT;
    } 

    
    if (NULL == (dset = H5D_open(obj_loc, dapl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, NULL, "unable to open dataset");

    ret_value = (void *)dset;

done:
    if (NULL == ret_value)
        if (dset && H5D_close(dset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release dataset");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dset_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc)
{
    H5D_obj_create_t *crt_info  = (H5D_obj_create_t *)_crt_info; 
    H5D_t            *dset      = NULL;                          
    void             *ret_value = NULL;                          

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(crt_info);
    assert(obj_loc);

    
    if (NULL ==
        (dset = H5D__create(f, crt_info->type_id, crt_info->space, crt_info->dcpl_id, crt_info->dapl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to create dataset");

    
    if (NULL == (obj_loc->oloc = H5D_oloc(dset)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get object location of dataset");
    if (NULL == (obj_loc->path = H5D_nameof(dset)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get path of dataset");

    
    ret_value = dset;

done:
    if (ret_value == NULL)
        if (dset && H5D_close(dset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release dataset");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5O_loc_t *
H5O__dset_get_oloc(hid_t obj_id)
{
    H5D_t     *dset;             
    H5O_loc_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dset = (H5D_t *)H5VL_object(obj_id)))
        HGOTO_ERROR(H5E_OHDR, H5E_BADID, NULL, "couldn't get object from ID");

    
    if (NULL == (ret_value = H5D_oloc(dset)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "unable to get object location from object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dset_bh_info(const H5O_loc_t *loc, H5O_t *oh, H5_ih_info_t *bh_info)
{
    H5O_layout_t layout;              
    H5O_efl_t    efl;                 
    bool         layout_read = false; 
    bool         efl_read    = false; 
    htri_t       exists;              
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(oh);
    assert(bh_info);

    
    if (NULL == H5O_msg_read_oh(loc->file, oh, H5O_LAYOUT_ID, &layout))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't find layout message");
    layout_read = true;

    
    if (layout.type == H5D_CHUNKED && H5D__chunk_is_space_alloc(&layout.storage)) {
        
        if (H5D__chunk_bh_info(loc, oh, &layout, &(bh_info->index_size)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine chunked dataset btree info");
    } 
    else if (layout.type == H5D_VIRTUAL && (layout.storage.u.virt.serial_list_hobjid.addr != HADDR_UNDEF)) {
        size_t virtual_heap_size;

        
        if (H5HG_get_obj_size(loc->file, &(layout.storage.u.virt.serial_list_hobjid), &virtual_heap_size) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "can't get global heap size for virtual dataset mapping");

        
        bh_info->heap_size = (hsize_t)virtual_heap_size;
    } 

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_EFL_ID)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "unable to check for EFL message");

    if (exists && H5D__efl_is_space_alloc(&layout.storage)) {
        
        memset(&efl, 0, sizeof(efl));

        
        if (NULL == H5O_msg_read_oh(loc->file, oh, H5O_EFL_ID, &efl))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't find EFL message");
        efl_read = true;

        
        if (H5D__efl_bh_info(loc->file, &efl, &(bh_info->heap_size)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't determine EFL heap info");
    } 

done:
    
    if (layout_read && H5O_msg_reset(H5O_LAYOUT_ID, &layout) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset data storage layout message");
    if (efl_read && H5O_msg_reset(H5O_EFL_ID, &efl) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset external file list message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dset_flush(void *_obj_ptr)
{
    H5D_t     *dset = (H5D_t *)_obj_ptr; 
    H5O_type_t obj_type;                 
    herr_t     ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    assert(dset);
    assert(&dset->oloc);

    
    if (H5O_obj_type(&dset->oloc, &obj_type) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get object type");
    if (obj_type != H5O_TYPE_DATASET)
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not a dataset");

    if (H5D__flush_real(dset) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to flush cached dataset info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
