/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5VMprivate.h" 

typedef struct H5D_compact_iovv_memmanage_ud_t {
    H5F_shared_t *f_sh;   
    void         *dstbuf; 
    const void   *srcbuf; 
} H5D_compact_iovv_memmanage_ud_t;

static herr_t  H5D__compact_construct(H5F_t *f, H5D_t *dset);
static herr_t  H5D__compact_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool open_op);
static bool    H5D__compact_is_space_alloc(const H5O_storage_t *storage);
static herr_t  H5D__compact_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t  H5D__compact_iovv_memmanage_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata);
static ssize_t H5D__compact_readvv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                   size_t dset_max_nseq, size_t *dset_curr_seq, size_t dset_size_arr[],
                                   hsize_t dset_offset_arr[], size_t mem_max_nseq, size_t *mem_curr_seq,
                                   size_t mem_size_arr[], hsize_t mem_offset_arr[]);
static ssize_t H5D__compact_writevv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                    size_t dset_max_nseq, size_t *dset_curr_seq, size_t dset_size_arr[],
                                    hsize_t dset_offset_arr[], size_t mem_max_nseq, size_t *mem_curr_seq,
                                    size_t mem_size_arr[], hsize_t mem_offset_arr[]);
static herr_t  H5D__compact_flush(H5D_t *dset);
static herr_t  H5D__compact_dest(H5D_t *dset);

const H5D_layout_ops_t H5D_LOPS_COMPACT[1] = {{
    H5D__compact_construct,      
    H5D__compact_init,           
    H5D__compact_is_space_alloc, 
    NULL,                        
    H5D__compact_io_init,        
    NULL,                        
    H5D__contig_read,            
    H5D__contig_write,           
    H5D__compact_readvv,         
    H5D__compact_writevv,        
    H5D__compact_flush,          
    NULL,                        
    H5D__compact_dest            
}};

H5FL_BLK_EXTERN(type_conv);

herr_t
H5D__compact_fill(const H5D_t *dset)
{
    H5D_fill_buf_info_t fb_info;                
    bool                fb_info_init = false;   
    herr_t              ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_COMPACT == dset->shared->layout.type);
    assert(dset->shared->layout.storage.u.compact.buf);
    assert(dset->shared->type);
    assert(dset->shared->space);

    
    
    if (H5D__fill_init(&fb_info, dset->shared->layout.storage.u.compact.buf, NULL, NULL, NULL, NULL,
                       &dset->shared->dcpl_cache.fill, dset->shared->type, (size_t)0,
                       dset->shared->layout.storage.u.compact.size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize fill buffer info");
    fb_info_init = true;

    
    if (fb_info.has_vlen_fill_type)
        
        if (H5D__fill_refill_vl(&fb_info, fb_info.elmts_per_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "can't refill fill value buffer");

done:
    
    if (fb_info_init && H5D__fill_term(&fb_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compact_construct(H5F_t *f, H5D_t *dset)
{
    hssize_t stmp_size;           
    hsize_t  tmp_size;            
    hsize_t  max_comp_data_size;  
    unsigned version;             
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);
    assert(dset->shared);

    
    for (u = 0; u < dset->shared->ndims; u++)
        if (dset->shared->max_dims[u] > dset->shared->curr_dims[u])
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "extendible compact dataset not allowed");

    
    stmp_size = H5S_GET_EXTENT_NPOINTS(dset->shared->space);
    assert(stmp_size >= 0);
    tmp_size = H5T_get_size(dset->shared->type);
    assert(tmp_size > 0);
    tmp_size = tmp_size * (hsize_t)stmp_size;
    H5_CHECKED_ASSIGN(dset->shared->layout.storage.u.compact.size, size_t, tmp_size, hssize_t);

    
    max_comp_data_size = H5O_MESG_MAX_SIZE - H5D__layout_meta_size(f, &(dset->shared->layout), false);
    if (dset->shared->layout.storage.u.compact.size > max_comp_data_size)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "compact dataset size is bigger than header message maximum size");

    
    if (dset->shared->layout.version < H5O_LAYOUT_VERSION_3) {
        version = MAX(dset->shared->layout.version,
                      MIN(H5O_layout_ver_bounds[H5F_LOW_BOUND(f)], H5O_LAYOUT_VERSION_3));

        
        if (version > H5O_layout_ver_bounds[H5F_HIGH_BOUND(f)])
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "layout version out of bounds");

        dset->shared->layout.version = version;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compact_init(H5F_t H5_ATTR_UNUSED *f, H5D_t *dset, hid_t H5_ATTR_UNUSED dapl_id,
                  bool H5_ATTR_UNUSED open_op)
{
    hssize_t snelmts;             
    hsize_t  nelmts;              
    size_t   dt_size;             
    hsize_t  data_size;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(H5D_COMPACT == dset->shared->layout.storage.type);

    
    if (0 == (dt_size = H5T_GET_SIZE(dset->shared->type)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get datatype size");
    if ((snelmts = H5S_GET_EXTENT_NPOINTS(dset->shared->space)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get number of elements in dataset's dataspace");
    nelmts = (hsize_t)snelmts;

    
    data_size = nelmts * dt_size;

    
    if (nelmts != (data_size / dt_size))
        HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "size of dataset's storage overflowed");

    
    if (dset->shared->layout.storage.u.compact.size != data_size)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                    "bad value from dataset header - size of compact dataset's data buffer doesn't match "
                    "size of dataset data");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5D__compact_is_space_alloc(const H5O_storage_t H5_ATTR_UNUSED *storage)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    
    FUNC_LEAVE_NOAPI(true)
} 

static herr_t
H5D__compact_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    FUNC_ENTER_PACKAGE_NOERR

    dinfo->store->compact.buf               = dinfo->dset->shared->layout.storage.u.compact.buf;
    dinfo->store->compact.dirty             = &dinfo->dset->shared->layout.storage.u.compact.dirty;
    dinfo->layout_io_info.contig_piece_info = NULL;

    
    io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
    io_info->no_selection_io_cause |= H5D_SEL_IO_NOT_CONTIGUOUS_OR_CHUNKED_DATASET;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__compact_iovv_memmanage_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_compact_iovv_memmanage_ud_t *udata = (H5D_compact_iovv_memmanage_ud_t *)_udata;
    H5FD_ctl_memcpy_args_t           op_args;
    uint64_t                         op_flags;
    H5FD_t                          *file_handle = NULL;
    herr_t                           ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (H5F_shared_get_file_driver(udata->f_sh, &file_handle) < 0)
        HGOTO_ERROR(H5E_IO, H5E_CANTGET, FAIL, "can't get file handle");

    
    op_flags = H5FD_CTL_ROUTE_TO_TERMINAL_VFD_FLAG | H5FD_CTL_FAIL_IF_UNKNOWN_FLAG;

    op_args.dstbuf  = udata->dstbuf;
    op_args.dst_off = dst_off;
    op_args.srcbuf  = udata->srcbuf;
    op_args.src_off = src_off;
    op_args.len     = len;

    
    if (H5FD_ctl(file_handle, H5FD_CTL_MEM_COPY, op_flags, &op_args, NULL) < 0)
        HGOTO_ERROR(H5E_IO, H5E_FCNTL, FAIL, "VFD memcpy request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5D__compact_readvv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info, size_t dset_max_nseq,
                    size_t *dset_curr_seq, size_t dset_size_arr[], hsize_t dset_offset_arr[],
                    size_t mem_max_nseq, size_t *mem_curr_seq, size_t mem_size_arr[],
                    hsize_t mem_offset_arr[])
{
    ssize_t ret_value = -1; 

    FUNC_ENTER_PACKAGE

    assert(io_info);
    assert(dset_info);

    
    if (H5F_SHARED_HAS_FEATURE(io_info->f_sh, H5FD_FEAT_MEMMANAGE)) {
        H5D_compact_iovv_memmanage_ud_t udata;

        
        udata.f_sh   = io_info->f_sh;
        udata.dstbuf = dset_info->buf.vp;
        udata.srcbuf = dset_info->store->compact.buf;

        
        if ((ret_value = H5VM_opvv(mem_max_nseq, mem_curr_seq, mem_size_arr, mem_offset_arr, dset_max_nseq,
                                   dset_curr_seq, dset_size_arr, dset_offset_arr,
                                   H5D__compact_iovv_memmanage_cb, &udata)) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");
    }
    else {
        
        if ((ret_value = H5VM_memcpyvv(dset_info->buf.vp, mem_max_nseq, mem_curr_seq, mem_size_arr,
                                       mem_offset_arr, dset_info->store->compact.buf, dset_max_nseq,
                                       dset_curr_seq, dset_size_arr, dset_offset_arr)) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5D__compact_writevv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info, size_t dset_max_nseq,
                     size_t *dset_curr_seq, size_t dset_size_arr[], hsize_t dset_offset_arr[],
                     size_t mem_max_nseq, size_t *mem_curr_seq, size_t mem_size_arr[],
                     hsize_t mem_offset_arr[])
{
    ssize_t ret_value = -1; 

    FUNC_ENTER_PACKAGE

    assert(io_info);
    assert(dset_info);

    
    if (H5F_SHARED_HAS_FEATURE(io_info->f_sh, H5FD_FEAT_MEMMANAGE)) {
        H5D_compact_iovv_memmanage_ud_t udata;

        
        udata.f_sh   = io_info->f_sh;
        udata.dstbuf = dset_info->store->compact.buf;
        udata.srcbuf = dset_info->buf.cvp;

        
        if ((ret_value = H5VM_opvv(dset_max_nseq, dset_curr_seq, dset_size_arr, dset_offset_arr, mem_max_nseq,
                                   mem_curr_seq, mem_size_arr, mem_offset_arr, H5D__compact_iovv_memmanage_cb,
                                   &udata)) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");
    }
    else {
        
        if ((ret_value = H5VM_memcpyvv(dset_info->store->compact.buf, dset_max_nseq, dset_curr_seq,
                                       dset_size_arr, dset_offset_arr, dset_info->buf.cvp, mem_max_nseq,
                                       mem_curr_seq, mem_size_arr, mem_offset_arr)) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vectorized memcpy failed");
    }

    
    *dset_info->store->compact.dirty = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compact_flush(H5D_t *dset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    if (dset->shared->layout.storage.u.compact.dirty) {
        dset->shared->layout.storage.u.compact.dirty = false;
        if (H5O_msg_write(&(dset->oloc), H5O_LAYOUT_ID, 0, H5O_UPDATE_TIME, &(dset->shared->layout)) < 0) {
            dset->shared->layout.storage.u.compact.dirty = true;
            HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to update layout message");
        }
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__compact_dest(H5D_t *dset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dset);

    
    dset->shared->layout.storage.u.compact.buf = H5MM_xfree(dset->shared->layout.storage.u.compact.buf);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5D__compact_copy(H5F_t *f_src, H5O_storage_compact_t *_storage_src, H5F_t *f_dst,
                  H5O_storage_compact_t *storage_dst, H5T_t *dt_src, H5O_copy_t *cpy_info)
{
    H5T_t        *dt_mem      = NULL; 
    H5T_t        *dt_dst      = NULL; 
    H5S_t        *buf_space   = NULL; 
    void         *buf         = NULL; 
    void         *bkg         = NULL; 
    void         *reclaim_buf = NULL; 
    H5D_shared_t *shared_fo =
        (H5D_shared_t *)cpy_info->shared_fo;           
    H5O_storage_compact_t *storage_src = _storage_src; 
    herr_t                 ret_value   = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(f_src);
    assert(storage_src);
    assert(f_dst);
    assert(storage_dst);
    assert(storage_dst->buf);
    assert(dt_src);

    
    if (shared_fo != NULL)
        storage_src = &(shared_fo->layout.storage.u.compact);

    
    if (H5T_detect_class(dt_src, H5T_VLEN, false) > 0) {
        H5T_path_t *tpath_src_mem, *tpath_mem_dst; 
        size_t      buf_size;                      
        size_t      nelmts;                        
        size_t      src_dt_size;                   
        size_t      tmp_dt_size;                   
        size_t      max_dt_size;                   
        hsize_t     buf_dim;                       

        
        if (NULL == (dt_mem = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");

        
        if (NULL == (dt_dst = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");
        if (H5T_set_loc(dt_dst, H5F_VOL_OBJ(f_dst), H5T_LOC_DISK) < 0) {
            (void)H5T_close_real(dt_dst);
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "cannot mark datatype on disk");
        } 

        
        if (NULL == (tpath_src_mem = H5T_path_find(dt_src, dt_mem)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to convert between src and mem datatypes");
        if (NULL == (tpath_mem_dst = H5T_path_find(dt_mem, dt_dst)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to convert between mem and dst datatypes");

        
        if (0 == (src_dt_size = H5T_get_size(dt_src)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        if (0 == (tmp_dt_size = H5T_get_size(dt_mem)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(src_dt_size, tmp_dt_size);
        if (0 == (tmp_dt_size = H5T_get_size(dt_dst)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(max_dt_size, tmp_dt_size);

        
        if (0 == (nelmts = storage_src->size / src_dt_size))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "element size too large");

        
        buf_size = nelmts * max_dt_size;

        
        buf_dim = nelmts;

        
        if (NULL == (buf_space = H5S_create_simple((unsigned)1, &buf_dim, NULL)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create simple dataspace");

        
        if (NULL == (reclaim_buf = H5FL_BLK_MALLOC(type_conv, buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (NULL == (buf = H5FL_BLK_MALLOC(type_conv, buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        H5MM_memcpy(buf, storage_src->buf, storage_src->size);

        
        if (NULL == (bkg = H5FL_BLK_MALLOC(type_conv, buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (H5T_convert(tpath_src_mem, dt_src, dt_mem, nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

        
        H5MM_memcpy(reclaim_buf, buf, buf_size);

        
        memset(bkg, 0, buf_size);

        
        if (H5T_convert(tpath_mem_dst, dt_mem, dt_dst, nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

        H5MM_memcpy(storage_dst->buf, buf, storage_dst->size);

        if (H5T_reclaim(dt_mem, buf_space, reclaim_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reclaim variable-length data");
    } 
    else if (H5T_get_class(dt_src, false) == H5T_REFERENCE) {
        if (f_src != f_dst) {
            
            if (cpy_info->expand_ref) {
                
                if (H5O_copy_expand_ref(f_src, dt_src, storage_src->buf, storage_src->size, f_dst,
                                        storage_dst->buf, cpy_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy reference attribute");
            } 
            else
                
                memset(storage_dst->buf, 0, storage_src->size);
        } 
        else
            
            H5MM_memcpy(storage_dst->buf, storage_src->buf, storage_src->size);
    } 
    else
        
        H5MM_memcpy(storage_dst->buf, storage_src->buf, storage_src->size);

    
    storage_dst->dirty = true;

done:
    if (dt_dst && (H5T_close(dt_dst) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (dt_mem && (H5T_close(dt_mem) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (buf_space && H5S_close(buf_space) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary dataspace");
    if (buf)
        buf = H5FL_BLK_FREE(type_conv, buf);
    if (reclaim_buf)
        reclaim_buf = H5FL_BLK_FREE(type_conv, reclaim_buf);
    if (bkg)
        bkg = H5FL_BLK_FREE(type_conv, bkg);

    FUNC_LEAVE_NOAPI(ret_value)
} 
