% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/two-column-layouts.R
\name{gt_two_column_layout}
\alias{gt_two_column_layout}
\title{Create a two-column layout from a list of two gt tables}
\usage{
gt_two_column_layout(
  tables = NULL,
  output = "viewer",
  filename = NULL,
  path = NULL,
  vwidth = 992,
  vheight = 600,
  ...,
  zoom = 2,
  expand = 5
)
}
\arguments{
\item{tables}{A \code{list()} of two tables, typically supplied by \code{gt_double_table()}}

\item{output}{A character string indicating the desired output, either \code{"save"} to save it to disk via \code{webshot}, \code{"viewer"} to return it to the RStudio Viewer, or \code{"html"} to return the raw HTML.}

\item{filename}{The filename of the table, must contain \code{.png} and only used if \code{output = "save"}}

\item{path}{An optional path of where to save the printed \code{.png}, used in conjunction with \code{filename}.}

\item{vwidth}{Viewport width. This is the width of the browser "window" when passed to \code{webshot2::webshot()}.}

\item{vheight}{Viewport height This is the height of the browser "window" when passed to \code{webshot2::webshot()}.}

\item{...}{Additional arguments passed to \code{webshot2::webshot()}, only to be used if \code{output = "save"}, saving the two-column layout tables to disk as a \code{.png}.}

\item{zoom}{Argument to \code{webshot2::webshot()}. A number specifying the zoom factor. A zoom factor of 2 will result in twice as many pixels vertically and horizontally. Note that using 2 is not exactly the same as taking a screenshot on a HiDPI (Retina) device: it is like increasing the zoom to 200 doubling the height and width of the browser window. This differs from using a HiDPI device because some web pages load different, higher-resolution images when they know they will be displayed on a HiDPI device (but using zoom will not report that there is a HiDPI device).}

\item{expand}{Argument to \code{webshot2::webshot()}. A numeric vector specifying how many pixels to expand the clipping rectangle by. If one number, the rectangle will be expanded by that many pixels on all sides. If four numbers, they specify the top, right, bottom, and left, in that order. When taking screenshots of multiple URLs, this parameter can also be a list with same length as url with each element of the list containing a single number or four numbers to use for the corresponding URL.}
}
\value{
Saves a \code{.png} to disk if \code{output = "save"}, returns HTML to the viewer via \code{htmltools::browsable()} when \code{output = "viewer"}, or returns raw HTML if \code{output = "html"}.
}
\description{
This function takes a \code{list()} of two gt-tables and returns
them as a two-column layout. The expectation is that the user either supplies
two tables like \code{list(table1, table2)}, or passes the output of \code{gt_double_table()}
into this function. The user should indicate whether they want to return the
HTML to R's viewer with \code{output = "viewer"} to "view" the final output, or to
save to disk as a \code{.png} via \verb{output = "save".} Note that this is a relatively
complex wrapper around \code{htmltools::div()} + \code{webshot2::webshot()}. Additional
arguments can be passed to \code{webshot2::webshot()} if the automatic output is not
satisfactory. In most situations, modifying the \code{vwidth} argument is sufficient
to get the desired output, but all arguments to \code{webshot2::webshot()} are
available by their original name via the passed \code{...}.
}
\section{Examples}{


\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(gt)
# add row numbers and drop some columns
my_cars <- mtcars \%>\%
  dplyr::mutate(row_n = dplyr::row_number(), .before = mpg) \%>\%
  dplyr::select(row_n, mpg:drat)

# create two tables, just split half/half
tab1 <- my_cars \%>\%
  dplyr::slice(1:16) \%>\%
  gt() \%>\%
  gtExtras::gt_color_rows(columns = row_n, domain = 1:32)

tab2 <- my_cars \%>\%
  dplyr::slice(17:32) \%>\%
  gt() \%>\%
  gtExtras::gt_color_rows(columns = row_n, domain = 1:32)

# put the tables in a list
listed_tables <- list(tab1, tab2)

# pass list to the gt_two_column_layout function
gt_two_column_layout(listed_tables)

# better option!
# create a one-argument function, passing data to `gt::gt()`
my_gt_fn <- function(x) \{
  gt(x) \%>\%
    gtExtras::gt_color_rows(columns = row_n, domain = 1:32)
\}

# pass data, your function and the nrows
my_tables <- gt_double_table(my_cars, my_gt_fn, nrows = nrow(my_cars) / 2)

# boom, this will return it to the viewer
my_output <- gt_two_column_layout(my_tables)
# if you wanted to save it out instead, could use the code below
# gt_two_column_layout(my_tables, output = "save",
#                      filename = "basic-two-col.png",
#                      vwidth = 550, vheight = 620)
}\if{html}{\out{</div>}}
}

\section{Figures}{

\if{html}{\figure{basic-two-col.png}{options: width=60\%}}
}

\seealso{
Other Utilities: 
\code{\link{add_text_img}()},
\code{\link{fa_icon_repeat}()},
\code{\link{fmt_pad_num}()},
\code{\link{fmt_pct_extra}()},
\code{\link{fmt_symbol_first}()},
\code{\link{generate_df}()},
\code{\link{gt_add_divider}()},
\code{\link{gt_badge}()},
\code{\link{gt_double_table}()},
\code{\link{gt_duplicate_column}()},
\code{\link{gt_fa_column}()},
\code{\link{gt_fa_rank_change}()},
\code{\link{gt_fa_rating}()},
\code{\link{gt_fa_repeats}()},
\code{\link{gt_highlight_cols}()},
\code{\link{gt_highlight_rows}()},
\code{\link{gt_img_circle}()},
\code{\link{gt_img_rows}()},
\code{\link{gt_index}()},
\code{\link{gt_merge_stack}()},
\code{\link{gtsave_extra}()},
\code{\link{pad_fn}()},
\code{\link{tab_style_by_grp}()}
}
\concept{Utilities}
