% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grpsel.R
\name{grpsel}
\alias{grpsel}
\title{Group subset selection}
\usage{
grpsel(
  x,
  y,
  group = seq_len(ncol(x)),
  penalty = c("grSubset", "grSubset+grLasso", "grSubset+Ridge"),
  loss = c("square", "logistic"),
  ls = FALSE,
  nlambda = 100,
  ngamma = 10,
  gamma.max = 100,
  gamma.min = 1e-04,
  lambda = NULL,
  gamma = NULL,
  pmax = ncol(x),
  gmax = length(unique(group)),
  subset.factor = NULL,
  lasso.factor = NULL,
  ridge.factor = NULL,
  alpha = 0.99,
  eps = 1e-04,
  max.cd.iter = 10000,
  max.ls.iter = 100,
  active.set = TRUE,
  active.set.count = 3,
  sort = TRUE,
  screen = 500,
  orthogonalise = TRUE,
  warn = TRUE
)
}
\arguments{
\item{x}{a predictor matrix}

\item{y}{a response vector}

\item{group}{a vector of length \code{ncol(x)} with the jth element identifying the group that
the jth predictor belongs to; alternatively, a list of vectors with the kth vector identifying
the predictors that belong to the kth group (useful for overlapping groups)}

\item{penalty}{the type of penalty to apply; one of 'grSubset', 'grSubset+grLasso', or
'grSubset+Ridge'}

\item{loss}{the type of loss function to use; 'square' for linear regression or 'logistic' for
logistic regression}

\item{ls}{a logical indicating whether to perform local search after coordinate descent;
typically leads to higher quality solutions}

\item{nlambda}{the number of group subset regularisation parameters to evaluate when
\code{lambda} is computed automatically; may evaluate fewer parameters if \code{pmax} or
\code{gmax} is reached first}

\item{ngamma}{the number of group lasso or ridge regularisation parameters to evaluate when
\code{gamma} is computed automatically}

\item{gamma.max}{the maximum value for gamma when \code{penalty='grSubset+Ridge'}; when
\code{penalty='grSubset+grLasso'} \code{gamma.max} is computed automatically from the data}

\item{gamma.min}{the minimum value for gamma when \code{penalty='grSubset+Ridge'} and the minimum
value for \code{gamma} as a fraction of \code{gamma.max} when \code{penalty='grSubset+grLasso'}}

\item{lambda}{an optional list of decreasing sequences of group subset parameters; the list
should contain a vector for each value of \code{gamma}}

\item{gamma}{an optional decreasing sequence of L21 or L22 parameters}

\item{pmax}{the maximum number of predictors ever allowed to be active; ignored if \code{lambda}
is supplied}

\item{gmax}{the maximum number of groups ever allowed to be active; ignored if \code{lambda} is
supplied}

\item{subset.factor}{a vector of penalty factors applied to the group subset penalty; equal to
the group sizes by default}

\item{lasso.factor}{a vector of penalty factors applied to the group lasso penalty; equal to the
square roots of the group sizes by default}

\item{ridge.factor}{a vector of penalty factors applied to the ridge penalty; equal to a
vector of ones by default}

\item{alpha}{the step size taken when computing \code{lambda} from the data; should be a value
strictly between 0 and 1; larger values typically lead to a finer grid of subset sizes}

\item{eps}{the convergence tolerance; convergence is declared when the relative maximum
difference in consecutive coefficients is less than \code{eps}}

\item{max.cd.iter}{the maximum number of coordinate descent iterations allowed per value of
\code{lambda} and \code{gamma}}

\item{max.ls.iter}{the maximum number of local search iterations allowed per value of
\code{lambda} and \code{gamma}}

\item{active.set}{a logical indicating whether to use active set updates; typically lowers the
run time}

\item{active.set.count}{the number of consecutive coordinate descent iterations in which a
subset should appear before running active set updates}

\item{sort}{a logical indicating whether to sort the coordinates before running coordinate
descent; required for gradient screening; typically leads to higher quality solutions}

\item{screen}{the number of groups to keep after gradient screening; smaller values typically
lower the run time}

\item{orthogonalise}{a logical indicating whether to orthogonalise within groups}

\item{warn}{a logical indicating whether to print a warning if the algorithms fail to converge}
}
\value{
An object of class grpsel; a list with the following components:
\item{beta}{a list of matrices whose columns contain fitted coefficients for a given value of
\code{lambda}; an individual matrix in the list for each value of \code{gamma}}
\item{gamma}{a vector containing the values of \code{gamma} used in the fit}
\item{lambda}{a list of vectors containing the values of \code{lambda} used in the fit; an
individual vector in the list for each value of \code{gamma}}
\item{np}{a list of vectors containing the number of active predictors per value of
\code{lambda}; an individual vector in the list for each value of \code{gamma}}
\item{ng}{a list of vectors containing the the number of active groups per value of
\code{lambda}; an individual vector in the list for each value of \code{gamma}}
\item{iter.cd}{a list of vectors containing the number of coordinate descent iterations per value
of \code{lambda}; an individual vector in the list for each value of \code{gamma}}
\item{iter.ls}{a list of vectors containing the number of local search iterations per value
of \code{lambda}; an individual vector in the list for each value of \code{gamma}}
\item{loss}{a list of vectors containing the evaluated loss function per value of \code{lambda}
evaluated; an individual vector in the list for each value of \code{gamma}}
}
\description{
Fits the regularisation surface for a regression model with a group subset selection
penalty. The group subset penalty can be combined with either a group lasso or ridge penalty
for shrinkage. The group subset parameter is \code{lambda} and the group lasso/ridge parameter is
\code{gamma}.
}
\details{
For linear regression (loss='square') the response and predictors are centred about
zero and scaled to unit l2-norm. For logistic regression (loss='logistic') only the predictors
are centred and scaled and an intercept is fit during the course of the algorithm.
}
\examples{
# Grouped data
set.seed(123)
n <- 100
p <- 10
g <- 5
group <- rep(1:g, each = p / g)
beta <- numeric(p)
beta[which(group \%in\% 1:2)] <- 1
x <- matrix(rnorm(n * p), n, p)
y <- x \%*\% beta + rnorm(n)
newx <- matrix(rnorm(p), ncol = p)

# Group subset selection
fit <- grpsel(x, y, group)
plot(fit)
coef(fit, lambda = 0.05)
predict(fit, newx, lambda = 0.05)

# Group subset selection with group lasso shrinkage
fit <- grpsel(x, y, group, penalty = 'grSubset+grLasso')
plot(fit, gamma = 0.05)
coef(fit, lambda = 0.05, gamma = 0.1)
predict(fit, newx, lambda = 0.05, gamma = 0.1)

# Group subset selection with ridge shrinkage
fit <- grpsel(x, y, group, penalty = 'grSubset+Ridge')
plot(fit, gamma = 0.05)
coef(fit, lambda = 0.05, gamma = 0.1)
predict(fit, newx, lambda = 0.05, gamma = 0.1)
}
\author{
Ryan Thompson <ryan.thompson@monash.edu>
}
