


#' Plot flight data
#'
#' Plotting function to visualize the flight connections from the [`flights`] dataset.
#' This function requires the package `ggplot2` to be installed.
#'
#' @param airportIndices The indices of the airports (w.r.t. `airports_sel`) to include.
#' @param airports_sel The airports to plot. Might be further subset by arguments `airportIndeices`, `graph`.
#' If `NULL`, then [`flights`]`$aiports` will be used.
#' @param connections_sel A three columns data frame as output by [flightCountMatrixToConnectionList()].
#' If `NULL`, then [`flights`]$`nFlights` will be used to construct one.
#' @param graph An optional [`igraph::graph`] object, containing a flight graph to plot.
#' Vertices should either match the selected airports in number and order,
#' or be named with the corresponding IATA codes of the airports they represent.
#' @param plotAirports Logical. Whether to plot the airports specified.
#' @param plotConnections Logical. Whether to plot the connections specified.
#' @param returnGGPlot If `TRUE`, a [`ggplot2::ggplot`] object is returned and not plotted immediately.
#' @param useAirportNFlights Logical. Whether to vary the size of the circles representing airports in the plot,
#' according to the number of flights at that airport.
#' @param useConnectionNFlights Logical. Whether to vary the size of the edges representing connections in the plot,
#' according to the number of flights on that connection.
#' @param minNFlights Numeric scalar. Only plot connections with at least this many flights.
#' @param map String or `NULL`. What map to use as the background image. Is passed to `ggplot2::map_data()`.
#' @param vertexColors Optional vector, named with IATA codes, to be used as colors for the vertices/airports.
#' @param vertexShapes Optional vector, named with IATA codes, to be used as shapes for the vertices/airports. Is coerced to `character`.
#' @param xyRatio Approximate X-Y-ratio (w.r.t. distance on the ground) of the area shown in the plot.
#' @param clipMap Logical or numeric scalar. Whether to ignore the map image when determining the axis limits of the plot.
#' If it is a positive scalar, the plot limits are extended by that factor.
#' @param useLatex Whether to format numbers etc. as latex code (useful when plotting to tikz).
#' @param edgeAlpha Numeric scalar between 0 and 1. The alpha value to be used when plotting edges/connections.
#'
#' @return If `returnGGPlot` is `TRUE`, a [`ggplot2::ggplot`] object, otherwise `NULL`.
#' @examples
#' # Plot all airports in the dataset
#' plotFlights(plotConnections = FALSE, map = 'world')
#' 
#' # Plot a selection of airports
#' plotFlights(c('JFK', 'SFO', 'LAX'), useConnectionNFlights = TRUE, useAirportNFlights = TRUE)
#' 
#' # Plot airports with a custom connections graph
#' IATAs <- c('ACV', 'BFL', 'EUG', 'SFO', 'MRY')
#' graph <- igraph::make_full_graph(length(IATAs))
#' plotFlights(IATAs, graph=graph, clipMap = 1.5)
#' 
#' @export
plotFlights <- function(
  airportIndices = NULL,
  airports_sel = NULL,
  connections_sel = NULL,
  graph = NULL,
  plotAirports = TRUE,
  plotConnections = TRUE,
  returnGGPlot = FALSE,
  useAirportNFlights = FALSE,
  useConnectionNFlights = FALSE,
  minNFlights = 0,
  map = 'state',
  vertexColors = NULL,
  vertexShapes = NULL,
  xyRatio = NULL,
  clipMap = FALSE,
  useLatex = FALSE,
  edgeAlpha = 0.2
) {
  # Make sure ggplot2 is installed
  ggplotAvailable <- requireNamespace('ggplot2')
  if(!ggplotAvailable){
    stop('ggplot2 needs to be installed')
  }
  
  # This makes sure `flights` is available, even if the package was not called with `library()`
  flights <- graphicalExtremes::flights

  ## Fill unspecified inputs
  # Use all airports, conenctions if not specified
  if(is.null(airports_sel)){
    airports_sel <- flights$airports
  }
  # Use airportIndices from graph, if not specified:
  if(!is.null(graph)){
    vNames <- igraph::V(graph)$name
    if(!is.null(vNames) && is.null(airportIndices)){
      airportIndices <- vNames
    }
  }
  # Set map to NULL if not specified:
  if(is.null(map) || is.na(map) || nchar(map) == 0){
    map <- NULL
  }
  # Make selection of airports:
  if(!is.null(airportIndices)){
    airports_sel <- airports_sel[airportIndices,]
  }
  IATAS <- airports_sel[,'IATA']
  # If clipMap is numeric, it is used to zoom in/out of the clipped map:
  stretchMap <- 1
  if(is.numeric(clipMap)){
    stretchMap <- fitInInterval(1 * clipMap, 0, Inf)
    clipMap <- (clipMap > 0)
  }
  
  ## If necessary, compute connections and flight counts
  computeConnections <- is.null(connections_sel) && plotConnections
  computeNFlights <- is.null(airports_sel$nFlights) && useAirportNFlights && plotAirports
  if(computeConnections || computeNFlights){
    # used multiple times below
    nFlightMat <- apply(flights$flightCounts[IATAS,IATAS,], c(1,2), sum)
  }
  # Make connections list if not given
  if(computeConnections){
    connections_sel <- flightCountMatrixToConnectionList(nFlightMat)
  }
  # Compute flight counts per airport
  if(computeNFlights){
    # add arriving and departing flights
    airports_sel$nFlights <- rowSums(nFlightMat) + colSums(nFlightMat)
  }

  # Make sure number of graph vertices and selected airports match:
  if(!is.null(graph)){
    nVertices <- igraph::vcount(graph)
    nAirports <- nrow(airports_sel)
    if(nVertices != nAirports){
      stop(sprintf(
        'Number of vertices (%d) and number of selected airports (%d) are different!',
        nVertices,
        nAirports
      ))
    }
  }

  # Add vertex colors to data frame if specified
  aesVertexColor <- NULL
  if(!is.null(vertexColors)){
    aesVertexColor <- 'vertexColor'
    airports_sel$vertexColor <- vertexColors[IATAS]
  }

  # Add vertex shapes to data frame if specified
  aesVertexShape <- NULL
  if(!is.null(vertexShapes)){
    aesVertexShape <- 'vertexShape'
    airports_sel$vertexShape <- as.character(vertexShapes[IATAS])
  }

  # Prepare connections plotting:
  if(plotConnections){
    # Make selection of connections:
    if(is.null(graph)){
      # Select all connections, that are between selected airports
      # and with >= minNFlights flights:
      ind <- (
        (
          connections_sel$departureAirport %in% IATAS
          | connections_sel$arrivalAirport %in% IATAS
        )
        & connections_sel$nFlights >= minNFlights
      )
      connections_sel <- connections_sel[ind,]
    } else{
      # Convert graph to connections list
      m <- igraph::get.edgelist(graph, names=FALSE)
      connections_graph <- data.frame(matrix(IATAS[m], ncol = 2))
      airportColNames <- c('departureAirport', 'arrivalAirport')
      colnames(connections_graph) <- airportColNames

      # Read nFlights per connection from connections_sel
      rownames(connections_sel) <- paste0(
        connections_sel$departureAirport, '_', connections_sel$arrivalAirport 
      )
      rownames(connections_graph) <- paste0(
        connections_graph$departureAirport, '_', connections_graph$arrivalAirport 
      )
      if(useConnectionNFlights){
        connections_graph$nFlights <- 0
        connections_graph$nFlights <- connections_sel[
          rownames(connections_graph),
          'nFlights'
        ]
        connections_graph$nFlights[is.na(connections_graph$nFlights)] <- 1
      }
      connections_sel <- connections_graph
    }


    # Add coordinates to selected connections:
    connections_sel$x0 <- airports_sel[connections_sel$departureAirport, 'Longitude']
    connections_sel$y0 <- airports_sel[connections_sel$departureAirport, 'Latitude']
    connections_sel$x1 <- airports_sel[connections_sel$arrivalAirport, 'Longitude']
    connections_sel$y1 <- airports_sel[connections_sel$arrivalAirport, 'Latitude']
  }
  
  # Specify whether to size vertices/edges by nFlights:
  aesSizeNodes <- NULL
  aesSizeEdges <- NULL
  if(useAirportNFlights){
    aesSizeNodes <- 'nFlights'
  }
  if(useConnectionNFlights){
    aesSizeEdges <- 'nFlights'
  }

  # Main plot object:
  ggp <- (
    ggplot2::ggplot()
    + ggplot2::xlab(NULL)
    + ggplot2::ylab(NULL)
    + ggplot2::scale_x_continuous(labels = function(x) formatDegrees(x, 'EW', useLatex))
    + ggplot2::scale_y_continuous(labels = function(x) formatDegrees(x, 'NS', useLatex))
  )
  
  # Plot US map in background:
  if(!is.null(map)){
    dmap <- ggplot2::map_data(map)
    ggp <- ggp + ggplot2::geom_polygon(
      data = dmap,
      ggplot2::aes_string(x = 'long', y = 'lat', group = 'group'),
      color = "grey65",
      fill = "#f9f9f9",
      size = 0.2
    )
  }
  
  # Manually set axes limits (clips map, sets aspect ratio):
  # Note: might be improced using a different crs from
  # https://ggplot2.tidyverse.org/reference/ggsf.html
  if(!is.null(xyRatio) || (clipMap && !is.null(map))){
    xData <- airports_sel$Longitude
    yData <- airports_sel$Latitude
    if(!clipMap && !is.null(map)){
      m <- ggplot2::map_data(map)
      xData <- c(xData, range(m$long))
      yData <- c(yData, range(m$lat))
    }
    limits <- computeLimits(
      xData,
      yData,
      xyRatio = xyRatio,
      stretch = stretchMap
    )
    ggp <- ggp + ggplot2::coord_cartesian(
      xlim = limits$xlim,
      ylim = limits$ylim
    )
    if(!is.null(limits$xyRatio)){
      ggp <- ggp + ggplot2::theme(
        aspect.ratio = 1/limits$xyRatio
      )
    }
  }

  # Plot airports:
  if(plotAirports){
    ggp <- ggp + ggplot2::geom_point(
      data = airports_sel,
      ggplot2::aes_string(
        x = 'Longitude',
        y = 'Latitude',
        size = aesSizeNodes,
        col = aesVertexColor,
        shape = aesVertexShape
      ),
      na.rm = TRUE,
      alpha = 1
    )
  }
  
  # Plot connections:
  if(plotConnections){
    ggp <- ggp + ggplot2::geom_segment(
      data = connections_sel,
      ggplot2::aes_string(
        x = 'x0',
        xend = 'x1',
        y = 'y0',
        yend = 'y1',
        size = aesSizeEdges
      ),
      alpha = edgeAlpha
    )
  }
  
  # Return ggplot object if specified:
  if(returnGGPlot){
    return(ggp)
  }

  # Call plot:
  plot(ggp)
  return(invisible(NULL))
}


#' Convert flight counts to connection list
#' 
#' Convert a numeric matrix containing flight counts between ariports to a data
#' frame containing a list of connections.
#' 
#' @param nFlightsPerConnection A square, numeric matrix with identical column- and row-names.
#' Each entry represents the number of flights from the airport indexing the row to
#' the airport indexing the column in some arbitrary time period.
#' @param directed Logical scalar. Whether flights A->B and B->A should be considered separately.
#' 
#' @return A data frame with columns `departureAirport`, `arrivalAirport`, `nFlights`.
#' Each row represents one connection with >=1 flights in the input matrix.
#' 
#' @export
flightCountMatrixToConnectionList <- function(nFlightsPerConnection, directed=TRUE){
  # sum up over years if necessary
  if(length(dim(nFlightsPerConnection))){
    nFlightsPerConnection <- apply(nFlightsPerConnection, c(1,2), sum)
  }
  # order rows/columns:
  perm <- order(colnames(nFlightsPerConnection))
  nFlightsPerConnection <- nFlightsPerConnection[perm, perm]
  allAirports <- colnames(nFlightsPerConnection)
  
  if(directed){
    mask <- matrix(TRUE, length(allAirports), length(allAirports))
  } else{
    nFlightsPerConnection <- nFlightsPerConnection + t(nFlightsPerConnection)
    mask <- upper.tri(nFlightsPerConnection)
  }

  ind <- which(
    (nFlightsPerConnection > 0) & mask,
    arr.ind = TRUE
  )
  flightCounts <- nFlightsPerConnection[ind]
  dep <- allAirports[ind[,1]]
  arr <- allAirports[ind[,2]]
  df <- data.frame(
    departureAirport = dep,
    arrivalAirport = arr,
    nFlights = flightCounts
  )
  return(df)
}

# Helper function to format (coordinate) degrees used as axis labels
formatDegrees <- function(decDeg, direction = 'NS', latex=TRUE){
  dir <- 1 + (decDeg < 0)
  dirStrings <- substring(direction, dir, dir)
  if(latex){
    degString <- '^{\\circ}'
    delim <- '$'
    dirStrings <- paste0('\\mathrm{', dirStrings, '}')
  } else{
    degString <- '\u00B0'
    delim <- ''
  }
  decDeg <- abs(decDeg)
  isNa <- is.na(decDeg)
  degStrings <- rep('', length(decDeg))
  degStrings[!isNa] <- paste0(
    delim,
    formatDegrees2(decDeg[!isNa], dirStrings[!isNa], degString),
    delim
  )
  return(degStrings)
}
formatDegrees2 <- function(decDeg, dirStrings, degString){
  dms <- decDegToDegMinSec(decDeg)
  x <- paste0(dms[,1], degString, dirStrings)
  if(!any(duplicated(x))){
    return(x)
  }
  print(x)
  x <- paste0(dms[,1], degString, ' ', dms[,2], "'", dirStrings)
  if(!any(duplicated(x))){
    return(x)
  }
  print(x)
  x <- paste0(dms[,1], degString, ' ', dms[,2], "' ", dms[,3], '"', dirStrings)
  return(x)
}

decDegToDegMinSec <- function(decDeg, asString = FALSE){
  deg <- floor(decDeg)
  decMin <- (decDeg - deg) * 60
  min <- floor(decMin)
  decSec <- (decMin - min) * 60
  if(asString){
    return(paste(deg, min, decSec))
  }
  return(cbind(deg, min, decSec))
}

#' Helper function to compute the axis limits of a plot
#' with given x, y data and optionally a fixed x-y-ratio and
#' correcting the latitude/longitude scale at different latitudes
#' @keywords internal
computeLimits <- function(xData, yData, xyRatio=1, convertLatLong=TRUE, stretch = 1){
  xRange <- range(xData)
  yRange <- range(yData)

  xMid <- mean(xRange)
  yMid <- mean(yRange)

  xRadius <- diff(xRange) / 2
  yRadius <- diff(yRange) / 2

  if(is.null(xyRatio)){
    # Just don't scale
    xScale <- 1
    yScale <- 1
    xyRatio0 <- NULL
  } else{
    xyRatio0 <- xyRatio
    # Use lat/long to account for spherical coords:
    if(convertLatLong){
      xyRatio <- xyRatio * cos(pi/180 * yMid)
    }

    # Scale x, y according to xyRatio
    xScale <- xyRatio / (xRadius / yRadius)
    yScale <- 1/xScale
    
    xScale <- max(1, xScale)
    yScale <- max(1, yScale)
  }

  return(list(
    xlim = xMid + xRadius * xScale * c(-1, 1) * stretch,
    ylim = yMid + yRadius * yScale * c(-1, 1) * stretch,
    xyRatio = xyRatio0
  ))
}
