#' Create a codebook for the oTree code
#' @description
#' Create a codebook of your oTree code by automatically scanning
#' your project folder and retrieving all variables' information.
#' @details
#' This code works only when there are no dictionaries used (for
#' example in the session configurations in \code{settings.py}).
#'
#' Caution 1: Multiline comments are ignored, meaning that all variables
#' commented out in this manner will nevertheless be included in the codebook.
#' In contrast, variables commented out with line comments will not
#' appear in the codebook.
#'
#' Caution 2: If there are commas in the value strings, they might be
#' used to split the text. Please manually insert a backslash symbol
#' in front of the commas to avoid that.
#' E.g. \code{"Yes, I will"} -> \code{"Yes\, I will"}.
#'
#' Caution 3: This code cannot interpret variables that were imported from other
#' files (for example CSV files) and that have special formatting
#' included (e.g., special string formatting in Python such
#' as \code{float(1.4)} to represent a float number).
#'
#' Caution 4: This code was developed and tested with basic oTree codes
#' and has not been verified for compatibility with oTree versions
#' later than 5.4.0.
#' If you experience issues with newer versions or more complex code structures,
#' please open an issue on GitHub.
#'
#' @param path Character. Path of the oTree experiment.
#' @param fsource Character. \code{"init"} if information should be taken
#' from the \code{init.py} files (newer oTree code with 5.x
#' format). \code{"models"}
#' (or \code{"model"}) if the information
#' should be taken from the \code{models.py} files
#' (older oTree code with 3.x format).
#' @param output Character. \code{"list"} if the output should contain a
#' list of variables and their information. \code{"file"} if the output
#' should be a file such as a Word or PDF file.
#' \code{"both"} if the output should contain a file and a list.
#' @param output_dir Character. The absolute path where
#' the function's output will be saved.
#' Only absolute paths are allowed for this parameter.
#' Relative paths can be specified in the \code{output_file} parameter.
#' @param output_file Character. The name of the output file generated by the function.
#' The file name can be provided with or without an extension.
#' Relative paths are also allowed in the file name.
#' @param output_format Character. Format of the file output.
#' This is the format that is passed to the \code{output_format}
#' argument of \link[rmarkdown:render]{rmarkdown::render}.
#' You must use either \code{"pdf_document"}, \code{"html_document"}, \code{"word_document"}, \code{"odt_document"}, \code{"rtf_document"}, \code{"md_document"}, or \code{"latex_document"}.
#' @param output_open Logical. \code{TRUE} if file output should be opened after creation.
#' @param app_doc Logical. \code{TRUE} if app documentation should be included in
#' output file.
#' @param app Character. Name of the included app(s).
#' Default is to use all apps.
#' This argument can not be used simultaneously as the argument \code{app_rm}.
#' @param app_rm Character. Name of the excluded app(s).
#' Default is to exclude no apps.
#' This argument can not be used simultaneously as the argument app.
#' @param doc_info Logical. \code{TRUE} if a message with information on all
#' variables without documentation should also be returned.
#' @param sort Character. Vector that specifies the order of
#' the apps in the codebook.
#' @param settings_replace Character or \code{NULL}.
#' Specifies how to handle references to settings variables.
#' Use \code{"global"} to replace references with the global settings variables
#' defined in \code{settings.py}.
#' Use \code{"user"} to replace references with the variables
#' provided in the \code{user_settings} argument.
#' Use \code{NULL} to leave references to settings variables unchanged.
#' Caution: This function does not use variables defined in \code{SESSION_CONFIGS}.
#' If you vary settings variables in \code{SESSION_CONFIGS},
#' set \code{settings_replace} to \code{"user"} and manually replace them using the
#' \code{user_settings} argument.
#' @param user_settings List. List of variables in the \code{settings.py} file
#' that are used to replace setting variable references.
#' This is only used if \code{settings_replace = "user"} and should be used when
#' setting variables are defined within the \code{SESSION_CONFIGS}.
#' @param preamb Logical.
#' \code{TRUE} if a preamble should be printed that explains
#' how oTree saves variables.
#' @param include_cons Logical.
#' \code{TRUE} if there should be a section for the constants in the codebook.
#' @param params List.
#' List of variable name and value pairs to be passed to the RmD file.
#' Only relevant if argument output \code{"file"} or \code{"both"} if chosen.
#' @param date Date that is passed to the Rmd file.
#' Either \code{"today"}, \code{NULL}, or a user defined date.
#' Only relevant if argument output \code{"file"} or \code{"both"} if chosen.
#' @param encoding Character. Encoding of the created Markdown file.
#' As in \link[knitr:knit]{knitr::knit}, this argument is
#' always assumed to be \code{UTF-8}
#' and ignored.
#' @param title Character. Title of output file.
#' @param subtitle Character. Subtitle of output file.
#' @import knitr
#' @import pander
#' @import rmarkdown
#' @import stringr
#' @import utils
#' @returns
#' The function returns two main types of outputs:
#'
#' (a) a list of variables along with their information
#'
#' (b) a file containing the codebook for the experiment
#'
#' If \code{doc_info} is \code{TRUE} it also returns a message containing the names of
#' all variables that have no documentation.
#' @examplesIf rlang::is_installed("withr")
#' # The examples use a slightly modified version of the official oTree sample codes.
#'
#' # Make a codebook and resort the apps
#' combined_codebook <- codebook(
#'   path = system.file("extdata/ocode_new", package = "gmoTree"),
#'   output = "list",
#'   fsource = "init",
#'   doc_info = FALSE)
#'
#' # Show the structure of the codebook
#' str(combined_codebook, 1)
#' str(combined_codebook$bargaining$Player, 1)
#'
#' # Make a codebook with only the "bargaining" app
#' combined_codebook <- codebook(
#'   path = system.file("extdata/ocode_new", package = "gmoTree"),
#'   output = "list",
#'   fsource = "init",
#'   app = "bargaining",
#'   doc_info = FALSE)
#'
#' # Show the structure of the codebook 
#' str(combined_codebook, 1)
#' str(combined_codebook$bargaining$Player, 1)
#'
#' # Make a codebook with all but the "bargaining" app
#' combined_codebook <- codebook(
#'   path = system.file("extdata/ocode_new", package = "gmoTree"),
#'   output = "list",
#'   fsource = "init",
#'   app_rm = "bargaining",
#'   doc_info = FALSE)
#'
#' # Show the structure of the codebook 
#' str(combined_codebook, 1)
#' str(combined_codebook$bargaining$Player, 1)
#'
#' # Use oTree code in 3.x format
#' combined_codebook <- codebook(
#'   path = system.file("extdata/ocode_z", package = "gmoTree"),
#'   fsource = "model",
#'   output = "list",
#'   doc_info = TRUE)
#'
#' # Show the structure of the codebook
#' str(combined_codebook, 1)
#'
#' \dontrun{
#'
#' # Create a codebook PDF with authors' names and todays' date
#' codebook(
#'   path = system.file("extdata/ocode_z", package = "gmoTree"),
#'   fsource = "init",
#'   doc_info = FALSE,
#'   output = "file",
#'   output_format = "pdf_document",
#'   date = "today",
#'   title = "My Codebook",
#'   subtitle = "codebook created with gmoTree",
#'   params = list(author = c("Max Mustermann", "John Doe"))
#'   )
#'
#' # Create a codebook PDF and save it in a subfolder of the
#' # current folder:
#' # "C:/Users/pzauchner/Nextcloud/R_analyses/cb/cb.pdf"
#' getwd() # "C:/Users/pzauchner/Nextcloud/R_analyses"
#' dir.create("cb")
#' combined_codebook <- gmoTree::codebook(
#'   path = "C:/Users/pzauchner/Nextcloud/R_analyses/oTree",
#'   fsource = "models",
#'   output = "both",
#'   output_file = "cb/cb.pdf",
#'   output_format = "pdf_document")
#'
#' # You can also omit *.pdf after the file name
#' combined_codebook <- gmoTree::codebook(
#'   path = "C:/Users/pzauchner/Nextcloud/R_analyses/oTree",
#'   fsource = "models",
#'   output = "both",
#'   output_file = "cb/cb",
#'   output_format = "pdf_document")
#' }

#' @export
codebook <- function(path = ".",
                     fsource = "init",
                     output = "both",
                     output_dir = getwd(),
                     output_file = "codebook",
                     output_format = "pdf_document",
                     output_open = TRUE,
                     app_doc = TRUE,
                     app = NULL,
                     app_rm = NULL,
                     doc_info = TRUE,
                     sort = NULL,
                     settings_replace = "global",
                     user_settings = NULL,
                     include_cons = TRUE,
                     preamb = FALSE,
                     encoding = "UTF-8",
                     title = "Codebook",
                     subtitle = "created with gmoTree",
                     params = NULL,
                     date = "today") {

  output_dir_input <- substitute(output_dir)

  # Stop part 1
    # Path and fsource  ####
      # Define path
      if (!is.null(path)) {
        # Change Windows paths to paths that can be read by Ubuntu
        path <- gsub("\\\\", "/", path)
      } else {
        stop("Path must not be NULL!")
      }

      # Check if path exists
      if (!dir.exists(path)) {
        stop("The path ", path, " does not exist!",
             " You are currently in ",
             getwd())
      }

      # Check if fsource is valid
      if (length(fsource) > 1L) {
        stop("Please enter only one fsource!")
      }

      if (is.null(fsource) ||
          (fsource != "init" &&
           fsource != "model" &&
           fsource != "models")) {
        stop("fsource must be either \"init\", \"model\", or \"models\"!")
      }

    # Others  ####
    # Check output format
    if (!is.character(output) ||
        length(output) != 1L ||
        !(output %in% c("list", "both", "file"))) {
      stop("Output should be \"list\", \"both\", or \"file\"!")
    }

    # Check pandoc
    if (output != "list") {
      pandoc.installed <- system('pandoc -v', ignore.stdout = TRUE, ignore.stderr = TRUE) == 0

      if (!pandoc.installed) {
        stop("Please install pandoc before proceeding!")
      }
    }

    # Check settings_replace
    if (
      !is.null(settings_replace) &&
      !settings_replace %in% c("global", "user")) {
      stop("settings_replace must be either \"global\", \"user\", or NULL!")
    }

  # Manipulate arguments and load files  ####
    # If path in file names
    # Change Windows paths to paths that can be read by Ubuntu
    if (!is.null(output_file)) {
      output_file <- gsub("\\\\", "/", output_file)
    }

    # If file name starts with /
    output_file <- gsub("^/", "", output_file)

    if (fsource == "model" || fsource == "models") {

      files <- list.files(path,
                          pattern = "models\\.py",
                          full.names = TRUE,
                          recursive = TRUE)

    } else if (fsource == "init") {
      files <- list.files(path,
                          pattern = "__init__\\.py",
                          full.names = TRUE,
                          recursive = TRUE)

      files <- files[grepl("(?<!\\_builtin\\/)__init__\\.py$",
                           files,
                           perl = TRUE)]

    }

  # Stop part 2  ####

    # Check files
    if (length(files) == 0L) {
      stop("No files to process. This could happen by ",
           "incorrectly specified sources (e.g. init instead of model) ",
           "or by choosing folders that don't contain init or model files.")
    }

    # If output_dir is part of output_file
    if (!is.null(output_file) &&
        !is.null(output_dir) &&
        startsWith(x = output_file, prefix = output_dir)) {
      output_dir <- NULL
    }

    # If output_file is an absolute path, output_dir should not be used
    if (!is.null(output_file) &&
        grepl("^([A-Z]:|/)", output_file) &&
        !is.null(output_dir)) {
      if (output_dir_input != "getwd()") {
        stop("When using an absolute path for ",
             "\"output_file,\" \"output_dir\" should not be used.")
      } else {
        output_dir <- NULL
      }
    }

    # Check if app(s) exist(s)
    if (!is.null(app)) {
      if (length(app) == 1L) {
        if (!(app %in% basename(dirname(files)))) {
          stop("App \"", app, "\"is not in oTree code!")
        }
      } else if (length(app) > 1L) {
        for (app_i in seq_along(app)) {

          if (!(app[app_i] %in% basename(dirname(files)))) {
            stop("At least one app, \"",
                 app[app_i],
                 "\" is not in oTree code!")
          }
        }
      }
    }

    # Allowed output formats
    allowed_formats <- c("html_document", "pdf_document", "word_document",
                         "odt_document", "rtf_document", "md_document")

    if (is.null(output_format) ||
        length(output_format) > 1L ||
        !(output_format %in% allowed_formats)) {
      stop("Output format should be one of the following: ",
           paste0(allowed_formats, collapse = ", "), "!")
    }

  # Create variables, and environment  ####


    # Create a new environment
    env <- new.env()

    # Settings.py file
    env$settingspy <- TRUE
    env$settingslist <- c()

    # Create vector of variables without documentation
    env$nodocs <- character()

    # Create vector of variables with complex codes
    env$complexcons <- character()

    # Create vector of warnings
    env$warnings <- character()

    # Specify non-variable names
    # (parts in settings.py not used in the codebook)
    nonvariables <- c("ROOMS", "SESSION_CONFIGS", "INSTALLED_APPS",
                      "SESSION_CONFIG_DEFAULTS", "with", "from")

  # Background functions  ####

    # Paths and directories  ####

      # Define and create output dir
      if (!is.null(output_dir)) {

        # If dir is a relative path
        if (grepl("^\\.", x = output_dir)) {
          stop("Please don't use relative paths in output_dir!")
        }

        # If dir is not there
        if (!dir.exists(output_dir)) {

          stop("The directory ",
               output_dir,
               " does not exist yet. ",
               "Please create it manually before running this function.")
        }

        # Change Windows paths to paths that can be read by Ubuntu
        output_dir <- gsub("\\\\", "/", output_dir)
      }

    # Stop if  ####

    # Settings_replace
    if (length(user_settings) > 0L &&
        settings_replace != "user") {
      stop("settings_replace must be set to \"user\"",
           "if user_settings are not empty!")
    }

    # Check if only app or app_rm is specified
    if (!is.null(app) && !is.null(app_rm)) {
      stop("Please specify only \"app\" or \"app_rm!\"")
    }

  # Helping functions  ####
    process_lists <- function(variablevalue,
                         type = "s",
                         folder_name = folder_name,
                         env = env) {

      # Empty list
      if (variablevalue == "[]") {
        return(variablevalue)
      }

      # Three level list
      if (stringr::str_detect(string = unlist(variablevalue),
                              pattern = "^\\[{3,}")) {

        # Not sure if ever needed
        stop("This function does not support lists with more than two levels.")
      }

      # Two level lists vs. one level = vector
      if (startsWith(variablevalue,
                       "[[")) {

        # Replace first and last square brackets
        variablevalue <- sub(x = variablevalue,
                             pattern = "^\\[",
                             replacement = "")

        variablevalue <- sub(x = variablevalue,
                             pattern = "\\][^]]*$",
                             replacement = "")

        # Extract  each [ ... ] block
        if (stringr::str_detect(variablevalue, "\\[")) {
          variablevalue <- unlist(stringr::str_match_all(variablevalue,
                                                         pattern = "\\[.*?\\]"))
        }

        # Replace first and last square brackets
        variablevalue <- sub(x = variablevalue,
                             pattern = "^\\[",
                             replacement = "")

        variablevalue <- sub(x = variablevalue,
                             pattern = "\\][^]]*$",
                             replacement = "")

        returnlist <- list()

        for (variablevalue_i in seq_along(variablevalue)) {

          elem <- variablevalue[variablevalue_i]

          # Split the element into key and value
          parts <- stringr::str_split(elem, ",")[1L]

          # Clean and assign key and value
          parts <- clean_string(parts,
                                quotes = TRUE,
                                folder_name = folder_name)

          returnlist[[variablevalue_i]] <- parts

        }
      } else if (startsWith(variablevalue, "[")) {

        # make [..] to c(...)  ####
            variablevalue <- gsub(pattern = "\\[",
                                  replacement = "\\c(",
                                  x = variablevalue)

            variablevalue <- gsub(pattern = "\\]",
                                  replacement = "\\)",
                                  x = variablevalue)

        # Transform string of vector to normal vector  ####

        try({
          tmp <- eval(parse(text = variablevalue))
          if (!is.function(tmp)) {
            variablevalue <- tmp
          }
        }, silent = TRUE)

        returnlist <- variablevalue
      }

      if (length(returnlist) == 1L) {
        returnlist <- unlist(returnlist)
      }

      return(returnlist)
    }

    # Get vars from Constants or settings.py
    const_sett_vars <- function(matches,
                                current_class,
                                filevars,
                                normalspace,
                                folder_name,
                                env = env) {
      # This is called by process_settings and process_files

      # Get variable names  ####
      # Only those that are on the same indent are measured!
      pattern <- paste0("^\\s{",
                        normalspace,
                        "}[a-zA-Z_0-9]+ *(?=\\s*=)")

      # Vector of variable names
      all_cons_sett_vars <-
        unlist(regmatches(
          x = matches,   # Here still in vector!!!
          m = gregexpr(pattern = pattern,
                       text = matches,
                       perl = TRUE)))

      all_cons_sett_vars <- trimws(all_cons_sett_vars) # Strip spaces etc.

      # Put everything in one line
      matches <- collapse_and_clean_matches(matches)

      # Check if "with" occurs
      if (grepl(x = matches, pattern = "\\nwith")) {
        env$complexcons <-
          c(env$complexcons, paste0("> App", folder_name, "(",
                                   current_class, ") (with)\n"))
      }

      # Check if "read_csv" occurs
      if (grepl(x = matches, pattern = "read_csv")) {
        env$complexcons <-
          c(env$complexcons, paste0("> App ", folder_name, "(",
                                   current_class, ") (read_csv)\n"))
      }

      # Get everything until the second variable is mentioned and modify
      for (cons_sett_i in seq_along(all_cons_sett_vars)) {

        if (all_cons_sett_vars[cons_sett_i] != "with" &&  # Unnecessary, there is no = sign
            all_cons_sett_vars[cons_sett_i] != "from") {

          # Make pattern
          if (cons_sett_i < length(all_cons_sett_vars)) {

            pattern <- paste0(
              "(?<=\n\\b", all_cons_sett_vars[cons_sett_i], "\\b)",
              "[\\s\\S]*?",
              "(?=",
              "\\n\\b", all_cons_sett_vars[cons_sett_i + 1L], "\\b *=|",
              "\\nwith",
              ")")

          } else if (cons_sett_i == length(all_cons_sett_vars)) {
            pattern <- paste0("(?<=\n",
                              all_cons_sett_vars[cons_sett_i],
                              ")", "[\\s\\S\\\\n]*")
          }

          # Create variable
          if (!(all_cons_sett_vars[cons_sett_i] %in% nonvariables)) {

              # Create variable value for file list
              varval <- unlist(regmatches(
                x = matches,
                m = gregexpr(pattern = pattern,
                             text = matches,
                             perl = TRUE)))

              varval <- clean_string(varval,
                                      equal = TRUE,
                                      n = TRUE,
                                      quotes = FALSE,
                                      space = TRUE,
                                      brackets = FALSE,
                                      sbrackets = FALSE,
                                     folder_name = folder_name)

              # Deal with lists  ####
              if (startsWith(varval, "[")) {

                varval <- process_lists(varval)

                # Stop if list contains another list
                if (any(vapply(varval, is.list,
                               FUN.VALUE = logical(1L)))) {
                  # Not sure if ever needed
                  stop("This function does not support overly complex lists.")
                }
              }

              # Replace variable references within Constants/settings  ####
              # See if there are references to previous variables
              if (is.character(varval)) {

                for (j in seq(cons_sett_i)) {

                  if (j != cons_sett_i && any(grepl(pattern = paste0("\\b",
                                               all_cons_sett_vars[j],
                                               "\\b"),
                                          x = as.character(varval)))) {

                      if (length(
                        filevars[[current_class]][[all_cons_sett_vars[j]]]) ==
                        1L) {

                        varval <-
                          gsub(x = varval,
                               pattern = paste0("(?<!settings.)",
                                                all_cons_sett_vars[j]),
                               replacement = filevars[[current_class]][[
                                 all_cons_sett_vars[j]]],
                               perl = TRUE)

                      } else {

                        varval <-
                          gsub(x = varval,
                               pattern = paste0("(?<!settings.)",
                                                all_cons_sett_vars[j]),
                               replacement = paste0("c(",
                                 paste(filevars[[current_class]][[
                                   all_cons_sett_vars[j]]],
                                   collapse = ","),
                               ")"),
                               perl = TRUE)
                      }
                  }
                }
              }

              # If string containing a vector, make this a vector

              # Evaluate variable value
              try({
                tmp <- eval(parse(text = varval))
                if (!is.function(tmp)) {
                  varval <- unlist(tmp)
                }
              }, silent = TRUE)

              # Add variable to file list
              filevars[[current_class]][[all_cons_sett_vars[cons_sett_i]]] <- varval

          }
        }
      }

      # Return all settings or Constants variables
      return(filevars)
    }

    # Replace unmatched parentheses
    replace_unmatched_parentheses <- function(string,
                                              current_class = current_class,
                                              folder_name,
                                              env = env) {

      open <- stringr::str_count(string, pattern = "\\(")
      close <- stringr::str_count(string, pattern = "\\)")
      opensq <- stringr::str_count(string, pattern = "\\[")
      closesq <- stringr::str_count(string, pattern = "\\]")

      # Round brackets
      if (open == 0L &&
          close == 1L) {
        string <- gsub(x = string, pattern = "\\)", replacement = "")
      } else if (open == 1L && close == 0L) {
        string <- gsub(x = string, pattern = "\\(", replacement = "")
      } else if (!(open == close)) {
        # Do nothing yet
      } else if (open == close &&
                 open == 1L) {

        # Replace first and last character if they are both brackets
        if (startsWith(string, "(")  &&
            endsWith(string, ")")) {
          string <- substr(x = string,
                           start = 2L,
                           stop = nchar(string) - 1L)
        }

        # Replace first and last character if they are both brackets with backslashes
        if (startsWith(string, "\\(")  &&
            endsWith(string, "\\)")) {
          string <- substr(x = string,
                           start = 2L,
                           stop = nchar(string) - 1L)
        }
      }

      # Square brackets
      if (opensq == 0L && closesq == 1L) {
        string <- gsub(x = string, pattern = "\\]", replacement = "")
      } else if (opensq == 1L && closesq == 0L) {
        string <- gsub(x = string, pattern = "\\[", replacement = "")
      } else if (opensq != closesq) {
        env$complexcons <-
          c(env$complexcons, paste0("> App", folder_name, "(",
                                    current_class,
                                    ") (unmatched square brackets)\n"))

      } # Don't remove square brackets if they are first and last yet!

      return(string)
    }

    # Clean string
    clean_string <- function(string,
                             equal = TRUE,
                             n = TRUE,
                             space = TRUE,
                             quotes = TRUE,
                             brackets = TRUE,
                             sbrackets = TRUE,
                             lastcomma = TRUE,
                             current_class = current_class,
                             folder_name = folder_name) {

      # Save real quotes first
      string <- gsub(pattern = "\\\\\"",
                     replacement = "<<realquote>>",
                     x = string)

      # Remove = signs
      if (equal) {
        string <- gsub(pattern = "=",
                       replacement = "",
                       x = string)
      }

      # Remove line breaks
      if (n) {
        string <- gsub(pattern = "\\n",
                       replacement = "",
                       x = string)
      }

      # Remove quotes: " or \".
      # Don't remove ' or \' because those are essential parts of text
      # Will be done in the end
      if (quotes) {
        # Important! Must be done before any \ removals
        string <- gsub(pattern = "\\\"|\"",
                       replacement = "",
                       x = string)
      }

      # Trim white space
      if (space) {
        string <- trimws(string)
      }

      # Remove last comma in a string
      if (lastcomma) {
        string <- gsub(pattern = ",$",
                       replacement = "",
                       x = string)
      }

      # Ensure that brackets are processed last
      if (brackets && !is.na(string)) {
        string <- replace_unmatched_parentheses(string, folder_name)
      }

      # Remove two single quotes only if they are at the beginning and end of the string
      # Must be done at the end because they could also be used as apostrophes
      if (quotes) {

        # Documentation
        string <- gsub(pattern = "^'''(.*)'''$",
                       replacement = "\\1",  # Keep the content in the middle
                       x = string,
                       perl = TRUE)

        # Normal quotes
        string <- gsub(pattern = "^'(.*)'$",
                       replacement = "\\1",  # Keep the content in the middle
                       x = string,
                       perl = TRUE)
      }


      # Get real quotes back
      string <- gsub(pattern = "<<realquote>>",
                     replacement = "\"",
                     x = string)

      return(string)

    }

    # Function to split each element at the last comma
    # = to split last part of variable 1 from variable 2 name
    split_at_last_comma <- function(part) {

      f_split_parts <- stringr::str_split(part,
                                          ",(?=[^,]*$)",
                                          n = 2L)[[1L]]

      if (length(f_split_parts) > 1L) {

        return(c(stringr::str_trim(f_split_parts[1L]),
                 stringr::str_trim(f_split_parts[2L])))
      } else {
        return(part)
      }
    }

    # Remove all line comments
    remove_line_comments <- function(file_content) {
      processed_lines <- character()

      for (line in file_content) {
        # Remove single-line comments
        line <- sub("#.*", "", line)

        # Add the processed line to the result
        processed_lines <- c(processed_lines, line)
        # Multiline comments are not processed
      }
      return(processed_lines)
    }

    # Get line numbers
    get_doc_lines <- function(file_content) {
      inside_doc <- FALSE
      for (line_nr in seq_along(file_content)) {
        if (startsWith(x = file_content[line_nr],
                       prefix = "doc")) {

          inside_doc <- TRUE
          start <- line_nr

        } else if (inside_doc &&
                   (startsWith(file_content[line_nr], "class") ||
                    startsWith(file_content[line_nr], "def"))) {

          end <- line_nr - 1L
          return(c(start, end))
        }

        if (line_nr == length(file_content)) {

          if (inside_doc) {
            # This should not happen because doc is always at
            # the beginning of a page! But its still there in case there is a
            # messy code

            end <- line_nr
            return(c(start, end))
          } else {
            return(NULL)
          }
        }
      }
    }

    # Get class lines
    get_class_lines <- function(file_content, class) {

      inside_class <- FALSE
      if (grepl(pattern = "Constants|C", x = class)) {

        class <- "Constants\\s*\\(|C\\s*\\("
      }

      for (line_nr in seq_along(file_content)) {
        if (stringr::str_detect(file_content[line_nr],
                                paste0("^class ", class))) {

          inside_class <- TRUE
          start <- line_nr

        } else if (inside_class &&
                   startsWith(x = file_content[line_nr],
                              prefix = "class")) {

          end <- line_nr - 1L
          return(c(start, end))
        }

        if (line_nr == length(file_content)) {
          end <- line_nr
          return(c(start, end))
        }
      }
    }

    # Clean matches
    collapse_and_clean_matches <- function(matches) {

      matches <- paste(matches, collapse = "\n") # Put all matches in one string

      # Remove all spaces at the beginning and after each \n
      matches <- gsub(pattern = "\n *",
                      replacement = "\n",
                      x = matches)

      # To make the lookbehind easier down there
      matches <- gsub(pattern = "^\\s*",
                      replacement = "\n",
                      x = matches)

      return(matches)
    }

    # Replace constants values references by actual values
    cons_replace <- function(string, filevars, folder_name, env = env) {
      # Replace Constants with the constants variable.

      pattern <- "(Constants\\.[^ ]+)|(C\\.[^ ]+)"

      # Find all references to constants
      consmatches <- stringr::str_extract_all(string, pattern)
      consmatches <- unlist(consmatches)

      if (!is.null(consmatches) && length(consmatches) > 0L) {

        for (fullvarpattern in consmatches) {

          if (!is.na(fullvarpattern)) {

            var <- sub(pattern = "(Constants\\.)|(C\\.)",
                       replacement = "",
                       x = fullvarpattern)

            myreplacement <- filevars[["Constants"]][[var]]

            if (!is.null(myreplacement)) {
              # First remove possible preceding +
              # (in Pyhon, a + adds strings together)
              string <- sub(pattern = paste0("\\+\\s*", fullvarpattern),
                            replacement = fullvarpattern,
                            x = string)

              string <- sub(pattern = paste0(fullvarpattern, "\\s*", "\\+"),
                            replacement = fullvarpattern,
                            x = string)

              # Replace the value
              string <- sub(pattern = fullvarpattern,
                            replacement = myreplacement,
                            x = string)
            } else {
              env$warnings <-
                c(env$warnings,
                  paste0("Variable ", fullvarpattern,
                     " in folder ", folder_name,
                     " is not in Constants and cannot be replaced!",
                     " Check your code before continuing making the",
                     " codebook and running the experiment!"))
            }
          }
        }
      }
      return(string)
    }

    # Replace settings values references by actual values
    settings_replace_f <- function(mystring,
                                 folder_name = NULL,  # Folder name
                                 combined_codebook = combined_codebook,
                                 user_settings = user_settings,
                                 settings_replace = settings_replace,
                                 e_variable = NULL,
                                 e_key = NULL) {

      pattern <- "settings\\.[_a-zA-Z0-9]+"

      # First check for sub-lists
      # Not necessary here because they are already sub-lists!

      # Check if the string refers to a settings variable
      settings_matches <-
        unlist(stringr::str_extract_all(mystring, pattern))

      if (!is.null(settings_matches) &&
          length(settings_matches) > 0L) {

        for (fullvarpattern in settings_matches) {

          myreplacement <- NULL

          if (!is.na(fullvarpattern)) {

            # Remove "settings." part of the variable name
            settings_var <- sub(pattern = "settings\\.",
                                replacement = "",
                                x = fullvarpattern)

            # Remove any brackets from the variable name
            settings_var <- gsub(pattern = "\\(",
                                 replacement = "",
                                 x = settings_var)

            settings_var <- gsub(pattern = "\\)",
                                 replacement = "",
                                 x = settings_var)

            if (!is.null(settings_replace) &&
                settings_replace == "global") {

              if (is.null(combined_codebook[["settings"]][[settings_var]])) {

                env$settingslist <- c(env$settingslist,
                                      paste0("> Folder \"", folder_name,
                                             "\", variable: \"", e_variable,
                                      "\", reference: \"settings.", settings_var,
                                      "\".\n"))
              } else {
                myreplacement <- combined_codebook[["settings"]][[settings_var]]
              }
            } else if (!is.null(settings_replace) &&
                       settings_replace == "user"){

              if (!is.null(user_settings) &&
                  settings_var %in% names(user_settings)) {
                myreplacement <- user_settings[[settings_var]]

              } else {

                stop("Variable \"", settings_var, "\" in app \"", folder_name,
                     "\" (and maybe others) is not in user_settings!")
              }
            }

            # Replace variable within the whole string
            if (!is.null(myreplacement)) {

              if (length(myreplacement) == 1L) {

                # Replace single value
                mystring <- sub(pattern = fullvarpattern,
                                replacement = myreplacement,
                                x = mystring)

              } else  {
                # If all numeric

                if (suppressWarnings(anyNA(as.integer(myreplacement)))) {
                  mystring <-
                    gsub(x = mystring,
                         pattern = fullvarpattern,
                         replacement = paste0("c(\"",
                                              paste(myreplacement,
                                                    collapse = "\",\""),
                                              "\")"),
                         perl = TRUE)

                } else {

                  # Replace with a vector
                  mystring <-
                    gsub(x = mystring,
                         pattern = fullvarpattern,
                         replacement = paste0("c(",
                                              paste(myreplacement,
                                                    collapse = ","),
                                              ")"),
                         perl = TRUE)

                }
              }
            } else {
              # Do nothing because this is dealt with before
            }
          }
        }
      }

      # If numeric, then evaluate
      if (length(mystring) == 1L) {
        try({

          tmp <- eval(parse(text = mystring))
          if (!is.null(tmp) && !is.function(tmp)) {
            mystring <- tmp

          }
        }, silent = TRUE)

      } else {

        # If single elements contain calculations
        for (mystring_i in seq_along(mystring)) {

          try({
            tmp <- eval(parse(text = mystring[mystring_i]))
            if (!is.null(tmp) && !is.function(tmp)) {
              mystring[mystring_i] <- tmp
            }
          }, silent = TRUE)
        }
      }

      # Might return an integer but that will be a string
      # as soon as it replaces the old variable!
      return(mystring)
    }

  # Functions to process a file  ####
  process_settingspy <- function(file_path, env) {

    file_path <- file.path(file_path, "settings.py")

    folder_name <- basename(dirname(file_path))
    file_content <- readLines(file_path, warn = FALSE)
    file_content <- remove_line_comments(file_content)
    settings <- list()

    # Get variables
    filevars <- const_sett_vars(matches = file_content,
                                current_class = "settings",
                                filevars = settings,
                                normalspace = 0L,
                                folder_name = folder_name,
                                env = env)

    return(filevars)
  }

  process_file <- function(file_path,
                           folder_name,
                           combined_codebook = combined_codebook,
                           env = env) {

    file_content <- readLines(file_path, warn = FALSE)

    # Sometimes, init.py only has 1 line in old oTree
    if (length(file_content) <= 2L) {
      stop("At least one of your init-files is empty. ",
           "Try using argument \"fsource = model\".")
    }

    file_content <- remove_line_comments(file_content)
    doc_lines <- get_doc_lines(file = file_content)
    constants_lines <- get_class_lines(file = file_content, "Constants")

    group_lines <- get_class_lines(file = file_content, "Group")
    player_lines <- get_class_lines(file = file_content, "Player")
    current_class <- ""
    filevars <- list()

    for (line_nr in seq_along(file_content)) {
      # The first time the class is mentioned
      # the class is set for the next lines

      # App documentation  ####
      if (!is.null(doc_lines) &&
          line_nr == doc_lines[[1L]]) {

        matches <- file_content[(line_nr):doc_lines[[2L]]]

        matches <- paste(matches, collapse = " ")

        matches <- gsub(x = matches,
                        pattern = "^doc",
                        replacement = "")

        matches <- clean_string(matches,
                                folder_name = folder_name)

        filevars[["doc"]] <- matches
      }

      # Constants  ####
      if (line_nr == constants_lines[1L]) {

        current_class <- "Constants"
        matches <- file_content[(line_nr + 1L):constants_lines[2L]]

        # Count the spaces at the beginning of each line
        cons_normalspace <- gregexpr("^\\s+", matches[1L])

        cons_normalspace <- attr(cons_normalspace[[1L]],
                                 "match.length")

        # Get variables
        filevars <- const_sett_vars(matches = matches,
                                    current_class = current_class,
                                    filevars = filevars,
                                    normalspace = cons_normalspace,
                                    folder_name = folder_name,
                                    env = env)

        # Clean constants
        for (cons_i in seq_along(filevars[["Constants"]])) {

          # If there is a second level
          if ((length(filevars[["Constants"]][[cons_i]])) > 1L) {

            for (cons_j in seq_along(filevars[["Constants"]][[cons_i]])) {

              # Delete print()
              filevars[["Constants"]][[cons_i]][[cons_j]] <-
                gsub(
                  x = filevars[["Constants"]][[cons_i]][[cons_j]],
                  pattern = "print\\(.*\\)",
                  replacement = "")

              # Replace all references to the settings with the actual variables
              filevars[["Constants"]][[cons_i]][[cons_j]] <-
                settings_replace_f(
                  mystring = filevars[["Constants"]][[cons_i]][[cons_j]],
                  folder_name = folder_name,
                  combined_codebook = combined_codebook,
                  user_settings = user_settings,
                  settings_replace = settings_replace,
                  e_variable = names(filevars[["Constants"]][cons_i])[[cons_j]])

            }

          } else {
            # Replace all references to the settings with the actual variables
            filevars[["Constants"]][[cons_i]] <-
              settings_replace_f(
                mystring = filevars[["Constants"]][[cons_i]],
                folder_name = folder_name,
                combined_codebook = combined_codebook,
                user_settings = user_settings,
                settings_replace = settings_replace,
                e_variable = names(filevars[["Constants"]][cons_i]))

          }
        }
      }

      # Player and Group  ####
      if (line_nr == player_lines[[1L]] ||
          line_nr == group_lines[[1L]]) {

        # Get all class text
        if (line_nr == player_lines[1L]) {
          matches <- file_content[(line_nr + 1L):player_lines[2L]]
          current_class <- "Player"
        } else {
          matches <- file_content[(line_nr + 1L):group_lines[2L]]
          current_class <- "Group"
        }

        matches <- collapse_and_clean_matches(matches)

        # If there is no class info
        if (
          stringr::str_detect(trimws(matches[1L]),
                              "^pass$")) {
          filevars[[current_class]] <- "Pass"
          next
        }

        # Get variables  ####

        # Variable names  ####
        variables <- unlist(regmatches(
          x = matches,
          m = gregexpr(pattern = "\n[a-zA-Z_0-9]+ *(?= *= *models)",
                       text = matches,
                       perl = TRUE)))

        # Strip spaces etc.
        variables <- trimws(variables)

        # Variable info  ####
        for (variables_i in seq_along(variables)) {

          variable <- variables[variables_i]

          if (variables_i < length(variables)) {
            pattern <- paste0("(?<=\n", variables[variables_i], ")",
                              " *=[\\s\\S]*",
                              "(?=\n", variables[variables_i + 1L], " *=)")

          } else {
            # Last variable until the end
            pattern <- paste0("(?<=\n",
                              variables[variables_i],
                              ")", " *=[\\s\\S\\\\n]*"
            )
          }

          # Create variable in filevars
          filevars[[current_class]][[variable]] <- list()

          # Varmatches
          varmatches <- unlist(regmatches(
            x = matches,
            m = gregexpr(pattern = pattern,
                         text = matches,
                         perl = TRUE)))

          # Remove possible subsequent functions from matches  ####
          varmatches <- sub(
            pattern = "(\\ndef )[\\s\\S\\\\n]*",
            replacement = "",
            x = varmatches,
            perl = TRUE
          )

          # Remove possible subsequent if statements from matches  ####
          varmatches <- sub(
            pattern = "(\\nif )[\\s\\S\\\\n]*",
            replacement = "",
            x = varmatches,
            perl = TRUE
          )

          # Get variable information  ####
          # Get field
          field <-
            stringr::str_extract(varmatches, "(?<=models\\.)[^(]+")

          # Remove field from matches + last )
          varmatches <- sub(
            pattern = paste0(" *= *models\\.", field),
            replacement = "",
            x = varmatches,
            perl = TRUE
          ) # First bracket stays but this is okay and stripped later.

          if (grepl(x = varmatches,
                    pattern = "\\)[\n ]*$",
                    perl = TRUE)) {
            varmatches <- sub(
              pattern = "\\)$",
              replacement = "",
              x = varmatches,
              perl = TRUE
            )
          } else {  # "Else" is just for development / should not happen
            stop("Internal gmoTree error!")
          }

          # If there are no arguments
          if (stringr::str_detect(string = varmatches,
                                  pattern = "[a-zA-Z][^\\n]",
                                  negate = TRUE)) {

            varmatches <- "noargs = TRUE"
          } else {
            varmatches <- paste("noargs = FALSE, ",
                                varmatches, sep = " ")
          }

          # If documentation does not start with doc, add it
          # Not sure if ever needed
          if (stringr::str_detect(varmatches, "^ *\"\"\"|^ *'''")) {
            varmatches <- paste("doc = ", varmatches)
          }

          # Variable information   ####
          # First split its content at every = sign  ####
          parts <- stringr::str_split(stringr::str_trim(varmatches),
                                      " *= *")[[1L]]

          # Now the value of one variable is together with
          # the variable name of the next variable
          # Apply split_at_last_comma to each element
          # except the first and last

          if (length(parts) > 2) {
            split_parts <- unlist(lapply(parts[2L:(length(parts) - 1L)],
                                         split_at_last_comma))
          } else {
            split_parts <- c()
          }

          # Sometimes, there is a comma at the end
          parts[length(parts)] <- sub(pattern = ",\\\n)$",
                                      replacement = "",
                                      x = parts[length(parts)])

          # Combine
          if (length(parts) == 2L) {
            parts <- c(parts[1L], parts[2L])
          } else if (length(parts) > 2L) {
            parts <- c(parts[1L],
                       split_parts,
                       parts[length(parts)])
          } else {
            stop("An unexpected error occurred. ",
                 "Please contact the maintainer with details.")
          }

          # Make key value frame  ####
          # Create an empty list to store your kv_frame
          kv_frame <- data.frame(key = c(),
                                 value = c())

          # Iterate over the vector and fill the kv_frame
          for (j in seq(1L, length(parts), by = 2L)) {
            key <- parts[j]
            value <- parts[j + 1L]
            kv_frame <- rbind(kv_frame,
                              data.frame(key = key,
                                         value = value))
          }

          # Last strip
          kv_frame$key <- gsub(x = kv_frame$key,
                               "\\n",
                               "")

          kv_frame$key <-
            sapply(kv_frame$key,
                   clean_string,
                   quotes = TRUE,
                   folder_name = folder_name)

          # Choices need to be specified  #####
          if ("choices" %in% kv_frame$key) {

            text <- kv_frame$value[kv_frame$key == "choices"]

            # Remove trailing and leading whitespace
            text <- trimws(text)

            # In case the kv_frame works with square brackets
            numbrack <- length(unlist(gregexpr(pattern = "\\[",
                                               text = text)))

            if (numbrack > 1L) {  # If key - value pairs

              # Replace first and last square brackets
              text <- sub(x = text,
                          pattern = "^\\[",
                          replacement = "")

              text <- sub(x = text,
                          pattern = "\\][^]]*$",
                          replacement = "")

              # Extract  each [ ... ] block
              text <- gsub(pattern = "\n",
                           replacement  = "",
                           x = text,
                           perl = TRUE)

              text <-
                unlist(stringr::str_match_all(text,
                                              pattern = "\\[.*?\\]"))

              # Combine into a single data frame
              # (not dict because values can appear several times)
              choices <- data.frame(
                key <- c(),
                value <- c())

              for (elem in text) {
                # Split the element into key and value
                parts <- stringr::str_split(string = elem,
                                            pattern = ",",
                                            n = 2L)[[1L]]

                # Clean and assign key and value
                key <- clean_string(parts[1L],
                                    quotes = TRUE,
                                    folder_name = folder_name)

                value <- clean_string(parts[2L],
                                      quotes = TRUE,
                                      folder_name = folder_name)

                value <- cons_replace(value, filevars,
                                      folder_name, env = env)

                value <- settings_replace_f(
                  mystring = value,
                  folder_name = folder_name,
                  combined_codebook = combined_codebook,
                  user_settings = user_settings,
                  settings_replace = settings_replace,
                  e_variable = variable)

                # Return key-value pair
                choices <- rbind(choices,
                                 data.frame(
                                   key = key,
                                   value = value))

              }

            } else if (numbrack == 1L) {
              # If not key-value pairs. E.g. choices=[1, 2, 3]

              # Replace first and last square brackets
              text <- sub(x = text,
                          pattern = "^\\[",
                          replacement = "")

              text <- sub(x = text,
                          pattern = "\\][^]]*$",
                          replacement = "")

              # Combine into a single data frame
              # (not dict because values can appear several times)
              # Important: Check Caution 2!

              choices <-
                stringr::str_split(text, "(?<!\\\\),", n = Inf)[[1L]]

              # Make manually constructed commas normal again
              choices <- gsub(x = choices,
                              pattern = "\\\\,",
                              replacement = ",",
                              perl = TRUE)

              # Clean choices (also: makes vector to named lists)
              choices <- sapply(choices,
                                clean_string,
                                quotes = TRUE,
                                folder_name = folder_name)

              choices <- cons_replace(choices, filevars,
                                      folder_name, env = env)

              choices <-
                settings_replace_f(mystring = choices,
                                   folder_name = folder_name,
                                   combined_codebook = combined_codebook,
                                   user_settings = user_settings,
                                   settings_replace = settings_replace,
                                   e_variable = variable)

              choices <- as.vector(choices)

            }

            # Remove it from kv_frame
            kv_frame <- kv_frame[kv_frame$key != "choices", ]
            text <- NULL
          }

          # Prettify variable information  ####
          # kv_frame$key was already cleaned before

          kv_frame$value <-
            sapply(kv_frame$value,
                   clean_string,
                   quotes = TRUE,
                   folder_name = folder_name)

          # Replace constant variable references
          # with actual constant variables
          kv_frame$value <- cons_replace(kv_frame$value,
                                         filevars,
                                         folder_name,
                                         env = env)

          # Replace settings variable references with
          # actual settings variables
          for (k in seq_along(kv_frame$value)) {
            kv_frame$value[k] <-
              settings_replace_f(mystring = kv_frame$value[k],
                                 folder_name = folder_name,
                                 combined_codebook = combined_codebook,
                                 user_settings = user_settings,
                                 settings_replace = settings_replace,
                                 e_variable = variable,
                                 e_key = kv_frame$key[k])

          }

          # Get everything into the variable

          # Transform left-over variable information to dictionary
          filevars[[current_class]][[variable]] <-
            stats::setNames(as.list(kv_frame$value), kv_frame$key)

          if (exists("choices")) {
            filevars[[current_class]][[variable]][["choices"]] <- choices
          }

          filevars[[current_class]][[variable]][["field"]] <- field

          # If there is no documentation, add this to info
          if (!("doc" %in% names(filevars[[current_class]][[variable]])) &&
              !("label" %in% names(filevars[[current_class]][[variable]])) &&
              !("verbose_name" %in% names(filevars[[current_class]][[variable]]))) {
            env$nodocs <- c(env$nodocs,
                            paste0(folder_name, "$",
                                   current_class, "$", variable))
          }

          # Delete kv_frame
          kv_frame <- NULL
          choices <- NULL

        }
      }

    }
    return(filevars)

  }

  # Function to process a directory  ####
  process_directory <- function(path,
                                combined_codebook,
                                files = files,
                                settings_replace = settings_replace,
                                app = app,
                                app_rm = app_rm,
                                env = env) {

    # Files on highest level
    settingsfiles <- list.files(path,
                              pattern = "settings\\.py",
                              full.names = TRUE,
                              recursive = FALSE)

    if (length(settingsfiles) == 1L &&
        !is.null(settings_replace) &&
        settings_replace == "global") {

      combined_codebook <- process_settingspy(file_path = path)
    } else if (length(settingsfiles) == 0) {
        env$settingspy <- FALSE
    }

    for (file_path in files) {
      folder_name <- basename(dirname(file_path))

      if (is.null(app_rm) && is.null(app)) {

        combined_codebook[[folder_name]] <-
          process_file(file_path = file_path,
                       folder_name = folder_name,
                       combined_codebook = combined_codebook,
                       env = env)

      } else if (!is.null(app_rm) && !(folder_name %in% app_rm)) {
          combined_codebook[[folder_name]] <-
            process_file(file_path,
                         folder_name,
                         combined_codebook = combined_codebook,
                         env = env)

      } else if (!is.null(app) &&
                 folder_name %in% app) {
          combined_codebook[[folder_name]] <-
            process_file(file_path,
                         folder_name,
                         combined_codebook = combined_codebook,
                         env = env)

      }
    }
    return(combined_codebook)
  }

  # Run  ####
  combined_codebook <- list(user_settings = list())
  combined_codebook <- process_directory(path,
                                         combined_codebook,
                                         files = files,
                                         settings_replace = settings_replace,
                                         app = app,
                                         app_rm = app_rm,
                                         env = env)

  # Change output file  ####

      # Add file path
      if (!is.null(output_dir)) {

        output_file <- file.path(output_dir, output_file)
      } else {
        # Check if output_file contains current path
        dname <- dirname(output_file)
        if (dname != "." && !dir.exists(dname)) {
          output_file <- file.path(getwd(),
                                   output_file)
        }
      }


  # Adjust settings  ####
  if ("settings" %in% names(combined_codebook)) {
    combined_codebook[["settings"]][nonvariables] <- NULL
  }

  # Sort apps in codebook  ####
  if (!is.null(sort)) {
    sort <- c("settings", sort)

    if (
      length(sort) == length(names(combined_codebook)) &&
      setequal(sort, names(combined_codebook))) {

      combined_codebook <- combined_codebook[sort]

    } else {
      if (length(sort[!(sort %in% names(combined_codebook))]) > 0L) {
        p1 <- paste0("\n\nSort elements not in apps are: ",
              paste(sort[!(sort %in% names(combined_codebook))],
                    collapse = ", "))
      } else {
        p1 <- ""
      }

      if (length(names(combined_codebook)[!(names(combined_codebook) %in%
                                            sort)]) > 0L) {
        p2 <-
          paste0("\n\nApps not in sort are: ",
                paste(names(combined_codebook)[!(names(combined_codebook) %in%
                                                   sort)],
                      collapse = ", "))
      } else {
        p2 <- ""
      }

      env$warnings <-
        c(env$warnings,
          paste0("Sort apps are not equal to all apps. Therefore, ",
              "sort is not applied. ", p1, p2))
    }
  }

  # If file  ####
      if (output == "file" || output == "both") {

        # If other files already have this name   ####
        nr_suffix <- 0L

        # Output extension as in output_format
        output_form_ext <- sub(pattern = "_.*$",
                               replacement = "",
                               x = output_format)

        output_form_ext[output_form_ext == "word"] <- "docx"
        output_form_ext[output_form_ext == "latex"] <- "tex"

        # Check if file extension is already in file name (strip if yes)
        output_file <- sub(pattern = paste0("\\.",
                                            output_form_ext,
                                            "$"),
                           replacement = "",
                           x = output_file)

        # Check for non-fitting file extensions
        if (!(tolower(tools::file_ext(output_file)) == "" ||
              tolower(tools::file_ext(output_file)) == tolower(output_form_ext)
        )) {
          stop("You are not allowed to use file extensions in the ",
               "output_file that do not correspond to the output format! ",
               "Your output_file extension is ",
               tools::file_ext(output_file),
               ". The extension according to your output_format should be ",
               output_form_ext, ".")
        }

        # Define dictionary that has to be checked
        checkdir <- dirname(output_file)

        # Check if there are files with the same name in the folder
        nr_doc_same <- sum(
          grepl(pattern = paste0("^", basename(output_file),
                                 "[_\\d]*\\.", output_form_ext),
                x = list.files(checkdir),
                perl = TRUE))

        # If yes, add number to file
        if (nr_doc_same > 0L) {
          nr_suffix <- nr_doc_same + 1L
          output_file <- paste0(output_file, "_", nr_suffix)
        }

        # Make parameters  ####
        params2 <- list(
          app_doc = app_doc,
          preamb = preamb,
          include_cons = include_cons,
          title = title,
          date = date,
          subtitle = subtitle,
          encoding = encoding)

        if (!is.null(params)) {

          if (is.list(params)) {

            params <- utils::modifyList(params2, params)

          } else {
            stop("params must be a list!")
          }
        } else {
          params <- params2
        }

        if (!is.null(params[["date"]]) && params[["date"]] == "today") {
          params[["date"]] <- format(Sys.time(), "%d %B %Y")
        }

        # Make output  ####
        tryCatch({

          # Specify output_format
          if (output_format == "pdf_document") {
            output_format <- rmarkdown::pdf_document(latex_engine = "xelatex")
          }

          # Don't use output_dir here,
          # because that's already included in file name!
          created_file <- rmarkdown::render(
            input = system.file("rmd", "codebook.Rmd", package = "gmoTree"),
            output_format = output_format,
            output_file = output_file,
            params = params,
            clean  = TRUE # Encoding is ignored here! Always UTF-8
          )

          # Open
          created_file <- normalizePath(created_file)

          if (output_open) {
            utils::browseURL(created_file)
          }
          message("File saved in ", created_file)

        }, error = function(e) {
          message("Error in rmarkdown::render: ", e$message)
        })
  }

  # Message: Variables with no documentation info  ####
      if (!is.null(doc_info) &&
          !is.na(doc_info) &&
          doc_info &&
          !(length(env$nodocs) == 0L)) {

          message(
            "Variables without documentation, label, or verbose name:\n",
            paste0("> ", env$nodocs, collapse = "\n"))
      }

  # Last check if there is complex code in the variables and return vector ####
      # Function to recursively check for the string "float"
      # around variable values and return paths
      float_check_paths <- function(codebook, path = "") {

        # List to collect paths
        collected_paths <- list()

        # If the element is a list, recurse deeper
        if (is.list(codebook)) {
          for (name in names(codebook)) {
            # Recursively collect apps and variable names
            deeper_paths <-
              float_check_paths(codebook[[name]],
                                paste0(path, "$", name))
            collected_paths <- c(collected_paths, deeper_paths)
          }
        } else {

          # Check if the element is a character string containing "float"
          if (length(codebook) == 1L &&
              is.character(codebook) &&
              grepl("float(?!Field)",
                    codebook,
                    ignore.case = TRUE,
                    perl = TRUE)) {
            # Add the current path to the list if "float" is found
            collected_paths <- c(collected_paths, path)
          }
        }

        return(collected_paths)
      }

      complex2 <- float_check_paths(codebook = combined_codebook)
      complex2 <- unlist(complex2)

      if (length(complex2) > 0L) {
      complex2 <- paste(">", complex2, "(float)\n")
      }

      env$complexcons <- c(env$complexcons, complex2)

      # Show warning if there is complex code in Constants, Player, Group or settings
      if (length(env$complexcons) > 0L) {
        env$warnings <-
          c(env$warnings,
            paste0("Some variables or code parts contain code that is too complex for this function. ",
                "Hence, this function might have overseen ",
                "important variables and references to them. ",
                "Check the output carefully! Found in:\n",
                paste(env$complexcons, collapse = "")))
      }

  # Return warnings  ####
    if (length(env$settingslist) > 0) {

      if (env$settingspy == TRUE) {
        env$warnings <-
          c(env$warnings,
            paste0("The following settings variable/s is/are not in settings and ",
                "cannot be replaced:\n",
                env$settingslist))
      } else {
        env$warnings <-
          c(env$warnings, paste0("There is no settings.py in your path! ",
                "The following settings variable/s is/are not in settings and ",
                "cannot be replaced:\n ",
                env$settingslist))
      }

    }

    if (length(env$warnings) > 0) {
      env$warnings <- paste(env$warnings, collapse = "\n\n")
      warning(env$warnings)
    }
  # Return list  ####
  if (output == "list" || output == "both") {
    return(combined_codebook)
  }
}
