#' Add decorations to spectrum plot (private)
#'
#' Add decorations to plots generated by the plot() methods defined in this
#' package. It collects code that is common to plot methods for different types
#' of spectra but as it may change in the future it is not exported.
#'
#' @param w.band waveband object or list of waveband objects
#' @param ymax,ymin,xmax,xmin numeric
#' @param annotations character vector
#' @param span numeric
#' @param label.qty character
#' @param summary.label character
#' @param text.size numeric
#' @param label.color color definition or name
#' @param pos.shift numeric
#' @param na.rm logical
#'
#' @keywords internal
#'
#' @return A list of ggplot "components" that can be added to a ggplot object
#'   with operator "+". The length of the list depends on the value of argument
#'   \code{annotations}.
#'
decoration <- function(w.band,
                       y.max,
                       y.min,
                       x.max,
                       x.min,
                       annotations,
                       span = NULL,
                       label.qty,
                       label.mult = 1,
                       summary.label,
                       unit.out = NULL,
                       time.unit = NULL,
                       text.size = 2.5,
                       label.color = NULL,
                       pos.shift = 0,
                       na.rm = TRUE) {
  if (grepl(".pc", label.qty, fixed = TRUE)) {
    label.mult = 100
    label.qty <- sub(".pc", "", label.qty, fixed = TRUE)
  }
  if (!"summaries" %in% annotations) {
    label.qty <- "none"
  }
  stat_wb_summary <- switch(label.qty,
                            total = stat_wb_total,
                            mean = stat_wb_mean,
                            average = stat_wb_mean,
                            irrad = stat_wb_irrad,
                            sirrad = stat_wb_sirrad,
                            contribution = stat_wb_contribution,
                            relative = stat_wb_relative,
                            none = stat_wb_label,
                            function(...) {NA_real_},
                            na.rm = na.rm)
  z <- list()
  if ("peaks" %in% annotations) {
    z <- c(z, stat_peaks(span = span, label.fmt = "%.4g",
                         ignore_threshold = 0.02, color = "red",
                         geom = "text", vjust = -0.5, size = text.size,
                         na.rm = na.rm),
           stat_peaks(color = "red",
                      span = span,
                      shape = 16,
                      ignore_threshold = 0.02))
  }
  if ("peak.labels" %in% annotations) {
    z <- c(z, stat_label_peaks(aes_(color = ~..BW.color..),
                         span = span, label.fmt = "%.4g",
                         ignore_threshold = 0.02,
                         geom = "label_repel",
                         segment.colour = "red",
                         min.segment.length = unit(0.02, "lines"),
                         size = text.size,
                         box.padding = unit(0.02, "lines"),
                         direction = "y",
                         vjust = 1,
                         na.rm = na.rm),
           stat_peaks(color = "red",
                      span = span,
                      shape = 16,
                      ignore_threshold = 0.02))
  }
  if ("valleys" %in% annotations) {
    z <- c(z, stat_valleys(span = span, label.fmt = "%.4g",
                           ignore_threshold = 0.02, color = "blue",
                           geom = "text", vjust = +1.2, size = text.size,
                           na.rm = na.rm),
           stat_valleys(color = "blue",
                        span = span,
                        shape = 16,
                        ignore_threshold = 0.02))
  }
  if ("valley.labels" %in% annotations) {
    z <- c(z, stat_label_valleys(span = span, label.fmt = "%.4g",
                                 ignore_threshold = 0.02,
                                 geom = "label_repel",
                                 segment.colour = "blue",
                                 min.segment.length = unit(0.02, "lines"),
                                 size = text.size,
                                 box.padding = unit(0.02, "lines"),
                                 direction = "y",
                                 vjust = 0,
                                 na.rm = na.rm),
           stat_valleys(color = "blue",
                        span = span,
                        shape = 16,
                        ignore_threshold = 0.02))
  }
  if ("colour.guide" %in% annotations) {
    z <- c(z, stat_wl_strip(ymax = y.max * 1.26, ymin = y.max * 1.22,
                            na.rm = na.rm, color = NA))
  }
  if ("boxes" %in% annotations) {
    z <- c(z, stat_wl_strip(w.band = w.band,
                            ymax = y.max * 1.20,
                            ymin = y.max * 1.08,
                            color = "white",
                            linetype = "solid",
                            na.rm = na.rm
    ))
  } else {
    label.color <- if (is.null(label.color)) {
      label.color <- "black"
    }
  }
  if ("segments" %in% annotations) {
    z <- c(z, stat_wl_strip(w.band = w.band,
                            ymax = y.max * 1.10,
                            ymin = y.max * 1.07,
                            color = "white",
                            linetype = "solid",
                            na.rm = na.rm
    ))
    label.color <- "black"
    pos.shift <- 0.01
  }
  if ("labels" %in% annotations || "summaries" %in% annotations) {

    if ("labels" %in% annotations && "summaries" %in% annotations) {
      mapping <- aes_(label = quote(paste(..wb.name.., ..y.label.., sep = "\n")),
                      color = ~..BW.color..)
    } else if ("labels" %in% annotations) {
      mapping <- aes_(label = ~..wb.name.., color = ~..BW.color..)
    } else if ("summaries" %in% annotations) {
      mapping <- aes_(label = ~..y.label.., color = ~..BW.color..)
    }

    if ("summaries" %in% annotations) {
      if (label.qty %in% c("irrad", "sirrad")) {
        if (is.null(label.color)) {
          z <- c(z, stat_wb_summary(geom = "text",
                                    unit.in = unit.out,
                                    time.unit = time.unit,
                                    w.band = w.band,
                                    label.mult = label.mult,
                                    ypos.fixed = y.max * 1.143 + pos.shift,
                                    mapping = mapping,
                                    size = text.size,
                                    na.rm = na.rm))
        } else {
          z <- c(z, stat_wb_summary(geom = "text",
                                    unit.in = unit.out,
                                    time.unit = time.unit,
                                    w.band = w.band,
                                    label.mult = label.mult,
                                    ypos.fixed = y.max * 1.143 + pos.shift,
                                    color = label.color,
                                    mapping = mapping,
                                    size = text.size,
                                    na.rm = na.rm))
        }
      } else {
        if (is.null(label.color)) {
          z <- c(z, stat_wb_summary(geom = "text",
                                    w.band = w.band,
                                    label.mult = label.mult,
                                    ypos.fixed = y.max * 1.143 + pos.shift,
                                    mapping = mapping,
                                    size = text.size,
                                    na.rm = na.rm))
        } else {
          z <- c(z, stat_wb_summary(geom = "text",
                                    w.band = w.band,
                                    label.mult = label.mult,
                                    ypos.fixed = y.max * 1.143 + pos.shift,
                                    color = label.color,
                                    mapping = mapping,
                                    size = text.size,
                                    na.rm = na.rm))
        }
      }
      z <- c(z,
             annotate(geom = "text",
                      x = x.min, y = y.max * 1.09 + 0.5 * y.max * 0.085,
                      size = rel(2), vjust = -0.3, hjust = 0.5, angle = 90,
                      label = summary.label, parse = TRUE,
                      na.rm = na.rm))
    } else {
      if (is.null(label.color)) {
        z <- c(z, stat_wb_label(mapping = aes_(color = ~..BW.color..),
                                w.band = w.band,
                                ypos.fixed = y.max * (1.143 + pos.shift),
                                size = text.size,
                                na.rm = na.rm))
      } else {
        z <- c(z, stat_wb_label(w.band = w.band,
                                ypos.fixed = y.max * (1.143 + pos.shift),
                                color = label.color,
                                size = text.size,
                                na.rm = na.rm))
      }

    }
  }
  z
}

#' Merge user supplied annotations with default ones
#'
#' Allow users to add and subract from default annotations in addition
#' to providing a given set of annotations.
#'
#' @param annotations,annotations.default character vector or a list of
#'   character vectors.
#'
#' @keywords internal
#'
decode_annotations <- function(annotations,
                               annotations.default = "colour.guide") {
  if (length(annotations) == 0L) { # handle character(0) and NULL without delay
    return(annotations.default)
  } else if (is.list(annotations)) {
    annotations.ls <- annotations
  } else if (is.character(annotations)) {
    annotations.ls <- list(annotations)
  }
  annotations <- NULL

  for (annotations in annotations.ls) {
    stopifnot(is.character(annotations))
    if ("color.guide" %in% annotations) {
      annotations <- c(setdiff(annotations, "color.guide"), "colour.guide")
    }
    if ("color.guide" %in% annotations.default) {
      annotations.default <- c(setdiff(annotations.default, "color.guide"), "colour.guide")
    }
    if (length(annotations) == 0L) { # we can receive character(0) from preceeding iteration
      z <- annotations.default
    } else if ("" %in% annotations) {
      # no annotations and do not not expand y scale
      z <- ""
    } else if ("reserve.space" %in% annotations) {
      # no annotations but expand y scale to accomodate them
      z <- "reserve.space"
    } else if (annotations[1] == "-") {
      # remove any member of a "family" of annotations if '*' wildcard is present
      if (any(grepl("^title[*]$", annotations))) {
        annotations.default <- annotations.default[!grepl("^title.*", annotations.default)]
      }
      if (any(grepl("^peaks[*]", annotations))) {
        annotations.default <- annotations.default[!grepl("^peak.*", annotations.default)]
      }
      if (any(grepl("^valleys[*]$", annotations))) {
        annotations.default <- annotations.default[!grepl("^valley.*", annotations.default)]
      }
      # remove exact matches
      z <- setdiff(annotations.default, annotations[-1])
    } else if (annotations[1] == "+") {
      annotations <- annotations[-1]
      # remove from default items to be replaced
      if (any(grepl("^title.*", annotations))) {
        annotations.default <- annotations.default[!grepl("^title.*", annotations.default)]
      }
      if (any(grepl("^peak.*", annotations))) {
        annotations.default <- annotations.default[!grepl("^peak.*", annotations.default)]
      }
      if (any(grepl("^valley.*$", annotations))) {
        annotations.default <- annotations.default[!grepl("^valley.*", annotations.default)]
      }
      if (any(grepl("^boxes$|^segments$", annotations))) {
        annotations.default <- annotations.default[!grepl("^boxes$|^segments$", annotations.default)]
      }
      # merge default with addition
      z <- union(annotations.default, annotations)
    } else if (annotations[1] == "=") {
      # replace
      z <- annotations[-1]
      # handle character(0), using "" is a kludge but produces intuitive behaviour
      if (length(z) == 0L) {
        z <- ""
      }
    } else {
      z <- annotations
    }
    annotations.default <- z
  }

  unique(z) # remove duplicates for tidyness
}

# photobiology.plot.annotations -----------------------------------------------------

#' @title Set defaults for autoplot annotations
#'
#' @description Set R options used when plotting spectra. Option
#'   "photobiology.plot.annotations" is used as default argument to formal
#'   parameter \code{annotations} and option "photobiology.plot.bands" is used
#'   as default argument to formal parameter \code{w.band} in all the
#'   \code{autoplot()} methods exported from package 'ggspectra'. These
#'   convenience functions makes it easier to edit these two option which are
#'   stored as a vector of characters strings and a list of waveband objects,
#'   respectively.
#'
#' @details Vectors of character strings passed as argument to
#'   \code{annotations} are parsed so that if the first member string is
#'   \code{"+"}, the remaining members are added to the current default for
#'   annotations; if it is \code{"-"} the remaining members are removed from the
#'   current default for annotations; and if it is \code{"="} the remaining
#'   members become the new default. If the first member is none of these three
#'   strings, the whole vector becomes the new default. If \code{annotations} is
#'   \code{NULL} the annotations are reset to the package defaults. When
#'   removing annotations \code{"title*"}, \code{"peaks*"} and \code{"valleys*"}
#'   will remove any variation of these annotations. The string \code{""} means
#'   no annotations while \code{"reserve.space"} means no annotations but expand
#'   y scale to reserve space for annotations. These two values take precedence
#'   over any other values in the character vector. The order of the names of
#'   annotations has no meaning: the vector is interpreted as a set except for
#'   the three possible "operators" at position 1.
#'
#' @param annotations character vector Annotations to add or remove from
#'   defaults used by the \code{autoplot()} methods defined in this package..
#'
#' @note The syntax used and behaviour are the same as for the
#'   \code{annotations} parameter of the \code{autoplot()} methods for spectra,
#'   but instead of affecting a single plot, \code{set_annotations_default()}
#'   changes the default used for subsequent calls to \code{autoplot()}.
#'
#' @return Previous value of option "photobiology.plot.annotations" returned
#'   invisibly.
#'
#' @family autoplot methods
#'
#' @export
#'
set_annotations_default <- function(annotations = NULL) {
  if (!is.null(annotations)) {
    annotations.default <-
      getOption("photobiology.plot.annotations",
                default = c("boxes", "labels", "summaries", "colour.guide", "peaks"))
    annotations <- decode_annotations(annotations = annotations,
                                      annotations.default = annotations.default)
  }
  options(photobiology.plot.annotations = annotations)
}

#' @rdname set_annotations_default
#'
#' @param w.band a single waveband object or a list of waveband objects.
#'
#' @export
#'
set_w.band_default <- function(w.band = NULL) {
  if (!is.null(w.band)) {
    # validation to avoid delayed errors
    if (photobiology::is.waveband(w.band)) {
      w.band <- list(w.band) # optimization: avoid repeating this step
    }
    if (!all(sapply(w.band, is.waveband))) {
      warning("Bad 'w.band' argument, default not changed.")
      return(getOption("photobiology.plot.bands"))
    }
  }
  options(photobiology.plot.bands = w.band)
}
