% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-quant-eq.R
\name{stat_quant_eq}
\alias{stat_quant_eq}
\title{Equation, p-value, R^2, AIC or BIC from quantile regression}
\usage{
stat_quant_eq(
  mapping = NULL,
  data = NULL,
  geom = "text_npc",
  position = "identity",
  ...,
  formula = NULL,
  quantiles = c(0.25, 0.5, 0.75),
  eq.with.lhs = TRUE,
  eq.x.rhs = NULL,
  coef.digits = 3,
  rho.digits = 2,
  label.x = "left",
  label.y = "top",
  label.x.npc = NULL,
  label.y.npc = NULL,
  hstep = 0,
  vstep = NULL,
  output.type = "expression",
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs to be
set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific dataset, only needed if you want to override
the plot defaults.}

\item{geom}{The geometric object to use display the data}

\item{position}{The position adjustment to use for overlapping points on this
layer}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}

\item{formula}{a formula object. Using aesthetic names instead of
original variable names.}

\item{quantiles}{numeric vector Values in 0..1 indicating the quantiles.}

\item{eq.with.lhs}{If \code{character} the string is pasted to the front of
the equation label before parsing or a \code{logical} (see note).}

\item{eq.x.rhs}{\code{character} this string will be used as replacement for
\code{"x"} in the model equation when generating the label before parsing
it.}

\item{coef.digits, rho.digits}{integer Number of significant digits to use for
the fitted coefficients and rho in labels.}

\item{label.x, label.y}{\code{numeric} with range 0..1 "normalized parent
coordinates" (npc units) or character if using \code{geom_text_npc()} or
\code{geom_label_npc()}. If using \code{geom_text()} or \code{geom_label()}
numeric in native data units. If too short they will be recycled.}

\item{label.x.npc, label.y.npc}{\code{numeric} with range 0..1 (npc units)
DEPRECATED, use label.x and label.y instead; together with a geom
using npcx and npcy aesthetics.}

\item{hstep, vstep}{numeric in npc units, the horizontal and vertical step
used between labels for different groups.}

\item{output.type}{character One of "expression", "LaTeX", "text",
"markdown" or "numeric".}

\item{na.rm}{a logical indicating whether NA values should be stripped before
the computation proceeds.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and shouldn't inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}
}
\description{
\code{stat_quant_eq} fits a polynomial model by quantile regression and
generates several labels including the equation, p-value, coefficient of
determination (R^2), 'AIC' and 'BIC'.
}
\details{
This stat can be used to automatically annotate a plot with R^2,
  adjusted R^2 or the fitted model equation. It supports only linear models
  fitted with function \code{lm()}. The R^2 and adjusted R^2 annotations can
  be used with any linear model formula. The fitted equation label is
  correctly generated for polynomials or quasi-polynomials through the
  origin. Model formulas can use \code{poly()} or be defined algebraically
  with terms of powers of increasing magnitude with no missing intermediate
  terms, except possibly for the intercept indicated by "- 1" or "-1" in the
  formula. The validity of the \code{formula} is not checked in the current
  implementation, and for this reason the default aesthetics sets R^2 as
  label for the annotation. This stat only generates labels, the predicted
  values need to be separately added to the plot, so to make sure that the
  same model formula is used in all steps it is best to save the formula as
  an object and supply this object as argument to the different statistics.

  A ggplot statistic receives as data a data frame that is not the one passed
  as argument by the user, but instead a data frame with the variables mapped
  to aesthetics. stat_quant_eq() mimics how stat_smooth() works, except that
  only polynomials can be fitted. In other words, it respects the grammar of
  graphics. This helps ensure that the model is fitted to the same data as
  plotted in other layers.
}
\note{
For backward compatibility a logical is accepted as argument for
  \code{eq.with.lhs}, giving the same output than the current default
  character value. By default "x" is retained as independent variable as this
  is the name of the aesthetic. However, it can be substituted by providing a
  suitable replacement character string through \code{eq.x.rhs}.

Support for the \code{angle} aesthetic is not automatic and requires
  that the user passes as argument suitable numeric values to override the
  defaults.
}
\section{Aesthetics}{
 \code{stat_quant_eq} understands \code{x} and \code{y},
  to be referenced in the \code{formula} and \code{weight} passed as argument
  to parameter \code{weights} of \code{lm()}. All three must be mapped to
  \code{numeric} variables. In addition, the aesthetics undertood by the geom
  used (\code{"text"} by default) are understood and grouping respected.
}

\section{Computed variables}{

If output.type different from \code{"numeric"} the returned tibble contains
columns:
\describe{
  \item{x,npcx}{x position}
  \item{y,npcy}{y position}
  \item{coef.ls, r.squared, adj.r.squared, AIC, BIC}{as numric values extracted from fit object}
  \item{eq.label}{equation for the fitted polynomial as a character string to be parsed}
  \item{rho.label}{\eqn{rho} of the fitted model as a character string to be parsed}
  \item{AIC.label}{AIC for the fitted model.}
  \item{n.label}{Number of observations used in the fit.}
  \item{rq.method}{character, method used.}
  \item{rho, n}{numeric values extracted or computed from fit object.}
  \item{hjust, vjust}{Set to "inward" to override the default of the "text" geom.}}

If output.type is \code{"numeric"} the returned tibble contains columns:
\describe{
  \item{x,npcx}{x position}
  \item{y,npcy}{y position}
  \item{coef.ls}{list containing the "coefficients" matrix from the summary of the fit object}
  \item{rho, AIC, n}{numeric values extracted or computed from fit object}
  \item{rq.method}{character, method used.}
  \item{hjust, vjust}{Set to "inward" to override the default of the "text" geom.}}

To explore the computed values returned for a given input we suggest the use
of \code{\link[gginnards]{geom_debug}} as shown in the example below.
}

\section{Parsing may be required}{
 if using the computed labels with
  \code{output.type = "expression"}, then \code{parse = TRUE} is needed,
  while if using \code{output.type = "LaTeX"} \code{parse = FALSE} is needed.
}

\examples{
# generate artificial data
set.seed(4321)
x <- 1:100
y <- (x + x^2 + x^3) + rnorm(length(x), mean = 0, sd = mean(x^3) / 4)
my.data <- data.frame(x = x, y = y,
                      group = c("A", "B"),
                      y2 = y * c(0.5,2),
                      w = sqrt(x))

# give a name to a formula
formula <- y ~ poly(x, 3, raw = TRUE)

# no weights
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE)

# angle
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE, angle = 90,
                hstep = 0.05, vstep = 0, hjust = 0,
                label.y = 0.5)

ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE, angle = 90,
                hstep = 0.05, vstep = 0, hjust = 0,
                label.y = 0.5)

# user set quantiles
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula, quantiles = 0.5) +
  stat_quant_eq(formula = formula, quantiles = 0.5, parse = TRUE)

# grouping
ggplot(my.data, aes(x, y, color = group)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE)

ggplot(my.data, aes(x, y, color = group)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE, angle = 90,
                hstep = 0.05, vstep = 0, hjust = 0,
                label.y = 0.5)

# labelling equations
ggplot(my.data, aes(x, y,  shape = group, linetype = group,
       grp.label = group)) +
  geom_point() +
  geom_quantile(formula = formula, color = "black") +
  stat_quant_eq(aes(label = paste(stat(grp.label), stat(eq.label), sep = "*\": \"*")),
                formula = formula, parse = TRUE) +
  theme_classic()

# setting non-default quantiles
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula,
                quantiles = c(0.1, 0.5, 0.9)) +
  stat_quant_eq(formula = formula, parse = TRUE,
               quantiles = c(0.1, 0.5, 0.9))

# Location of equations
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE,
               label.y = "bottom", label.x = "right")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE,
               label.y = 0.03, label.x = 0.95, vstep = 0.04)

# using weights
ggplot(my.data, aes(x, y, weight = w)) +
  geom_point() +
  geom_quantile(formula = formula) +
  stat_quant_eq(formula = formula, parse = TRUE)

# no weights, quantile set to upper boundary
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(formula = formula, quantiles = 1) +
  stat_quant_eq(formula = formula, quantiles = 1, parse = TRUE)

# user specified label
ggplot(my.data, aes(x, y, color = group, grp.label = group)) +
  geom_point() +
  geom_quantile(method = "rq", formula = formula,
                quantiles = c(0.05, 0.5, 0.95)) +
  stat_quant_eq(aes(label = paste(stat(grp.label), "*\": \"*",
                                   stat(eq.label), sep = "")),
                quantiles = c(0.05, 0.5, 0.95),
                formula = formula, parse = TRUE)

# geom = "text"
ggplot(my.data, aes(x, y)) +
  geom_point() +
  geom_quantile(method = "rq", formula = formula, quantiles = 0.5) +
  stat_quant_eq(label.x = "left", label.y = "top",
                formula = formula, parse = TRUE)

# Examples using geom_debug() to show computed values
#
# This provides a quick way of finding out which variables are available for
# use in mapping of aesthetics when using other geoms as in the examples
# above.

library(gginnards)

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, geom = "debug")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(aes(label = stat(eq.label)),
               formula = formula, geom = "debug",
               output.type = "markdown")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, geom = "debug", output.type = "text")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, geom = "debug", output.type = "numeric")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, quantiles = c(0.25, 0.5, 0.75),
                geom = "debug", output.type = "text")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, quantiles = c(0.25, 0.5, 0.75),
                geom = "debug", output.type = "numeric")

# show the content of a list column
ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quant_eq(formula = formula, geom = "debug", output.type = "numeric",
               summary.fun = function(x) {x[["coef.ls"]][[1]]})

}
\references{
Written as an answer to a question by Mark Neal at Stackoverflow.
  \url{https://stackoverflow.com/questions/65695409/is-there-a-neat-approach-to-label-a-ggplot-plot-with-the-equation-and-other-stat}
}
\seealso{
This \code{stat_quant_eq} statistic can return ready formatted labels
  depending on the argument passed to \code{output.type}. This is possible
  because only polynomial models are supported. For other types of models,
  statistics \code{\link{stat_fit_glance}},  \code{\link{stat_fit_tidy}} and
  \code{\link{stat_fit_glance}} should be used instead and the code for
  construction of character strings from numeric values and their mapping to
  aesthetic \code{label} needs to be explicitly supplied in the call.

Other statistics for linear model fits: 
\code{\link{stat_fit_deviations}()},
\code{\link{stat_fit_residuals}()},
\code{\link{stat_poly_eq}()}
}
\concept{statistics for linear model fits}
