\name{gglasso}
\alias{gglasso}
\title{Fits the regularization paths for group-lasso penalized learning problems}
\description{Fits regularization paths for group-lasso penalized learning problems at a sequence of regularization parameters lambda.}
\usage{
gglasso	(x, y, group = NULL, 
		loss = c("ls", "logit", "sqsvm", "hsvm"), 
		nlambda = 100, 
		lambda.factor = ifelse(nobs < nvars, 0.05, 0.001), 
		lambda = NULL, 
		pf = sqrt(bs), 
		dfmax = as.integer(max(group)) + 1, 
		pmax = min(dfmax * 1.2, as.integer(max(group))), 
		eps = 1e-08, maxit = 3e+08, delta, intercept=TRUE)
}
\arguments{
		\item{x}{matrix of predictors, of dimension \eqn{n \times p}{n*p}; each row is an observation vector.}
		\item{y}{response variable. This argument should be quantitative for regression (least squares), and a two-level factor for classification (logistic model, huberized SVM, squared SVM).}
		\item{group}{a vector of consecutive integers describing the grouping of the coefficients (see example below).}
		\item{loss}{a character string specifying the loss function to use, valid options are:
		\itemize{
		\item \code{"ls"} least squares loss (regression),
		\item \code{"logit"} logistic loss (classification).
		\item \code{"hsvm"} Huberized squared hinge loss (classification),
		\item \code{"sqsvm"} Squared hinge loss (classification),
		}Default is \code{"ls"}.}
		\item{nlambda}{the number of \code{lambda} values - default is 100.}
		\item{lambda.factor}{the factor for getting the minimal lambda in \code{lambda} sequence, where \code{min(lambda)} = \code{lambda.factor} * \code{max(lambda)}.  \code{max(lambda)} is the smallest value of \code{lambda} for which all coefficients are zero. The default depends on the relationship between \eqn{n} (the number of rows in the matrix of predictors) and \eqn{p} (the number of predictors). If \eqn{n >= p}, the default is \code{0.001},
		close to zero.  If \eqn{n<p}, the default is \code{0.05}.
		A very small value of \code{lambda.factor} will lead to a saturated fit. It takes no effect if there is user-defined \code{lambda} sequence.} 
		\item{lambda}{a user supplied \code{lambda} sequence. Typically, by leaving this option unspecified users can have 
		the program compute its own \code{lambda} sequence based on
		\code{nlambda} and \code{lambda.factor}. Supplying a value of
		\code{lambda} overrides this. It is better to supply
		a decreasing sequence of \code{lambda} values than a single (small) value, if not, the program will sort user-defined \code{lambda} sequence in decreasing order automatically.}
		\item{pf}{penalty factor, a vector in length of bn (bn is the total number of groups). Separate penalty weights can be applied to each group of \eqn{\beta}{beta's}s to allow
		differential shrinkage. Can be 0 for some groups, which implies
		no shrinkage, and results in that group always being included in the
		model. Default value for each entry is the square-root of the corresponding size of each group.}
		
		\item{dfmax}{limit the maximum number of groups in the
		model. Useful for very large \code{bs} (group size), if a partial path is desired. Default is \code{bs+1}.}

		\item{pmax}{limit the maximum number of groups ever to be nonzero. For example once a group enters the model, no matter how many times it exits or re-enters model through the path, it will be counted only once. Default is \code{min(dfmax*1.2,bs)}.}

		\item{eps}{convergence termination tolerance. Defaults value is \code{1e-8}.}

		\item{maxit}{maximum number of outer-loop iterations allowed at fixed lambda value. Default is 3e8. If models do not converge, consider increasing \code{maxit}.}

		\item{delta}{the parameter \eqn{\delta}{delta} in \code{"hsvm"} (Huberized squared hinge loss). Default is 1.}
	 
	\item{intercept}{Whether to include intercept in the model. Default is TRUE.}
}

\details{
Note that the objective function for \code{"ls"} least squares is
\deqn{RSS/(2*n) + lambda * penalty;}
for \code{"hsvm"} Huberized squared hinge loss, \code{"sqsvm"} Squared hinge loss and \code{"logit"} logistic regression, the objective function is
\deqn{-loglik/n + lambda * penalty.}
Users can also tweak the penalty by choosing different penalty factor. 

For computing speed reason, if models are not converging or running slow, consider increasing \code{eps}, decreasing
\code{nlambda}, or increasing \code{lambda.factor} before increasing
\code{maxit}.
}


\value{
An object with S3 class \code{\link{gglasso}}.
		\item{call}{the call that produced this object}
		\item{b0}{intercept sequence of length \code{length(lambda)}}
		\item{beta}{a \code{p*length(lambda)} matrix of coefficients.}
		\item{df}{the number of nonzero groups for each value of
		\code{lambda}.}
		\item{dim}{dimension of coefficient matrix (ices)}
		\item{lambda}{the actual sequence of \code{lambda} values used}
		\item{npasses}{total number of iterations (the most inner loop) summed over all lambda values}
		\item{jerr}{error flag, for warnings and errors, 0 if no error.}
		\item{group}{a vector of consecutive integers describing the grouping of the coefficients.}
}

\author{Yi Yang and Hui Zou\cr
Maintainer: Yi Yang  <yiyang@umn.edu>}
\references{
Yang, Y. and Zou, H. (2012), ``A Fast Unified Algorithm for Computing Group-Lasso Penalized Learning Problems,'' \emph{Statistics and Computing}. Accepted.\cr
BugReport: \url{http://code.google.com/p/gglasso/}\cr
}


\seealso{\code{plot.gglasso}}
\examples{
# load gglasso library
library(gglasso)

# load bardet data set
data(bardet)

# define group index
group1 <- rep(1:20,each=5)

# fit group lasso penalized least squares
m1 <- gglasso(x=bardet$x,y=bardet$y,group=group1,loss="ls")

# load colon data set
data(colon)

# define group index
group2 <- rep(1:20,each=5)

# fit group lasso penalized logistic regression
m2 <- gglasso(x=colon$x,y=colon$y,group=group2,loss="logit")
}
\keyword{models}
\keyword{regression}
