% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/density.R
\name{density_histogram}
\alias{density_histogram}
\title{Histogram density estimator}
\usage{
density_histogram(
  x,
  weights = NULL,
  breaks = "Scott",
  align = "none",
  outline_bars = FALSE,
  na.rm = FALSE,
  ...,
  range_only = FALSE
)
}
\arguments{
\item{x}{numeric vector containing a sample to compute a density estimate for.}

\item{weights}{optional numeric vector of weights to apply to \code{x}.}

\item{breaks}{Determines the breakpoints defining bins. Defaults to \code{"Scott"}.
Similar to (but not exactly the same as) the \code{breaks} argument to \code{\link[graphics:hist]{graphics::hist()}}.
One of:
\itemize{
\item A scalar (length-1) numeric giving the number of bins
\item A vector numeric giving the breakpoints between histogram bins
\item A function taking \code{x} and \code{weights} and returning either the
number of bins or a vector of breakpoints
\item A string giving the suffix of a function that starts with
\code{"breaks_"}. \pkg{ggdist} provides weighted implementations of the
\code{"Sturges"}, \code{"Scott"}, and \code{"FD"} break-finding algorithms from
\code{\link[graphics:hist]{graphics::hist()}}, as well as \code{\link[=breaks_fixed]{breaks_fixed()}} for manually setting
the bin width. See \link{breaks}.
}

For example, \code{breaks = "Sturges"} will use the \code{\link[=breaks_Sturges]{breaks_Sturges()}} algorithm,
\code{breaks = 9} will create 9 bins, and \code{breaks = breaks_fixed(width = 1)} will
set the bin width to \code{1}.}

\item{align}{Determines how to align the breakpoints defining bins. Default
(\code{"none"}) performs no alignment. One of:
\itemize{
\item A scalar (length-1) numeric giving an offset that is subtracted from the breaks.
The offset must be between \code{0} and the bin width.
\item A function taking a sorted vector of \code{breaks} (bin edges) and returning
an offset to subtract from the breaks.
\item A string giving the suffix of a function that starts with
\code{"align_"} used to determine the alignment, such as \code{\link[=align_none]{align_none()}},
\code{\link[=align_boundary]{align_boundary()}}, or \code{\link[=align_center]{align_center()}}.
}

For example, \code{align = "none"} will provide no alignment, \code{align = align_center(at = 0)}
will center a bin on \code{0}, and \code{align = align_boundary(at = 0)} will align a bin
edge on \code{0}.}

\item{outline_bars}{Should outlines in between the bars (i.e. density values of
0) be included?}

\item{na.rm}{Should missing (\code{NA}) values in \code{x} be removed?}

\item{...}{Additional arguments (ignored).}

\item{range_only}{If \code{TRUE}, the range of the output of this density estimator
is computed and is returned in the \verb{$x} element of the result, and \code{c(NA, NA)}
is returned in \verb{$y}. This gives a faster way to determine the range of the output
than \code{density_XXX(n = 2)}.}
}
\value{
An object of class \code{"density"}, mimicking the output format of
\code{\link[stats:density]{stats::density()}}, with the following components:
\itemize{
\item \code{x}: The grid of points at which the density was estimated.
\item \code{y}: The estimated density values.
\item \code{bw}: The bandwidth.
\item \code{n}: The sample size of the \code{x} input argument.
\item \code{call}: The call used to produce the result, as a quoted expression.
\item \code{data.name}: The deparsed name of the \code{x} input argument.
\item \code{has.na}: Always \code{FALSE} (for compatibility).
\item \code{cdf}: Values of the (possibly weighted) empirical cumulative distribution
function at \code{x}. See \code{\link[=weighted_ecdf]{weighted_ecdf()}}.
}

This allows existing methods for density objects, like \code{\link[=print]{print()}} and \code{\link[=plot]{plot()}}, to work if desired.
This output format (and in particular, the \code{x} and \code{y} components) is also
the format expected by the \code{density} argument of the \code{\link[=stat_slabinterval]{stat_slabinterval()}}
and the \code{\link[=smooth_density]{smooth_}} family of functions.
}
\description{
Histogram density estimator.

Supports \link[=automatic-partial-functions]{automatic partial function application}.
}
\examples{
library(distributional)
library(dplyr)
library(ggplot2)

# For compatibility with existing code, the return type of density_unbounded()
# is the same as stats::density(), ...
set.seed(123)
x = rbeta(5000, 1, 3)
d = density_histogram(x)
d

# ... thus, while designed for use with the `density` argument of
# stat_slabinterval(), output from density_histogram() can also be used with
# base::plot():
plot(d)

# here we'll use the same data as above with stat_slab():
data.frame(x) \%>\%
  ggplot() +
  stat_slab(
    aes(xdist = dist), data = data.frame(dist = dist_beta(1, 3)),
    alpha = 0.25
  ) +
  stat_slab(aes(x), density = "histogram", fill = NA, color = "#d95f02", alpha = 0.5) +
  scale_thickness_shared() +
  theme_ggdist()
}
\seealso{
Other density estimators: 
\code{\link{density_bounded}()},
\code{\link{density_unbounded}()}
}
\concept{density estimators}
