% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/impacts.R
\name{spill}
\alias{spill}
\alias{impacts}
\alias{print.impacts_slm}
\title{Spillover/diffusion effects for spatial lag models}
\source{
LeSage, James and Pace, R. Kelley (2009). \emph{Introduction to Spatial Econometrics}. CRC Press.

LeSage, James (2014). What Regional Scientists Need to Know about Spatial Econometrics. \emph{The Review of Regional Science} 44: 13-32 (2014 Southern Regional Science Association Fellows Address).
}
\usage{
spill(beta, gamma = 0, rho, W, approx = TRUE, K = 15)

impacts(object, approx = TRUE, K = 15)

\method{print}{impacts_slm}(x, digits = 2, ...)
}
\arguments{
\item{beta}{Coefficient for covariates (numeric vector)}

\item{gamma}{Coefficient for spatial lag of covariates}

\item{rho}{Spatial dependence parameter (single numeric value)}

\item{W}{Spatial weights matrix}

\item{approx}{For a computationally efficient approximation to the required matrix inverse (after LeSage and Pace 2009, pp. 114--115); if \code{FALSE}, then a proper matrix inverse will be computed using \code{Matrix::solve}.}

\item{K}{Degree of polynomial in the expansion to use when 'approx = TRUE`.}

\item{object}{A fitted spatial lag  model (from \code{stan_sar})}

\item{x}{An object of class 'impacts_slm', as returned by \code{geostan::impacts}}

\item{digits}{Round results to this many digits}

\item{...}{Additional arguments will be passed to \code{base::print}}
}
\description{
Spillover/diffusion effects for spatial lag models
}
\details{
These methods apply only to the spatial lag and spatial Durbin lag models (SLM and SDLM) as fit by \code{geostan::stan_sar}.

The equation for these SAR models specifies simultaneous feedback between all units, such that changing the outcome in one location has a spill-over effect that may extend to all other locations (a ripple or diffusion effect); the induced changes will also react back onto the first unit. (This presumably takes time, even if the observations are cross-sectional.)

These spill-overs have to be incorporated into the interpretation and reporting of the regression coefficients of SLM and SDLM models. A unit change in the value of \eqn{X} in one location will impact \eqn{y} in that same place ('direct' impact) and will also impact \eqn{y} elsewhere through the diffusion process ('indirect' impact). The 'total' expected impact of a unit change in \code{X} is the sum of the direct and indirect effects (LeSage and Pace 2009).

The \code{spill} function is for quickly calculating average spillover effects given point estimates of parameters.

The \code{impacts} function calculates the (average) direct, indirect, and total effects once for every MCMC sample to produce samples from the posterior distribution for the impacts; the samples are returned together with a summary of the posterior distribution (mean, median, and select quantiles).
}
\examples{
##
## SDLM data
##

parts <- prep_sar_data2(row = 9, col = 9, quiet = TRUE)
W <- parts$W
x <- sim_sar(w=W, rho=.6)
Wx <- (W \%*\% x)[,1]
mu <- .5 * x + .25 * Wx
y <- sim_sar(w=W, rho=0.6, mu = mu, type = "SLM")
dat <- cbind(y, x)

# impacts per the above parameters
spill(0.5, 0.25, 0.6, W)

##
## impacts for SDLM
##

fit <- stan_sar(y ~ x, data = dat, sar = parts,
                type = "SDLM", iter = 500,
                slim = TRUE, quiet = TRUE) 

# impacts (posterior distribution)
impax <- impacts(fit)
print(impax)

# plot posterior distributions
og = par(mfrow = c(1, 3),
         mar = c(3, 3, 1, 1))
S <- impax$samples[[1]]
hist(S[,1], main = 'Direct')
hist(S[,2], main = 'Indirect')
hist(S[,3], main = 'Total')
par(og)

##
## The approximate method
##

# High rho value requires more K; rho^K must be near zero
Ks <- c(10, 15, 20, 30, 35, 40)
print(cbind(Ks, 0.9^Ks))

# understand sensitivity of results to K when rho is high
spill(0.5, -0.25, 0.9, W, approx = TRUE, K = 10)
spill(0.5, -0.25, 0.9, W, approx = TRUE, K = 20)
spill(0.5, -0.25, 0.9, W, approx = TRUE, K = 30)
spill(0.5, -0.25, 0.9, W, approx = TRUE, K = 50)

# the correct results
spill(0.5, -0.25, 0.9, W, approx = FALSE)

# moderate and low rho values are fine with smaller K
spill(0.5, -0.25, 0.7, W, approx = TRUE, K = 15)
spill(0.5, -0.25, 0.7, W, approx = FALSE)

}
