#' Estimate the parameters of a GARMA model.
#'
#' The garma function is the main function for the garma package. Depending on the parameters it will
#' calculate the parameter estimates for the GARMA process, and if available the standard errors (se's)
#' for those parameters.
#'
#' The GARMA model is specified as
#' \deqn{\displaystyle{\phi(B)\prod_{i=1}^{k}(1-2u_{i}B+B^{2})^{d_{i}}(1-B)^{id} (X_{t}-\mu)= \theta(B) \epsilon _{t}}}{\prod(i=1 to k) (1-2u(i)B+B^2)^d(i) \phi(B)(1-B)^{id} (X(t) - \mu) = \theta(B) \epsilon(t)}
#'
#' where
#' \itemize{
#' \item \eqn{\phi(B)}{\phi(B)} represents the short-memory Autoregressive component of order p,
#' \item \eqn{\theta(B)}{\theta(B)} represents the short-memory Moving Average component of order q,
#' \item \eqn{(1-2u_{i}B+B^{2})^{d_{i}}}{(1-2u(i)B+B^2)^d(i)} represents the long-memory Gegenbauer component (there may in general be k of these),
#' \item \eqn{id} represents the degree of integer differencing.
#' \item \eqn{X_{t}}{X(t)} represents the observed process,
#' \item \eqn{\epsilon_{t}}{\epsilon(t)} represents the random component of the model - these are assumed to be uncorrelated but identically distributed variates.
#'       Generally the routines in this package will work best if these have an approximate Gaussian distribution.
#' \item \eqn{B}{B} represents the Backshift operator, defined by \eqn{B X_{t}=X_{t-1}}{B X(t) = X(t-1)}.
#' }
#' when k=0, then this is just a short memory model as fit by the stats "arima" function.
#'
#' @param x (num) This should be a numeric vector representing the process to estimate. A minimum length of 96 is required.
#' @param order (list) This should be a list (similar to the stats::arima order parameter) which will give the order of the process to fit.
#'     The format should be list(p,d,q) where p, d, and q are all positive integers. p represents the degree of the
#'     autoregressive process to fit, q represents the order of the moving average process to fit and d is the (integer)
#'     differencing to apply prior to any fitting. WARNING: Currently only d==0 or d==1 are allowed.
#' @param k (int) This is the number of (multiplicative) Gegenbauer terms to fit. Note the 'QML' method only allows k=1.
#' @param include.mean (bool) A boolean value indicating whether a mean should be fit.
#'     Note that no mean term is fit if the series is integer differenced.
#' @param include.drift (bool) A boolean value indicating whether a 'drift' term should be fit to the predictions.
#'     The default is to fit a drift term to the predictions if the process is integer-differenced.
#' @param method (character) This defines the estimation method for the routine. The valid values are 'CSS', 'Whittle', 'QML' and 'WLL'.
#'     The default (Whittle) will generally return very accurate estimates quite quickly, provided the assumption of a Gaussian
#'     distribution is even approximately correct, and is probably the method of choice for most users. For the theory behind this, refer Giraitis et. al. (2001)
#'     'CSS' is a conditional 'sum-of-squares' technique and can be quite slow. Reference: Chung (1996).
#'     'QML' is a Quasi-Maximum-Likelihood technique, and can also be quite slow. Reference Dissanayake (2016). (k>1 is not supported for QML)
#'     'WLL' is a new technique which appears to work well even if the \eqn{\epsilon_{t}}{\epsilon(t)} are highly skewed and/or have heavy tails (skewed and/or lepto-kurtic).
#'     However the asymptotic theory for the WLL method is not complete and so standard errors are not available for most parameters.
#' @param allow_neg_d (bool) A boolean value indicating if a negative value is allowed for the fractional differencing component
#'     of the Gegenbauer term is allowed. This can be set to FALSE (the default) to force the routine to find a positive value.
## @param maxeval (int) the maximum function eveluations to be allowed during each optimisation.
#' @param opt_method (character) This names the optimisation method used to find the parameter estimates.
#' This may be a list of methods, in which case the methods are applied in turn,
#' each using the results of the previous one as the starting point for the next. The default is to use c('directL', 'solnp') when k<2 and 'solnp' when k>=2. The
#' directL algorithm is used to perform a global search for the minima, and solnp to refine the values.
#' For some data or some models, however, other methods may work well.
#' Supported algorithms include:
#'     \itemize{
#'     \item cobyla algorithm in package nloptr
#'     \item directL algorithm in package nloptr
#'     \item BBoptim from package BB
#'     \item psoptim from package pso
#'     \item hjkb from dfoptim package
#'     \item nmkb from dfoptim package
#'     \item solnp from Rsolnp package
#'     \item gosolnp from Rsolnp package
#'     \item best - this option evaluates all the above options in turn and picks the one which finds the lowest value of the objective. This can be quite time consuming to run,
#'     particularly for the 'CSS' method.
#'     }
#' Note further that if you specify a k>1, then inequality constraints are required, and this will further limit the list of supported routines.
#' @param m_trunc Used for the QML estimation method. This defines the AR-truncation point when evaluating the likelihood function. Refer to Dissanayake et. al. (2016) for details.
#' @param min_freq (num) When searching for Gegenbauer peaks, this is the minimum frequency used. Default 0. Note that when there is an
#' AR(1) component, the peaks corresponding to the AR(1) can be higher than the Gegenbauer peaks. Setting this parameter to 0.05 or above can help.
#' @param max_freq (num) default 0.5. When searching for Gegenbauer peaks, this is the maximum frequency used.
#' @param control (list) list of optimisation routine specific values.
#' @return An S3 object of class "garma_model".
#'
#' @references
#' C Chung. A generalized fractionally integrated autoregressive moving-average process.
#' Journal of Time Series Analysis, 17(2):111–140, 1996.
#'
#' G Dissanayake, S Peiris, and T Proietti. State space modelling of Gegenbauer processes with long memory.
#' Computational Statistics and Data Analysis, 100:115–130, 2016.
#'
#' L Giraitis, J Hidalgo, and P Robinson. Gaussian estimation of parametric spectral density with unknown pole.
#' The Annals of Statistics, 29(4):987–1023, 2001.
#' @examples
#' data(AirPassengers)
#' ap  <- as.numeric(diff(AirPassengers,12))
#' print(garma(ap,order=c(9,1,0),k=0,method='CSS',include.mean=FALSE))
#' # Compare with the built-in arima function
#' print(arima(ap,order=c(9,1,0),include.mean=FALSE))
#' @export
garma<-function(x,
                order=list(0,0,0),
                k=1,
                include.mean=(order[2]==0),
                include.drift=(order[2]>0),
                method='Whittle',
                allow_neg_d=FALSE,
                # maxeval=10000,
                opt_method=NULL,
                m_trunc=50,
                min_freq=0,
                max_freq=0.5,
                control=NULL) {

  ## Start of "garma" function logic.
  ## 1. Check parameters
  if (length(x)<96)
    # This is a bit arbitary but I would be concerned about a process of length even 96.
    # But no real evidence for this restriction apart from simulations showing large std errs.
    stop('y should have at least 96 observations.')
  if (is.data.frame(x)) {
    if (ncol(x)>1)
      stop('x should be a numeric vector - not an entire data frame. Please select a single column and try again.')
    else x<-x[,1]
  }

  # now save the ts elements, if any - we re-apply them to the output values
  x_start <- stats::start(x)
  x_end   <- stats::end(x)
  x_freq  <- stats::frequency(x)

  x<-as.numeric(x)
  if (!is.numeric(x))
    stop('x should be numeric or a ts object.\n')
  if (any(is.na(x)))
    stop('x should not have any missing values.\n')
  if (length(order)!=3)
    stop('order parameter must be a 3 integers only.\n')
  if (any(order<0))
    stop('order parameter must consist of positive integers.\n')
  # if (order[2]!=0&order[2]!=1) # this restriction exists because the "predict" function cannot handle higher differencing
  #   stop('Sorry. Currently only d==0 or d==1 are supported.\n')
  if (k<0)
    stop('k parameter must be a non-negative integer.\n')
  if (order[1]+order[3]+k<=0)
    stop('At least one of p, q or k must be positive.\n')
  if (order[2]>0&include.mean) {
    warning('"include.mean" is ignored since integer differencing is specified.\n')
    include.mean <- FALSE
  }

  allowed_methods <- c('CSS','Whittle','WLL','QML')
  if (!method%in%allowed_methods)
    stop('Method must be one of CSS, Whittle, QML or WLL.\n')
  if (method=='QML'&k>1)
    stop('QML method does not support k>1. It is suggested you try either the CSS or Whittle methods.\n')

  if (is.null(opt_method)) {
    if (k>=2) opt_method <- 'solnp'
    else opt_method <- c('directL','solnp')
  }

  for (om in opt_method) {
    if (!om%in%.supported_optim())
      stop(sprintf('\nError: function %s not available.\n\nSupported functions are:\n%s\n', om, .supported_optim()))

    optimisation_packages <- .optim_packages()
    if (!isNamespaceLoaded(optimisation_packages[[om]]))
      stop(sprintf('Package %s is required to use method %s\n',optimisation_packages[[om]],om))

  }
  # check min_freq and max_freq
  if (!is.numeric(min_freq)|!is.numeric(max_freq)|min_freq<0|min_freq>=0.5|max_freq<=0|max_freq>0.5|min_freq>=max_freq)
    stop('min_freq and max_freq must be numeric and between 0 and 0.5 and min_freq<max_freq.\n')

  if (missing(control)|is.null(control)) control <- list(tol=1e-15,maxeval=10000,max_eval=10000,maxit=10000,trace=0,delta=1e-10)

  ##
  ## 2. Next calc parameter  estimates
  p=as.integer(order[1])
  d=as.integer(order[2])
  q=as.integer(order[3])
  storage.mode(x) <- 'double'

  if (d>0) y <- diff(x,differences=d) else y <- x
  mean_y <- mean(y)
  sd_y   <- stats::sd(y)
  # ss<-stats::spectrum((y-mean_y)/sd_y,plot=FALSE,detrend=FALSE,demean=FALSE,method='pgram',taper=0,fast=FALSE)
  no_scaling <- TRUE
  if (no_scaling) ss<-stats::spectrum((y-mean_y),plot=FALSE,detrend=TRUE,demean=TRUE,method='pgram',taper=0,fast=FALSE)
  else ss<-stats::spectrum((y-mean_y)/sd_y,plot=FALSE,detrend=TRUE,demean=TRUE,method='pgram',taper=0,fast=FALSE)

  # Now set up params and lb (lower bounds) and ub (upper bounds)
  n_pars   <- 0
  pars     <- numeric(0)
  lb       <- numeric(0)
  ub       <- numeric(0)
  lb_finite<- numeric(0)
  ub_finite<- numeric(0)

  mean_methods <- c('QML','CSS')
  if (include.mean&method%in%mean_methods) {
    n_pars    <- n_pars+1
    mean_y    <- mean(y)
    pars      <- c(pars,mean_y)
    lb_finite <- c(lb_finite,ifelse(mean_y<0, 2*mean_y, -2*mean_y))
    ub_finite <- c(ub_finite,ifelse(mean_y<0,-2*mean_y,  2*mean_y))
    lb        <- c(lb,-Inf)
    ub        <- c(ub,Inf)
  }

  if (k>0) {# initial parameter estimates for Gegenbauer factors
    gf <- ggbr_semipara(y,k=k,min_freq=min_freq,max_freq=max_freq)
    for (k1 in 1:k) {
      n_pars    <- n_pars+2
      gf1 <- gf$ggbr_factors[[k1]]
      start_u <- gf1$u
      start_d <- gf1$fd

      max_u <- cos(2*pi*min_freq)
      min_u <- cos(2*pi*max_freq)
      #params
      if (start_u< min_u|start_u > max_u) start_u <- (min_u+max_u)/2
      if (start_d>=0.5)  start_d <- 0.49
      if (allow_neg_d) {
        if (start_d<= -0.5) start_d<- -0.49
      } else {
        if (start_d<=0.0) start_d<-0.01
      }

      pars      <- c(pars,start_u,start_d)
      lb        <- c(lb,min_u,ifelse(allow_neg_d,-1,0))
      ub        <- c(ub,max_u,0.5)
      lb_finite <- c(lb_finite,min_u,ifelse(allow_neg_d,-1,0))
      ub_finite <- c(ub_finite,max_u,0.5)
    }
  }

  n_pars    <- n_pars + p + q
  methods_to_estimate_var <- c('WLL')     # WLL method estimates VAR along with other params; For other methods this falls out of the objective value
  if (p+q>0) {
    # if any ARMA params to be estimated, we use the semipara estimates to get ggbf factors then get the underlying ARMA process,
    # and then ask "arima" for estimates. Semi para estimates should make good starting points for optimisation.
    if (k>0) arma_y <- extract_arma(y,gf$ggbr_factors)
    else arma_y <- y
    a <- stats::arima(arma_y,order=c(p,0,q),include.mean=FALSE)
    pars <- c(pars,a$coef)
    if (method%in%methods_to_estimate_var) {
      pars <- c(pars,a$sigma2)
    }
    if (p==1&q==0) { # special limits for AR(1)
      lb<-c(lb,-1)
      if (method%in%methods_to_estimate_var) lb <- c(lb, 1e-10)
      lb_finite <- c(lb_finite,-1)
      if (method%in%methods_to_estimate_var) lb_finite <- c(lb_finite, 1e-10)
      ub<-c(ub,1)
      if (method%in%methods_to_estimate_var) ub<-c(ub,Inf)
      ub_finite <- c(ub_finite,1)
      if (method%in%methods_to_estimate_var) ub_finite <- c(ub_finite,2*stats::var(y))
    } else {
      lb<- c(lb,rep(-Inf,p+q))
      if (method%in%methods_to_estimate_var) lb <- c(lb,1e-10)
      ub<- c(ub,rep(Inf,p+q))
      if (method%in%methods_to_estimate_var) ub <- c(ub,Inf)
      lb_finite <- c(lb_finite,rep(-10,p+q))
      if (method%in%methods_to_estimate_var) lb_finite <- c(lb_finite,1e-10)
      ub_finite <- c(ub_finite,rep(10,p+q))
      if (method%in%methods_to_estimate_var) ub_finite <- c(ub_finite,2*stats::var(y))
    }
  }

  # create a list of all possible params any 'method' might need. The various objective functions can extract the parameters which are relevant to that method.
  params <- list(y=y, orig_y=x, ss=ss, p=p,q=q,d=d,k=k,
                 include.mean=include.mean,
                 method=method,
                 est_mean=ifelse(method%in%mean_methods,TRUE,FALSE),
                 scale=sd_y,m_trunc=m_trunc)
  message <- c()

  # Optimisation functions for each method
  fcns <- list('CSS'=.css.ggbr.obj,'Whittle'=.whittle.ggbr.obj,'QML'=.qml.ggbr.obj,'WLL'=.wll.ggbr.obj)

  if (opt_method[[1]]=='best') {
    fit <- .best_optim(initial_pars=pars, fcn=fcns[[method]], lb=lb, ub=ub, lb_finite=lb_finite, ub_finite=ub_finite, params=params, control=control)
  } else {
      fit <- .generic_optim_list(opt_method_list=opt_method, initial_pars=pars, fcn=fcns[[method]],
                                   lb=lb, ub=ub, lb_finite=lb_finite, ub_finite=ub_finite, params=params, control=control)
  }
  if (fit$convergence== -999) stop('Failed to converge.')

  hh <- fit$hessian

  # sigma2
  if (method=='WLL') {
    # adjust sigma2 for theoretical bias...
    sigma2 <- fit$par[length(fit$par)] <- fit$par[length(fit$par)]/(2*pi) * exp(-digamma(1))
  }
  else if (method=='QML')     sigma2 <- .qml.ggbr.se2(fit$par, params=params)
  else if (method=='CSS')     sigma2 <- fit$value[length(fit$value)]/length(y)
  else if (method=='Whittle') sigma2 <- ifelse(no_scaling,1,sd_y) * .whittle.ggbr.obj.short(fit$par,params) * 4 * pi / length(y)  # 1997 Ferrara & Geugen eqn 3.7

  # log lik
  loglik <- numeric(0)
  if (method=='CSS')
    loglik <- -0.5* (fit$value[length(fit$value)]/sigma2 + length(y)*(log(2*pi) + log(sigma2)))
  if (method=='QML')
    loglik <- -fit$value[length(fit$value)]
  if (method=='Whittle')
    loglik <- fit$value[length(fit$value)] /(2*pi)

  se <- numeric(length(fit$par))

  # check convergence. Unfortunately "solnp" routine uses positive values to indicate an error.
  conv_ok <- TRUE
  if (opt_method[[1]]=='solnp') conv_ok <- (fit$convergence==0)
  else conv_of <- (fit$convergence>=0)

  if (conv_ok&method!='WLL'&!is.null(hh)) {
    # Next, find the se's for coefficients
    start<-1
    se<-c()   # default to set this up in the right environment
    # next check the hessian
    h_inv_diag <- diag(inv_hessian <- pracma::pinv(hh))
    if (!any(h_inv_diag<0)) { # if hessian doesn't look valid then... generate another one
      hh <- pracma::hessian(fcns[[method]], fit$par, params=params)
      h_inv_diag <- diag(inv_hessian <- pracma::pinv(hh))
    }
    if (method=='Whittle') {
      se <- suppressWarnings(sqrt(2*pi*h_inv_diag)) #var(y)/length(y) *
      vcov <- 2*pi*inv_hessian
    }
    if (method=='CSS') {
      se <- suppressWarnings(sqrt(h_inv_diag*sigma2*2))
      vcov <- inv_hessian*2*sigma2
    }
    if (method=='QML') {
      se <- suppressWarnings(sqrt(h_inv_diag*length(y)))
      vcov <- inv_hessian*length(y)
    }
  }
  if (method=='WLL') {
    se<-rep(NA,length(par))
    if (k==1) se[2] <- .wll_d_se(fit$par[1],ss)  # result only holds for k=1
    vcov <- matrix(nrow=length(fit$par),ncol=length(fit$par))
  }
  nm<-list()
  if (include.mean) nm <- c(nm,'intercept')

  if (k>0) nm<-c(nm,unlist(lapply(1:k,function(x) paste0(c('u','fd'),x))))
  if (p>0) nm<-c(nm,paste0('ar',1:p))
  if (q>0) nm<-c(nm,paste0('ma',1:q))

  n_coef    <- ifelse(method%in%methods_to_estimate_var,length(fit$par)-1,length(fit$par))  # ignore var on end if it is there
  temp_coef <- fit$par[1:n_coef]
  temp_se   <- se[1:n_coef]
  if (include.mean&!method%in%mean_methods) {# then add an Intercept anyway; use Kunsch 1987 result for se
    temp_coef <- c(mean(y),temp_coef)
    # calc se of mean using Kunsch (1987) thm 1; 1-H = 1-(d+.5) = .5-d
    if (k==0) mean_se <- sqrt(sigma2/length(y))
    else mean_se <- suppressWarnings(sqrt(sigma2/(length(y)^(0.5-fit$par[2]))))
    temp_se   <- c(mean_se, temp_se)
    n_coef <- n_coef+1
  }
  coef <- t(matrix(c(temp_coef,temp_se),nrow=n_coef))
  colnames(coef) <- nm
  rownames(coef) <- c('','s.e.')
  colnames(vcov) <- rownames(vcov) <- tail(nm,nrow(vcov))

  # build a ggbr_factors object
  gf <- list()
  if (k>0) {
    if (include.mean) start_idx <- 2
    else start_idx <- 1
    for (k1 in 1:k) {
      u <- coef[1,start_idx]
      gf1 <- list(u=u, f=acos(u)/(2*pi), fd=coef[1,start_idx+1], m=NA, f_idx=NA)
      gf <- c(gf,list(gf1))
      start_idx <- start_idx+2
    }
  }
  class(gf) <- 'ggbr_factors'

  # get fitted values and residuals
  fitted_and_resid <- .fitted_values(fit$par,params,gf,sigma2)
  fitted <- ts(fitted_and_resid$fitted, start=x_start,frequency=x_freq)
  resid <- ts(fitted_and_resid$residuals, start=x_start,frequency=x_freq)

  res<-list('call' = match.call(),
            'series' = deparse(match.call()$x),
            'coef'=coef,
            'var.coef'=vcov,
            'sigma2'=sigma2,
            'obj_value'=fit$value,
            'loglik'=loglik,
            'aic'=-2*loglik + 2*(n_coef+1),
            'convergence'=fit$convergence,
            'conv_message'=c(fit$message,message),
            'method'=method,
            'opt_method'=opt_method,
            'control'=control,
            'order'=order,
            'k'=k,
            'y'=x,
            'diff_y'=y,
            'y_start'=x_start,
            'y_end'=x_end,
            'y_freq'=x_freq,
            'include.mean'=include.mean,
            'include.drift'=include.drift,
            'fitted'=fitted,
            'residuals'=resid,
            fit_par = fit$par,
            params = params,
            'm_trunc'=m_trunc)
  if (opt_method[1]=='best') res<-c(res,'opt_method_selected'=fit$best_method)
  if (k>0) res<-c(res, 'ggbr_factors' = list(gf))

  class(res)<-c('garma_model','arima')

  return(res)
}




#' Predict future values.
#'
#' Predict ahead using algorithm of (2009) Godet, F
#' "Linear prediction of long-range dependent time series", ESAIM: PS 13 115-134.
#' DOI: 10.1051/ps:2008015
#'
#' @param object (garma_model) The garma_model from which to predict the values.
#' @param n.ahead (int) The number of time periods to predict ahead. Default: 1
#' @param ... Other parameters. Ignored.
#' @return A "ts" object containing the requested forecasts.
#' @examples
#' data(AirPassengers)
#' ap  <- as.numeric(diff(AirPassengers,12))
#' mdl <- garma(ap,order=c(9,1,0),k=0,method='CSS',include.mean=FALSE)
#' predict(mdl, n.ahead=12)
#' @export
predict.garma_model<-function(object,n.ahead=1,...) {
  ## Start of Function "predict"

  if (n.ahead<=0) {
    message('n.ahead must be g.t. 0.')
    return(NA)
  }

  coef <- unname(object$coef[1,])
  p  <- object$order[1]
  id <- object$order[2]
  q  <- object$order[3]
  k  <- object$k
  y  <- as.numeric(object$diff_y)
  orig_y <- as.numeric(object$y)
  n <- length(orig_y)
  resid  <- as.numeric(object$resid)

  mean_y <- beta0  <- 0
  start  <- 1
  if (object$include.mean&id==0) {
    mean_y <- mean(y)
    if (names(object$coef[1,])[1]=='intercept') {
      beta0  <- coef[1]
      start  <- 2
    } else beta0 <- mean_y
  }
  # jump over the ggbr params; we get those separately.
  start <- start + (k*2)

  phi_vec <- theta_vec <- ggbr_inv_vec <-  1
  if (p>0) phi_vec   <- c(1,-coef[start:(start+p-1)])
  if (q>0) theta_vec <- c(1,-coef[(p+start):(length(coef))])
  if (k>0) {
    for (gf in object$ggbr_factors) ggbr_inv_vec <- pracma::conv(ggbr_inv_vec,.ggbr.coef(n+n.ahead+3,-gf$fd,gf$u))
    # Next line multiplies and divides the various polynomials to get psi = theta * delta * ggbr / phi
    # pracma::conv gives polynomial multiplication, and pracma::deconv gives polynomial division.
    # we don't bother with the remainder. For non-ggbr models this may be a mistake.
    pi1 <- pracma::conv(phi_vec,ggbr_inv_vec)
    pi_vec  <- pracma::deconv(pi1,theta_vec)$q

    if (id==0) y_dash <- y-beta0 else y_dash <- y-mean_y
    for (h in 1:n.ahead) {
      yy <- y_dash
      vec <- pi_vec[(length(yy)+1):2]
      y_dash <- c(yy, -sum(yy*vec))
    }
    pred <- tail(y_dash,n.ahead)
  } else { # ARIMA forecasting only
    if (id==0) y_dash <- y-beta0 else y_dash <- y-mean_y
    phi_vec <- rev(-phi_vec[2:length(phi_vec)])
    theta_vec <- rev(-theta_vec[2:length(theta_vec)])
    # testing
    # if (q==2) theta_vec <- rev(c(0.2357262, -0.2934927))
    # if (p==2) phi_vec <- rev(c(0.2396479, -0.1674436))
    pp <- length(phi_vec)
    qq <- length(theta_vec)
    pred <- rep(beta0,n.ahead)

    for (i in 1:n.ahead) {
      if (p>0) {
        if (i>1) ar_vec <- tail(c(rep(0,pp),y_dash,pred[1:(i-1)]),pp)
        else ar_vec <- tail(c(rep(0,pp),y_dash),pp)
        pred[i] <- pred[i] + sum(phi_vec*ar_vec)
      }
      if (q>0) {
        if (i>qq) ma_vec <- rep(0,qq)
        else if (i>1) ma_vec <- tail(c(rep(0,qq),resid,rep(0,i-1)),qq)
        else ma_vec <- rep(0,qq)
        pred[i] <- pred[i] + sum(theta_vec*ma_vec)
      }
    }
  }

  if (id>0) {
    # print(tail(y,n.ahead))
    # print(round(pred,2))
    # print(mean(pred))
    if (object$include.drift) pred <- pred + mean(y)
    pred<-diffinv(pred,differences=id,xi=tail(orig_y,id))
    if (length(pred)>n.ahead) pred <- tail(pred,n.ahead)
    # last_y <- y
    # for (id1 in 1:id) {
    #   if (id1==id) dy <- as.numeric(orig_y)
    #   else dy <- diff(as.numeric(orig_y),differences=(id-id1))
    #   pred[1] <- pred[1] + as.numeric(dy[length(dy)])
    #   if (object$include.drift) pred <- pred + mean(last_y)
    #   pred <- cumsum(pred)
    #   last_y <- dy
    # }
  } else pred <- pred + beta0

  # Now we have the forecasts, we set these up as a "ts" object
  y_end = object$y_end
  if(length(y_end)>1) {
    if (object$y_freq >= y_end[2]) {
      y_end[1] <- y_end[1]+1
      y_end[2] <- y_end[2]-object$y_freq+1
    }
    else y_end[2] <- y_end[2] + 1
  } else y_end <- y_end +1

  res <- stats::ts(pred,start=y_end,frequency=object$y_freq)
  return(list(pred=res))
}


#' Forecast future values.
#'
#' The forecast function predicts future values of a "garma_model" object, and is exactly the same as the "predict" function with slightly different parameter values.
#' @param object (garma_model) The garma_model from which to forecast the values.
#' @param h (int) The number of time periods to predict ahead. Default: 1
#' @param ... Other parameters passed to the forecast function. For "garma_model" objects, these are ignored.
#' @return - a "ts" object containing the requested forecasts.
#' @examples
#' library(forecast)
#'
#' data(AirPassengers)
#' ap  <- as.numeric(diff(AirPassengers,12))
#' mdl <- garma(ap,order=c(9,1,0),k=0,method='CSS',include.mean=FALSE)
#' forecast(mdl, h=12)
#' @export
forecast.garma_model<-function(object,h=1,...) {
  res <- predict.garma_model(object,n.ahead=h)
  return(list(mean=res$pred))
}

.printf<-function(val) {
  if (class(val)=='integer') fmtstr <- '%s: %d\n'
  else fmtstr <- '%s: %f\n'
  cat(sprintf(fmtstr,as.character(substitute(val)),val))
}

.fitted_values<-function(par,params,ggbr_factors,sigma2) { # Generate fitted values and residuals for GARMA process
  y <- as.numeric(params$y)
  orig_y <- as.numeric(params$orig_y)
  p <- params$p
  q <- params$q
  id <- params$d
  k <- params$k
  include.mean <- params$include.mean
  method <- params$method

  beta0  <- 0
  start  <- 1
  if (include.mean) {
    if (method%in%c('CSS','QML')) {
      beta0  <- par[1]
      start <- 2
      } else beta0 <- mean(y)
  }

  # skip over Gegenbauer parameters
  start <- start + k*2

  n       <- length(y)
  phi_vec <- theta_vec <- 1
  if (p>0) phi_vec   <- par[start:(start+p-1)]
  if (q>0) theta_vec <- par[(p+start):(length(par))]
  # testing
  # if (q==2) theta_vec <- c(0.2357262, -0.2934927)
  # if (p==2) phi_vec <- c(0.2396479, -0.1674436)

  # now we generate polynomial coefficients for MA part to include ggbr factors
  # basic tool is pracma::conv which does polynomial multiplication (same as polymul())
  qk <- q
  if (k>0) {
    if (length(theta_vec)>1|theta_vec[1]!=1) theta_vec <- c(1,theta_vec)
    for (gf in ggbr_factors) {
      gc <- .ggbr.coef(n,gf$fd,gf$u)
      theta_vec <- pracma::conv(theta_vec,gc)
    }
    if (theta_vec[1]==1) theta_vec <- theta_vec[2:min(length(theta_vec),n)]
    qk <- length(theta_vec)
  }

  phi_vec   <- rev(phi_vec)
  theta_vec <- rev(theta_vec)

  resid   <- rep(0,p)
  for (i in (p+1):n) {
    s <- beta0
    if (p>0) {
      ar_vec <- tail(c(rep(0,p),y[1:(i-1)]),p)
      s <- s + sum(ar_vec*phi_vec)
    }
    if (qk>0) {
      ma_vec <- tail(c(rep(0,qk),resid),qk)
      s <- s + sum(theta_vec*ma_vec)
    }
    resid <- c(resid,y[i]-s)
  }

  if (id>0) resid <- c(rep(0,id),resid)
  fitted <- orig_y - resid

  return(list(fitted=fitted,residuals=resid))
}

# .fitted_values<-function(par,params,ggbr_factors) { # Generate fitted values and residuals for GARMA process
#   y <- as.numeric(params$y)
#   orig_y <- as.numeric(params$orig_y)
#   p <- params$p
#   q <- params$q
#   id <- params$d
#   k <- params$k
#   include.mean <- params$include.mean
#   method <- params$method
#
#   beta0  <- 0
#   start  <- 1
#   if (include.mean) {
#     if (method%in%c('CSS','QML')) {
#       beta0  <- par[1]
#       start <- 2
#     } else beta0 <- mean(y)
#   }
#
#   # skip over Gegenbauer paramaters
#   start <- start + k*2
#
#   y_dash  <- y - beta0
#   n       <- length(y_dash)
#   phi_vec <- theta_vec <- 1
#   if (p>0) phi_vec   <- par[start:(start+p-1)]
#   if (q>0) theta_vec <- par[(p+start):(length(par))]
#   if (q==2) theta_vec <- c(0.2357262, -0.2934927)
#   print('params')
#   .printf(include.mean)
#   .printf(p)
#   .printf(q)
#   .printf(start)
#   print(theta_vec)
#   print(par)
#   print(" ")
#
#   # now we generate polynomial coefficients for MA part to include ggbr factors
#   # basic tool is pracma::conv which does polynomial multiplication (same as polymul())
#   if (k>0) {
#     print(theta_vec)
#     if (length(theta_vec)>1|theta_vec[1]!=1) theta_vec <- c(1,theta_vec)
#     print(theta_vec)
#     for (gf in ggbr_factors) {
#       gc <- .ggbr.coef(n,gf$fd,gf$u)
#       theta_vec <- pracma::conv(theta_vec,gc)
#     }
#     print(head(theta_vec,10))
#   }
#   if (p>0) resid <- rep(0,p) else resid <- numeric(0)
#   qk <- ifelse(k>0,n,q)
#   theta_vec_rev <- rev(theta_vec)[1:qk]
#   flt <- signal::Arma(a=1, b=phi_vec)
#   for (i in (p+1):n) {
#     fitted_i <- as.numeric(y_dash[1:(i-1)])
#     # if (p>0) fitted_i <- stats::filter(fitted_i, filter=phi_vec, method='convolution', sides=1)
#     if (p>0) fitted_i <- signal::filter(flt,fitted_i)
#     if (qk>0) {
#       ma_vec <- tail(c(rep(0,qk),resid),qk)
#       ma_resid <- sum(theta_vec_rev*ma_vec)
#       #   if (i<p+3) {
#       #     .printf(i)
#       #     .printf(qk)
#       #     print(head(ma_vec))
#       #     print(head(theta_vec_rev))
#       #     .printf(ma_resid)
#       #   }
#     } #else ma_resid <- 0
#     if (p>0) ar_resid <- y_dash[i]-tail(fitted_i,1) else ar_resid<-0
#     resid <- c(resid,-ma_resid+ar_resid)
#   }
#
#   fitted <- y - resid
#   if (id>0) {
#     for (id1 in 1:id) {
#       if (id1==id) dy <- as.numeric(orig_y)
#       else dy <- diff(as.numeric(orig_y),differences=(id-id1))
#       fitted <- dy - c(0,resid)
#       resid <- dy-fitted
#     }
#   }
#
#   return(list(residuals=resid,fitted=fitted))
# }

#' Fitted values
#'
#' Fitted values are 1-step ahead predictions.
#' @param object The garma_model object
#' @param ... Other parameters. Ignored.
#' @export
fitted.garma_model<-function(object,...) {
  return(object$fitted)
}


#' Residuals
#'
#' Response Residuals from the model.
#' @param object The garma_model object
#' @param type (chr) The type of residuals. Must be 'response'.
#' @param h (int) The number of periods ahead for the residuals. Must be 1.
#' @param ... Other parameters. Ignored.
#' @export
residuals.garma_model<-function(object,type='response',h=1,...) {
  if (!missing(type)) {
    if (type!='response') stop('Only response residuals are available.')
  }
  if (!missing(h))
    if (h!=1) stop('Only h=1 response residuals are available.')
  return(object$residuals)
}

#' Model Coefficients
#'
#' Model Coefficients/parameters.
#' @param object The garma_model object
#' @param ... Other parameters. Ignored.
#' @export
coef.garma_model<-function(object,...) {
  return(object$coef[1,])
}

#' AIC for model
#'
#' AIC for model if available.
#' @param object The garma_model object
#' @param ... Other parameters. Ignored.
#' @export
AIC.garma_model<-function(object,...) {
  return(object$aic)
}

#' Covariance matrix
#'
#' Covariance matrix of parameters if available
#' @param object The garma_model object
#' @param ... Other parameters. Ignored.
#' @export
vcov.garma_model<-function(object,...) {
  return(object$var.coef)
}

#' Log Likelihood
#'
#' Log Likelihood, or approximate likelihood or part likelihood, depending on the method.
#' @param object The garma_model object
#' @param ... Other parameters. Ignored.
#' @export
logLik.garma_model<-function(object,...) {
  # Need to figure out how to indicate these are REML estimates not true LL.
  res <-  structure(object$loglik, df=length(object$y)-1, nobs=length(object$y), class="logLik")
  return(res)
}

#' garma package version
#'
#' The version function returns the garma package version.
#' @examples
#' library(garma)
#' garma::version()
#' @export
version<-function() {message(.getPackageVersion('garma'))}

