test_that("galamm reproduces gamm4", {
  dat <- subset(cognition, domain == 1 & item == "11" & id < 50)

  mod <- galamm(formula = y ~ s(x), data = dat)
  # Not running gamm4 in tests, but taking outputs from it
  # mod_comp <- gamm4::gamm4(formula = y ~ s(x), data = dat, REML = FALSE)

  expect_equal(unname(mod$gam$edf), c(
    1, 0.917922212177643, 0.835640672846923, 0.473700240730942,
    -0.0205750525522368, 0.170693269673505, -0.340640339069972, 0.0486758453821619,
    1.58020490382619, 1.00000000000004
  ), tolerance = .001)

  expect_snapshot(print(summary(mod$gam), digits = 2))
  expect_type(plot_smooth(mod), "list")

  expect_equal(head(predict(mod)), structure(c(
    `1` = 0.432267467571946, `4` = 0.850917209622746,
    `7` = 2.22066429268562, `10` = 2.22494341092419, `13` = 1.74697112222858,
    `16` = 1.05407575060804
  ), dim = 6L, dimnames = list(c(
    "1", "4",
    "7", "10", "13", "16"
  ))), tolerance = .01)

  mod <- galamm(formula = y ~ t2(x), data = dat)
  # mod_comp <- gamm4::gamm4(formula = y ~ t2(x), data = dat, REML = FALSE)
  expect_equal(unname(mod$gam$edf), c(
    1, 0.778128590494658, 0.921997998438558, 0.987589732872169,
    0.999999999999999
  ), tolerance = .001)

  expect_snapshot(print(summary(mod$gam), digits = 2))
  expect_equal(head(predict(mod)), structure(c(
    `1` = 0.430919582024448, `4` = 0.845539207864478,
    `7` = 2.25395029739734, `10` = 2.25707170189091, `13` = 1.75570899492922,
    `16` = 1.05353978411553
  ), dim = 6L, dimnames = list(c(
    "1", "4",
    "7", "10", "13", "16"
  ))), tolerance = .01)

  mod <- galamm(formula = y ~ s(x, fx = TRUE) + (1 | id), data = dat)
  # mod_comp <- gamm4::gamm4(
  #   formula = y ~ s(x, fx = TRUE),
  #   random = ~ (1 | id), data = dat, REML = FALSE
  # )

  expect_equal(deviance(mod), 712.425690398506, tolerance = .0001)
  expect_snapshot(print(summary(mod$gam), digits = 2))
  expect_equal(head(predict(mod)), structure(c(
    `1` = 0.599156451545575, `4` = 1.0293037482984, `7` = 2.04517471008913,
    `10` = 2.04799855501091, `13` = 1.71247171001133, `16` = 1.02413440845315
  ), dim = 6L, dimnames = list(c("1", "4", "7", "10", "13", "16"))), tolerance = .01)

  set.seed(1)
  dat <- mgcv::gamSim(verbose = FALSE, scale = .1)
  # mod0 <- mgcv::gamm(y ~ s(x1) + t2(x2), data = dat, method = "ML")
  mod1 <- galamm(y ~ s(x1) + t2(x2), data = dat)
  expect_equal(998.321391967764, deviance(mod1),
    tolerance = .0001
  )
  expect_equal(structure(c(
    `1` = 6.28914275751487, `2` = 3.06558132814435, `3` = 7.29524646673424,
    `4` = 9.79229774562216, `5` = 15.9499391403392, `6` = 8.10716399874615
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))), head(predict(mod1)), tolerance = .01)

  set.seed(1)
  dat <- mgcv::gamSim(4, verbose = FALSE)
  # mod0 <- gamm4::gamm4(y ~ fac + s(x2, by = fac), data = dat, REML = FALSE)
  mod1 <- galamm(formula = y ~ fac + s(x2, by = fac), data = dat)
  expect_equal(1782.3083187478, deviance(mod1), tolerance = .0001)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(mod1$gam$edf, c(`(Intercept)` = 1, fac2 = 1, fac3 = 1, `s(x2):fac1.1` = 0.396403284991457,
                               `s(x2):fac1.2` = 0.0205790695713345, `s(x2):fac1.3` = 0.0442962849174009,
                               `s(x2):fac1.4` = -0.17418200340773, `s(x2):fac1.5` = 0.00955739598405336,
                               `s(x2):fac1.6` = -0.195958664896826, `s(x2):fac1.7` = 0.0069464792646512,
                               `s(x2):fac1.8` = 1.3316327257046, `s(x2):fac1.9` = 0.999999999999998,
                               `s(x2):fac2.1` = 0.274720853831099, `s(x2):fac2.2` = -0.0224772222135663,
                               `s(x2):fac2.3` = 0.028286927385353, `s(x2):fac2.4` = -0.147016799821575,
                               `s(x2):fac2.5` = 0.00127220572737463, `s(x2):fac2.6` = -0.16295132045179,
                               `s(x2):fac2.7` = 0.00115382656098317, `s(x2):fac2.8` = 1.17044680469749,
                               `s(x2):fac2.9` = 1.00000000000001, `s(x2):fac3.1` = 1.00592642033534,
                               `s(x2):fac3.2` = 1.0632747514456, `s(x2):fac3.3` = 0.856507781762346,
                               `s(x2):fac3.4` = 0.686913401758021, `s(x2):fac3.5` = 0.58698383464359,
                               `s(x2):fac3.6` = 0.125572788111435, `s(x2):fac3.7` = 0.323436341176489,
                               `s(x2):fac3.8` = 1.33079782332557, `s(x2):fac3.9` = 1.00000000000017
  ), tolerance = .0001)
  expect_equal(structure(c(`1` = 1.51695601325662, `2` = 1.00083072197106, `3` = -0.453952988244748,
                           `4` = 2.67159496409, `5` = 8.76001592526375, `6` = -0.046108810500684
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))), head(predict(mod1)), tolerance = .01)
  nd <- data.frame(fac = head(dat$fac, 6), x2 = runif(6))
  expect_equal(structure(c(`1` = 2.28480127096885, `2` = 2.35443816910562, `3` = 2.68408223952036,
                           `4` = 8.3650291501216, `5` = 8.0091923451643, `6` = 1.7690544424336
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .001
  )

  # mod0 <- gamm4::gamm4(y ~ s(x0, by = x2), data = dat, REML = FALSE)
  mod1 <- galamm(y ~ s(x0, by = x2), data = dat)
  expect_equal(c(`(Intercept)` = 1.2330344323451, `s(x0):x2.1` = 0, `s(x0):x2.2` = 0,
                 `s(x0):x2.3` = 0, `s(x0):x2.4` = 0, `s(x0):x2.5` = 0, `s(x0):x2.6` = 0,
                 `s(x0):x2.7` = 0, `s(x0):x2.8` = 0, `s(x0):x2.9` = 0.255340763016034,
                 `s(x0):x2.10` = -0.765892955579749), mod1$gam$coefficients, tolerance = .0001)
  expect_equal(2130.12383920738, deviance(mod1), tolerance = .0001)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  nd <- data.frame(x0 = runif(6), x2 = runif(6))
  expect_equal(structure(c(`1` = 2.04741395992067, `2` = 0.907179644677066,
                           `3` = 0.282280428906123, `4` = 1.77839797103925, `5` = 1.11818989108203,
                           `6` = 1.72900460806641), dim = 6L, dimnames = list(c("1", "2",
                                                                                "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .01
  )

  # mod0 <- gamm4::gamm4(y ~ t2(x0, by = x2), data = dat, REML = FALSE)
  mod1 <- galamm(y ~ t2(x0, by = x2), data = dat)
  expect_equal(c(`(Intercept)` = 1.22018159241993, `t2(x0):x2.1` = 0.016546928343366,
                 `t2(x0):x2.2` = -0.0204655262986532, `t2(x0):x2.3` = 0.392312044072024,
                 `t2(x0):x2.4` = -0.491270165993921, `t2(x0):x2.5` = -0.944967059967863
  ), mod1$gam$coefficients, tolerance = .0001)
  expect_equal(2127.73121634104, deviance(mod1), tolerance = .0001)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(structure(c(`1` = 2.29593962104184, `2` = 1.02796921385926, `3` = 1.04248798784385,
                           `4` = 1.74961861242407, `5` = 1.14552081164797, `6` = 1.9700177729132
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .01
  )

  set.seed(1)
  dat <- mgcv::gamSim(1, verbose = FALSE, dist = "binary")
  # mod0 <- gamm4::gamm4(y ~ s(x0), family = binomial, data = dat)
  mod1 <- galamm(y ~ s(x0), family = binomial, data = dat)
  expect_equal(412.12442634643, deviance(mod1), tolerance = .001)
  expect_equal(structure(c(`1` = 1.28525640324483, `2` = 1.09236039864884, `3` = 0.832553984375998,
                           `4` = 1.38081563301891, `5` = 1.11957842538751, `6` = 1.22135585234529
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 0.783343204044773, `2` = 0.748825939421105,
                           `3` = 0.696894685416063, `4` = 0.799121962541592, `5` = 0.753910510314315,
                           `6` = 0.772302066234553), dim = 6L, dimnames = list(c("1", "2",
                                                                                 "3", "4", "5", "6"))),
    predict(mod1, newdata = nd, type = "response"),
    tolerance = .001
  )
  expect_equal(structure(c(`1` = 0.797449418590645, `2` = 0.814101045678496,
                           `3` = 0.819375295051365, `4` = 0.719391319182074, `5` = 0.784607350520732,
                           `6` = 0.724790688221669, `7` = 0.69827327287264, `8` = 0.808074474630294,
                           `9` = 0.813093048963882, `10` = 0.755788142920408, `11` = 0.785513570273205,
                           `12` = 0.77925403576986, `13` = 0.803052441781975, `14` = 0.815514974973666,
                           `15` = 0.781095857864995, `16` = 0.822474824095721, `17` = 0.796094348284374,
                           `18` = 0.669013378609681, `19` = 0.815046633984512, `20` = 0.778550622231617,
                           `21` = 0.704198064639782, `22` = 0.786808551245264, `23` = 0.809633290474455,
                           `24` = 0.768489384601581, `25` = 0.797769112624326, `26` = 0.815741787509488,
                           `27` = 0.746379176809368, `28` = 0.815318980823032, `29` = 0.739759162956013,
                           `30` = 0.809868012184242, `31` = 0.822319970952501, `32` = 0.81680485462742,
                           `33` = 0.822460840866064, `34` = 0.781318647835277, `35` = 0.759379936704651,
                           `36` = 0.806690588781044, `37` = 0.772582497290114, `38` = 0.764896301994907,
                           `39` = 0.794556040321225, `40` = 0.818307350581371, `41` = 0.762090317054461,
                           `42` = 0.810386018298234, `43` = 0.776653421860502, `44` = 0.820795366931099,
                           `45` = 0.821921168437902, `46` = 0.774367619572183, `47` = 0.748321524946484,
                           `48` = 0.822215392679399, `49` = 0.79229261768122, `50` = 0.801847544330766,
                           `51` = 0.822224774295104, `52` = 0.743933794251506, `53` = 0.820428313613501,
                           `54` = 0.793463950360913, `55` = 0.757518858345003, `56` = 0.763193815992607,
                           `57` = 0.806231925761394, `58` = 0.822244600323395, `59` = 0.807861127408733,
                           `60` = 0.817892517476302, `61` = 0.716778209474735, `62` = 0.802492092832199,
                           `63` = 0.821590419383973, `64` = 0.808705998067353, `65` = 0.809766050992158,
                           `66` = 0.796032374089641, `67` = 0.822246379118504, `68` = 0.782245080569899,
                           `69` = 0.76017629841794, `70` = 0.736923222731611, `71` = 0.809684252261209,
                           `72` = 0.754097000445812, `73` = 0.810764975887488, `74` = 0.808910446257994,
                           `75` = 0.822193600160589, `76` = 0.728125360919453, `77` = 0.742406935745166,
                           `78` = 0.816170265312917, `79` = 0.778593083206735, `80` = 0.688597589816085,
                           `81` = 0.820194876900893, `82` = 0.797342741266104, `83` = 0.817221527026536,
                           `84` = 0.807644673698688, `85` = 0.78515115548093, `86` = 0.784820968754669,
                           `87` = 0.797677292922172, `88` = 0.767694642418135, `89` = 0.793600287376763,
                           `90` = 0.772186126607975, `91` = 0.792437950547182, `92` = 0.755234278132682,
                           `93` = 0.811139909540634, `94` = 0.736440814478393, `95` = 0.778047579132028,
                           `96` = 0.771439690691857, `97` = 0.821414494536697, `98` = 0.81819792546169,
                           `99` = 0.766194582434525, `100` = 0.816198531318197, `101` = 0.809122626895352,
                           `102` = 0.811660440415894, `103` = 0.798332830709319, `104` = 0.668517907419182,
                           `105` = 0.812464513395373, `106` = 0.787031322525275, `107` = 0.769278262004222,
                           `108` = 0.822236527802437, `109` = 0.710395749262246, `110` = 0.816893188467776,
                           `111` = 0.678950766552969, `112` = 0.79243284440341, `113` = 0.812134047492272,
                           `114` = 0.819968146754535, `115` = 0.77321999485083, `116` = 0.746317916812836,
                           `117` = 0.796600746848592, `118` = 0.763938412110156, `119` = 0.820936405317795,
                           `120` = 0.811477144055524, `121` = 0.669056334233569, `122` = 0.822470219183823,
                           `123` = 0.822359749388083, `124` = 0.778588211311509, `125` = 0.785844183028847,
                           `126` = 0.821346694904479, `127` = 0.822384091935509, `128` = 0.78584415919249,
                           `129` = 0.790223037578644, `130` = 0.817221701572994, `131` = 0.819207177692536,
                           `132` = 0.758766412921057, `133` = 0.750696555059156, `134` = 0.811060957983884,
                           `135` = 0.707764362689181, `136` = 0.816966049326643, `137` = 0.820282073160907,
                           `138` = 0.822044231256347, `139` = 0.673334771079182, `140` = 0.822427926519615,
                           `141` = 0.803919758453564, `142` = 0.816585196370178, `143` = 0.792285942254912,
                           `144` = 0.796060865388623, `145` = 0.793095314835375, `146` = 0.821279663332406,
                           `147` = 0.778948305091723, `148` = 0.788261437075802, `149` = 0.764300731538844,
                           `150` = 0.742306134378321, `151` = 0.815025100005455, `152` = 0.820534554103578,
                           `153` = 0.808164634443984, `154` = 0.821308259002428, `155` = 0.82247320187908,
                           `156` = 0.780175391831699, `157` = 0.821924313783279, `158` = 0.758416455440296,
                           `159` = 0.799701577148661, `160` = 0.78692503154494, `161` = 0.80095778520825,
                           `162` = 0.726572623633636, `163` = 0.820933566724293, `164` = 0.777678917314832,
                           `165` = 0.734208877978221, `166` = 0.818474923050879, `167` = 0.756181166554986,
                           `168` = 0.809162482210761, `169` = 0.794552187274747, `170` = 0.80947310145682,
                           `171` = 0.812908590160307, `172` = 0.753569462486789, `173` = 0.746376171497972,
                           `174` = 0.816318912139754, `175` = 0.815100055126883, `176` = 0.726383408328665,
                           `177` = 0.810822643150833, `178` = 0.789873722123906, `179` = 0.816155611738058,
                           `180` = 0.722227081350931, `181` = 0.802513841440028, `182` = 0.782394169827704,
                           `183` = 0.731170347275621, `184` = 0.822462118172887, `185` = 0.736038523382703,
                           `186` = 0.78195367966991, `187` = 0.784837814322412, `188` = 0.794353209617323,
                           `189` = 0.698842501807608, `190` = 0.821108542858073, `191` = 0.7975281343194,
                           `192` = 0.816051538831885, `193` = 0.76347520637567, `194` = 0.70852779850509,
                           `195` = 0.800681958460973, `196` = 0.817753477136556, `197` = 0.765384860622741,
                           `198` = 0.753617868186109, `199` = 0.80649891532452, `200` = 0.77668194581422,
                           `201` = 0.797822653464654, `202` = 0.788163077045424, `203` = 0.822284810801491,
                           `204` = 0.798090562874097, `205` = 0.780239937012408, `206` = 0.822245930968875,
                           `207` = 0.820149378711718, `208` = 0.769233633957016, `209` = 0.795716585832203,
                           `210` = 0.796015664764625, `211` = 0.688111049412366, `212` = 0.763328695768636,
                           `213` = 0.783233407047392, `214` = 0.696295338710826, `215` = 0.763047551103247,
                           `216` = 0.804949871556789, `217` = 0.809977911172905, `218` = 0.693034139127361,
                           `219` = 0.692804869387135, `220` = 0.809814864950355, `221` = 0.796878955915421,
                           `222` = 0.776881995557376, `223` = 0.807154932705336, `224` = 0.82239855220751,
                           `225` = 0.710456885484131, `226` = 0.82238710023245, `227` = 0.795956763665716,
                           `228` = 0.752814995768604, `229` = 0.818889721536636, `230` = 0.747387885807141,
                           `231` = 0.810841256459946, `232` = 0.769707513085345, `233` = 0.814388228633566,
                           `234` = 0.81276460352483, `235` = 0.81618001096406, `236` = 0.802507750982556,
                           `237` = 0.802552912205818, `238` = 0.82067971272429, `239` = 0.81983200542792,
                           `240` = 0.821287307778238, `241` = 0.804647411805795, `242` = 0.818907139365523,
                           `243` = 0.718184379395221, `244` = 0.772038962829785, `245` = 0.818645947720924,
                           `246` = 0.786553818762817, `247` = 0.819766512610857, `248` = 0.769966730805847,
                           `249` = 0.821635573242944, `250` = 0.69930165464536, `251` = 0.783628683758116,
                           `252` = 0.705253750968449, `253` = 0.822032088037649, `254` = 0.816353307832278,
                           `255` = 0.822369911491476, `256` = 0.765070517657637, `257` = 0.794040271033994,
                           `258` = 0.822475242001252, `259` = 0.814191754089163, `260` = 0.704206228301842,
                           `261` = 0.822105719325606, `262` = 0.806369890694492, `263` = 0.799739495820516,
                           `264` = 0.775020892143837, `265` = 0.799696838865592, `266` = 0.776791054693099,
                           `267` = 0.756307371921768, `268` = 0.785879230915544, `269` = 0.814281774988892,
                           `270` = 0.777762166264419, `271` = 0.755871261967359, `272` = 0.765115571384655,
                           `273` = 0.815241659993716, `274` = 0.777705379317867, `275` = 0.803351075196713,
                           `276` = 0.782596399367941, `277` = 0.795870405221158, `278` = 0.780253508808001,
                           `279` = 0.822217421501416, `280` = 0.780801067642986, `281` = 0.749189459408778,
                           `282` = 0.822003193954564, `283` = 0.734363745129044, `284` = 0.814215703744451,
                           `285` = 0.753105478157563, `286` = 0.771205373575972, `287` = 0.807050181035845,
                           `288` = 0.77462133190927, `289` = 0.76987074207442, `290` = 0.788713688592572,
                           `291` = 0.78975751922115, `292` = 0.769702159412253, `293` = 0.675563733457169,
                           `294` = 0.807897801334724, `295` = 0.8224349548872, `296` = 0.804190683803057,
                           `297` = 0.763134462471642, `298` = 0.767123099001871, `299` = 0.753586412017629,
                           `300` = 0.707392295749603, `301` = 0.805704328913309, `302` = 0.762275231157787,
                           `303` = 0.822454898844089, `304` = 0.821697372470971, `305` = 0.814476097149143,
                           `306` = 0.669526863626396, `307` = 0.779209983669044, `308` = 0.765166595119479,
                           `309` = 0.757083752355132, `310` = 0.817267442446577, `311` = 0.771732522207763,
                           `312` = 0.782830642394835, `313` = 0.753237059079651, `314` = 0.795520953093404,
                           `315` = 0.797767513266535, `316` = 0.822468009491087, `317` = 0.759953367091411,
                           `318` = 0.811753264102085, `319` = 0.683292258520114, `320` = 0.813703976629063,
                           `321` = 0.807393587621049, `322` = 0.805628963193499, `323` = 0.817783306599863,
                           `324` = 0.666352405876342, `325` = 0.746875492598037, `326` = 0.692916879176545,
                           `327` = 0.765620936402528, `328` = 0.776915952804563, `329` = 0.797891469293347,
                           `330` = 0.783572603333147, `331` = 0.672565190245094, `332` = 0.802492466233459,
                           `333` = 0.817156375560821, `334` = 0.76569051901693, `335` = 0.758783502287153,
                           `336` = 0.81304471263331, `337` = 0.820715730008589, `338` = 0.775021019294349,
                           `339` = 0.818560401818885, `340` = 0.68269988038025, `341` = 0.6705434868082,
                           `342` = 0.779231646944954, `343` = 0.82139628903212, `344` = 0.815537696434354,
                           `345` = 0.805232082295829, `346` = 0.798154142132752, `347` = 0.821985791024245,
                           `348` = 0.778237204287174, `349` = 0.813738869604233, `350` = 0.794118783458201,
                           `351` = 0.822387129139239, `352` = 0.756179997351547, `353` = 0.776085885316198,
                           `354` = 0.818929424125342, `355` = 0.675907182704931, `356` = 0.800620064587514,
                           `357` = 0.750251540172676, `358` = 0.759781410253252, `359` = 0.731166244645063,
                           `360` = 0.822070833911554, `361` = 0.765130061942365, `362` = 0.808836972110633,
                           `363` = 0.75500063639882, `364` = 0.799537763855241, `365` = 0.818103585724176,
                           `366` = 0.755304578451073, `367` = 0.757611497901203, `368` = 0.799624857164381,
                           `369` = 0.800515396543034, `370` = 0.82179466168814, `371` = 0.82035045004056,
                           `372` = 0.820192525879794, `373` = 0.707841181293022, `374` = 0.790591323741441,
                           `375` = 0.788818527164227, `376` = 0.819088987453837, `377` = 0.808872892554757,
                           `378` = 0.742177209110683, `379` = 0.779390397714587, `380` = 0.822459533694458,
                           `381` = 0.819838630061534, `382` = 0.820042313152159, `383` = 0.808878184799348,
                           `384` = 0.677456349923036, `385` = 0.790935370379879, `386` = 0.792673681099844,
                           `387` = 0.77161678673692, `388` = 0.757527094987, `389` = 0.766023169480773,
                           `390` = 0.686940335690427, `391` = 0.773032436382862, `392` = 0.772253944492227,
                           `393` = 0.709728448940366, `394` = 0.822434026457272, `395` = 0.774625449558231,
                           `396` = 0.810990052741657, `397` = 0.808245863607786, `398` = 0.80551365087714,
                           `399` = 0.811439789859273, `400` = 0.773142761645541), dim = 400L, dimnames = list(
                             c("1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11",
                               "12", "13", "14", "15", "16", "17", "18", "19", "20", "21",
                               "22", "23", "24", "25", "26", "27", "28", "29", "30", "31",
                               "32", "33", "34", "35", "36", "37", "38", "39", "40", "41",
                               "42", "43", "44", "45", "46", "47", "48", "49", "50", "51",
                               "52", "53", "54", "55", "56", "57", "58", "59", "60", "61",
                               "62", "63", "64", "65", "66", "67", "68", "69", "70", "71",
                               "72", "73", "74", "75", "76", "77", "78", "79", "80", "81",
                               "82", "83", "84", "85", "86", "87", "88", "89", "90", "91",
                               "92", "93", "94", "95", "96", "97", "98", "99", "100", "101",
                               "102", "103", "104", "105", "106", "107", "108", "109", "110",
                               "111", "112", "113", "114", "115", "116", "117", "118", "119",
                               "120", "121", "122", "123", "124", "125", "126", "127", "128",
                               "129", "130", "131", "132", "133", "134", "135", "136", "137",
                               "138", "139", "140", "141", "142", "143", "144", "145", "146",
                               "147", "148", "149", "150", "151", "152", "153", "154", "155",
                               "156", "157", "158", "159", "160", "161", "162", "163", "164",
                               "165", "166", "167", "168", "169", "170", "171", "172", "173",
                               "174", "175", "176", "177", "178", "179", "180", "181", "182",
                               "183", "184", "185", "186", "187", "188", "189", "190", "191",
                               "192", "193", "194", "195", "196", "197", "198", "199", "200",
                               "201", "202", "203", "204", "205", "206", "207", "208", "209",
                               "210", "211", "212", "213", "214", "215", "216", "217", "218",
                               "219", "220", "221", "222", "223", "224", "225", "226", "227",
                               "228", "229", "230", "231", "232", "233", "234", "235", "236",
                               "237", "238", "239", "240", "241", "242", "243", "244", "245",
                               "246", "247", "248", "249", "250", "251", "252", "253", "254",
                               "255", "256", "257", "258", "259", "260", "261", "262", "263",
                               "264", "265", "266", "267", "268", "269", "270", "271", "272",
                               "273", "274", "275", "276", "277", "278", "279", "280", "281",
                               "282", "283", "284", "285", "286", "287", "288", "289", "290",
                               "291", "292", "293", "294", "295", "296", "297", "298", "299",
                               "300", "301", "302", "303", "304", "305", "306", "307", "308",
                               "309", "310", "311", "312", "313", "314", "315", "316", "317",
                               "318", "319", "320", "321", "322", "323", "324", "325", "326",
                               "327", "328", "329", "330", "331", "332", "333", "334", "335",
                               "336", "337", "338", "339", "340", "341", "342", "343", "344",
                               "345", "346", "347", "348", "349", "350", "351", "352", "353",
                               "354", "355", "356", "357", "358", "359", "360", "361", "362",
                               "363", "364", "365", "366", "367", "368", "369", "370", "371",
                               "372", "373", "374", "375", "376", "377", "378", "379", "380",
                               "381", "382", "383", "384", "385", "386", "387", "388", "389",
                               "390", "391", "392", "393", "394", "395", "396", "397", "398",
                               "399", "400"))),
    predict(mod1, type = "response"),
    tolerance = .001
  )

  # mod0 <- gamm4::gamm4(y ~ s(x0, by = x2), data = dat, family = binomial)
  mod1 <- galamm(y ~ s(x0, by = x2), data = dat, family = binomial)
  expect_equal(c(`(Intercept)` = 2.83510694726345, `s(x0):x2.1` = 0, `s(x0):x2.2` = 0,
                 `s(x0):x2.3` = 0, `s(x0):x2.4` = 0, `s(x0):x2.5` = 0, `s(x0):x2.6` = 0,
                 `s(x0):x2.7` = 0, `s(x0):x2.8` = 0, `s(x0):x2.9` = -2.81863263565184,
                 `s(x0):x2.10` = -0.0806257159389692), mod1$gam$coefficients, tolerance = .01)
  expect_equal(376.071953181812, deviance(mod1), tolerance = .001)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(structure(c(`1` = 0.682793746204226, `2` = 0.822203086350053,
                           `3` = 0.504081881835665, `4` = 0.747380899574395, `5` = 0.913415166891762,
                           `6` = 0.849320461742116), dim = 6L, dimnames = list(c("1", "2",
                                                                                 "3", "4", "5", "6"))),
    predict(mod1, newdata = nd, type = "response"),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 0.615750845407161, `2` = 0.939303241178068,
                           `3` = 0.518945470690856, `4` = 0.655931163619005, `5` = 0.889734008979694,
                           `6` = 0.699939195775794, `7` = 0.859216868459989, `8` = 0.529910529902621,
                           `9` = 0.866146641271373, `10` = 0.939895863688225, `11` = 0.86622237976635,
                           `12` = 0.855639232713986, `13` = 0.859234010265614, `14` = 0.539849446304111,
                           `15` = 0.84847635412674, `16` = 0.784906273492274, `17` = 0.541588343366046,
                           `18` = 0.529508471440229, `19` = 0.896498672802649, `20` = 0.68021781983543,
                           `21` = 0.858982539581956, `22` = 0.788851022746616, `23` = 0.669694634555337,
                           `24` = 0.648433106778445, `25` = 0.608479310222666, `26` = 0.939068160828505,
                           `27` = 0.904268754206907, `28` = 0.859670876649264, `29` = 0.874360492965642,
                           `30` = 0.693372048978937, `31` = 0.703728095680366, `32` = 0.560188422504612,
                           `33` = 0.611022879916639, `34` = 0.670414248766013, `35` = 0.841098311540818,
                           `36` = 0.764946737103888, `37` = 0.93170695264326, `38` = 0.615889902818874,
                           `39` = 0.51893136766619, `40` = 0.805689143282972, `41` = 0.53538646693893,
                           `42` = 0.920558706690678, `43` = 0.894931339708313, `44` = 0.809319507698393,
                           `45` = 0.816663058341661, `46` = 0.828019129094513, `47` = 0.938764154200294,
                           `48` = 0.735359303527053, `49` = 0.683210258930132, `50` = 0.86099039121253,
                           `51` = 0.542752476516089, `52` = 0.659538258238444, `53` = 0.936598915662334,
                           `54` = 0.535570745646648, `55` = 0.904465836977806, `56` = 0.9391875506023,
                           `57` = 0.892225101928627, `58` = 0.689810020910038, `59` = 0.893238044987496,
                           `60` = 0.89833524454686, `61` = 0.941217180270997, `62` = 0.891347113179938,
                           `63` = 0.709313511360438, `64` = 0.939289427785676, `65` = 0.93433480419103,
                           `66` = 0.850849191104047, `67` = 0.810920163009005, `68` = 0.732320214444413,
                           `69` = 0.707230451064884, `70` = 0.931516648897003, `71` = 0.841051282393807,
                           `72` = 0.49818407694599, `73` = 0.640127977065087, `74` = 0.901566158381596,
                           `75` = 0.811091376539492, `76` = 0.942774665431506, `77` = 0.888782202948999,
                           `78` = 0.827411665600484, `79` = 0.664966277642159, `80` = 0.477133983822286,
                           `81` = 0.870388680771855, `82` = 0.527970248500059, `83` = 0.537441900088327,
                           `84` = 0.585212826645815, `85` = 0.803546975166594, `86` = 0.665713927143066,
                           `87` = 0.627301011709136, `88` = 0.730630124242208, `89` = 0.775303640351822,
                           `90` = 0.873984943043952, `91` = 0.944354383648694, `92` = 0.934498073860024,
                           `93` = 0.930158397430584, `94` = 0.942497169309657, `95` = 0.865406557930892,
                           `96` = 0.923326363879199, `97` = 0.762880252354769, `98` = 0.9388219647722,
                           `99` = 0.943066110584791, `100` = 0.914853755866995, `101` = 0.610955600474384,
                           `102` = 0.669260687379209, `103` = 0.88932818649039, `104` = 0.906994601237551,
                           `105` = 0.899310380198972, `106` = 0.934958259912322, `107` = 0.935390131430429,
                           `108` = 0.918201975870802, `109` = 0.932154032243002, `110` = 0.937784465760681,
                           `111` = 0.784025244896636, `112` = 0.63551328267774, `113` = 0.711851543358259,
                           `114` = 0.933927653440435, `115` = 0.941930399839801, `116` = 0.83895111554687,
                           `117` = 0.915345355968022, `118` = 0.708949909863812, `119` = 0.70277869781501,
                           `120` = 0.574999682222332, `121` = 0.866604610283116, `122` = 0.524458697514972,
                           `123` = 0.515452326425288, `124` = 0.938675880547228, `125` = 0.561845773733424,
                           `126` = 0.628661393453755, `127` = 0.687826956843887, `128` = 0.885411759155339,
                           `129` = 0.811356729343139, `130` = 0.834108466533186, `131` = 0.750496715095514,
                           `132` = 0.583365026811254, `133` = 0.899364774605414, `134` = 0.886651069292004,
                           `135` = 0.661629601160757, `136` = 0.666840618114729, `137` = 0.54831517787716,
                           `138` = 0.821486439762622, `139` = 0.571094060745681, `140` = 0.87591429240361,
                           `141` = 0.903548070776855, `142` = 0.552485972027558, `143` = 0.936455122487288,
                           `144` = 0.846091444433744, `145` = 0.820184975592116, `146` = 0.841291519751188,
                           `147` = 0.732535685499247, `148` = 0.656114145020006, `149` = 0.591031587366035,
                           `150` = 0.607219879912788, `151` = 0.928763305975075, `152` = 0.911360446245151,
                           `153` = 0.741850128996024, `154` = 0.594120206013744, `155` = 0.762126281111666,
                           `156` = 0.892755586615687, `157` = 0.582933349747587, `158` = 0.911238260862763,
                           `159` = 0.815740719789948, `160` = 0.943118744285269, `161` = 0.863065489331271,
                           `162` = 0.823067207639977, `163` = 0.922901660379464, `164` = 0.735618546056507,
                           `165` = 0.875302275875375, `166` = 0.89708391743092, `167` = 0.86633801683679,
                           `168` = 0.588851615633179, `169` = 0.494166642585068, `170` = 0.69078179096922,
                           `171` = 0.499328072718588, `172` = 0.778895808952883, `173` = 0.82731418254166,
                           `174` = 0.536085356620809, `175` = 0.918482495088461, `176` = 0.933387540264624,
                           `177` = 0.940793283047067, `178` = 0.743622828363373, `179` = 0.700355218385421,
                           `180` = 0.618155699254652, `181` = 0.92091710803581, `182` = 0.916731049294708,
                           `183` = 0.483792957059481, `184` = 0.846877031948599, `185` = 0.907587615833538,
                           `186` = 0.822112943392842, `187` = 0.778905257323624, `188` = 0.868543077872718,
                           `189` = 0.471527432016726, `190` = 0.792352820765463, `191` = 0.920538254314786,
                           `192` = 0.833732290806581, `193` = 0.760608023005632, `194` = 0.90357723163952,
                           `195` = 0.92165954131752, `196` = 0.651031968171797, `197` = 0.752738636561175,
                           `198` = 0.881993571551401, `199` = 0.909352129243278, `200` = 0.88738945354482,
                           `201` = 0.797983141130373, `202` = 0.722986798156968, `203` = 0.852225377710566,
                           `204` = 0.551201909490704, `205` = 0.924994374441543, `206` = 0.938463409425013,
                           `207` = 0.802647799776921, `208` = 0.780007433870551, `209` = 0.628813747496237,
                           `210` = 0.720675463443831, `211` = 0.511132779850409, `212` = 0.810200083725136,
                           `213` = 0.930156452941869, `214` = 0.475016020249314, `215` = 0.841000268387166,
                           `216` = 0.58872615371859, `217` = 0.580252549294268, `218` = 0.876516532378296,
                           `219` = 0.92009696492783, `220` = 0.890653641458251, `221` = 0.943712379243512,
                           `222` = 0.646326669918136, `223` = 0.913051416128865, `224` = 0.61174795614551,
                           `225` = 0.912486025379553, `226` = 0.663227403160589, `227` = 0.803435517937664,
                           `228` = 0.891084553017628, `229` = 0.867065800028112, `230` = 0.684378542536627,
                           `231` = 0.767253188713175, `232` = 0.826552913518981, `233` = 0.869731383673948,
                           `234` = 0.51708058236784, `235` = 0.737703725469518, `236` = 0.90641481085307,
                           `237` = 0.859122909653533, `238` = 0.798980202092462, `239` = 0.589666908055303,
                           `240` = 0.917328639834314, `241` = 0.900071469298345, `242` = 0.914642199675721,
                           `243` = 0.736749976573986, `244` = 0.813396450302569, `245` = 0.909631401076801,
                           `246` = 0.848041322284931, `247` = 0.769321506146452, `248` = 0.880448006866653,
                           `249` = 0.815188177135154, `250` = 0.516178209511685, `251` = 0.937129627193908,
                           `252` = 0.778906319810184, `253` = 0.830661037297823, `254` = 0.562520803134146,
                           `255` = 0.587564547531179, `256` = 0.916307688607177, `257` = 0.683210801038827,
                           `258` = 0.845355550514775, `259` = 0.783254176118254, `260` = 0.930788935779782,
                           `261` = 0.92452692517713, `262` = 0.568229629166417, `263` = 0.601781203768783,
                           `264` = 0.913816524688982, `265` = 0.786602501707175, `266` = 0.575162120846256,
                           `267` = 0.876269638444826, `268` = 0.914486680251668, `269` = 0.530763597506452,
                           `270` = 0.828542116469509, `271` = 0.903823927919176, `272` = 0.944256690514751,
                           `273` = 0.752107892951301, `274` = 0.850488360411364, `275` = 0.586397277197287,
                           `276` = 0.810523588559756, `277` = 0.697873800803491, `278` = 0.940482443297673,
                           `279` = 0.85413960503175, `280` = 0.847339467881999, `281` = 0.873089012603203,
                           `282` = 0.858673358969587, `283` = 0.723714112091434, `284` = 0.9395441971017,
                           `285` = 0.723676306357883, `286` = 0.650281407096366, `287` = 0.806928024435669,
                           `288` = 0.92613694580316, `289` = 0.592783607006064, `290` = 0.845884999802513,
                           `291` = 0.822949958932683, `292` = 0.710613263434479, `293` = 0.892745242232795,
                           `294` = 0.88132019400313, `295` = 0.699865908978481, `296` = 0.649744809822101,
                           `297` = 0.942550281325257, `298` = 0.792663771852559, `299` = 0.746477426842534,
                           `300` = 0.522796349047643, `301` = 0.701198324759165, `302` = 0.8955915436434,
                           `303` = 0.842457670471014, `304` = 0.923427588704093, `305` = 0.921843700238387,
                           `306` = 0.718774534356296, `307` = 0.868816126023859, `308` = 0.832610633407565,
                           `309` = 0.575195793732678, `310` = 0.853636426501084, `311` = 0.901514837090205,
                           `312` = 0.721027023251245, `313` = 0.943016691226765, `314` = 0.701833594055702,
                           `315` = 0.658274501087003, `316` = 0.802885004195176, `317` = 0.8811563163552,
                           `318` = 0.687188677176487, `319` = 0.873707967234981, `320` = 0.650662777541903,
                           `321` = 0.925084725432681, `322` = 0.652132178177369, `323` = 0.812089223319568,
                           `324` = 0.590469592324845, `325` = 0.649713165201086, `326` = 0.542847562100712,
                           `327` = 0.883347077664347, `328` = 0.823683569386242, `329` = 0.889553729173905,
                           `330` = 0.867548335607517, `331` = 0.568907591974231, `332` = 0.926342964865351,
                           `333` = 0.847129605245244, `334` = 0.839291458308872, `335` = 0.839908186667382,
                           `336` = 0.912365607513407, `337` = 0.88992978226295, `338` = 0.738947087670992,
                           `339` = 0.62572081364827, `340` = 0.830569131695348, `341` = 0.913796658306697,
                           `342` = 0.885761535117959, `343` = 0.637110083967201, `344` = 0.915084154610744,
                           `345` = 0.638087067429417, `346` = 0.800850448954217, `347` = 0.873832984789658,
                           `348` = 0.889452297852388, `349` = 0.64651136992156, `350` = 0.926552182692447,
                           `351` = 0.616447459826153, `352` = 0.932237173551797, `353` = 0.679597248669064,
                           `354` = 0.602738329138805, `355` = 0.484612568811129, `356` = 0.895493492637649,
                           `357` = 0.94061409788971, `358` = 0.903192630274664, `359` = 0.922530905573505,
                           `360` = 0.85216099348424, `361` = 0.931298763485063, `362` = 0.886123030369388,
                           `363` = 0.765226226653641, `364` = 0.873216870335661, `365` = 0.537466217412586,
                           `366` = 0.84992721121887, `367` = 0.565828110146207, `368` = 0.885775778964806,
                           `369` = 0.70198991739967, `370` = 0.920572061568728, `371` = 0.637535157598144,
                           `372` = 0.661586222385373, `373` = 0.822337885820513, `374` = 0.829592355361457,
                           `375` = 0.725272422401882, `376` = 0.932578236168038, `377` = 0.739329171525406,
                           `378` = 0.627961707646841, `379` = 0.90257301375165, `380` = 0.938179200410728,
                           `381` = 0.852962265233312, `382` = 0.865081838760188, `383` = 0.825759562108804,
                           `384` = 0.844263674587638, `385` = 0.578423101519522, `386` = 0.913518018573737,
                           `387` = 0.91678748864049, `388` = 0.740106350117115, `389` = 0.942046019091874,
                           `390` = 0.933376411672288, `391` = 0.672176578367123, `392` = 0.572677825647163,
                           `393` = 0.902481542408217, `394` = 0.734817080904346, `395` = 0.873649367859615,
                           `396` = 0.545457769499469, `397` = 0.62064165722528, `398` = 0.687303500374188,
                           `399` = 0.901725189044765, `400` = 0.937661504274053), dim = 400L, dimnames = list(
                             c("1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11",
                               "12", "13", "14", "15", "16", "17", "18", "19", "20", "21",
                               "22", "23", "24", "25", "26", "27", "28", "29", "30", "31",
                               "32", "33", "34", "35", "36", "37", "38", "39", "40", "41",
                               "42", "43", "44", "45", "46", "47", "48", "49", "50", "51",
                               "52", "53", "54", "55", "56", "57", "58", "59", "60", "61",
                               "62", "63", "64", "65", "66", "67", "68", "69", "70", "71",
                               "72", "73", "74", "75", "76", "77", "78", "79", "80", "81",
                               "82", "83", "84", "85", "86", "87", "88", "89", "90", "91",
                               "92", "93", "94", "95", "96", "97", "98", "99", "100", "101",
                               "102", "103", "104", "105", "106", "107", "108", "109", "110",
                               "111", "112", "113", "114", "115", "116", "117", "118", "119",
                               "120", "121", "122", "123", "124", "125", "126", "127", "128",
                               "129", "130", "131", "132", "133", "134", "135", "136", "137",
                               "138", "139", "140", "141", "142", "143", "144", "145", "146",
                               "147", "148", "149", "150", "151", "152", "153", "154", "155",
                               "156", "157", "158", "159", "160", "161", "162", "163", "164",
                               "165", "166", "167", "168", "169", "170", "171", "172", "173",
                               "174", "175", "176", "177", "178", "179", "180", "181", "182",
                               "183", "184", "185", "186", "187", "188", "189", "190", "191",
                               "192", "193", "194", "195", "196", "197", "198", "199", "200",
                               "201", "202", "203", "204", "205", "206", "207", "208", "209",
                               "210", "211", "212", "213", "214", "215", "216", "217", "218",
                               "219", "220", "221", "222", "223", "224", "225", "226", "227",
                               "228", "229", "230", "231", "232", "233", "234", "235", "236",
                               "237", "238", "239", "240", "241", "242", "243", "244", "245",
                               "246", "247", "248", "249", "250", "251", "252", "253", "254",
                               "255", "256", "257", "258", "259", "260", "261", "262", "263",
                               "264", "265", "266", "267", "268", "269", "270", "271", "272",
                               "273", "274", "275", "276", "277", "278", "279", "280", "281",
                               "282", "283", "284", "285", "286", "287", "288", "289", "290",
                               "291", "292", "293", "294", "295", "296", "297", "298", "299",
                               "300", "301", "302", "303", "304", "305", "306", "307", "308",
                               "309", "310", "311", "312", "313", "314", "315", "316", "317",
                               "318", "319", "320", "321", "322", "323", "324", "325", "326",
                               "327", "328", "329", "330", "331", "332", "333", "334", "335",
                               "336", "337", "338", "339", "340", "341", "342", "343", "344",
                               "345", "346", "347", "348", "349", "350", "351", "352", "353",
                               "354", "355", "356", "357", "358", "359", "360", "361", "362",
                               "363", "364", "365", "366", "367", "368", "369", "370", "371",
                               "372", "373", "374", "375", "376", "377", "378", "379", "380",
                               "381", "382", "383", "384", "385", "386", "387", "388", "389",
                               "390", "391", "392", "393", "394", "395", "396", "397", "398",
                               "399", "400"))),
    predict(mod1, type = "response"),
    tolerance = .01
  )

  # mod0 <- gamm4::gamm4(y ~ t2(x0, by = x2), data = dat, family = binomial)
  mod1 <- galamm(y ~ t2(x0, by = x2), data = dat, family = binomial)
  expect_equal(c(`(Intercept)` = 2.84389184721155, `t2(x0):x2.1` = 0.0386720215446201,
                 `t2(x0):x2.2` = 0.0252124623626143, `t2(x0):x2.3` = -0.209946743997483,
                 `t2(x0):x2.4` = 2.75294985616477, `t2(x0):x2.5` = -0.0395505549222071
  ), mod1$gam$coefficients, tolerance = .1)
  expect_equal(371.26280366012, deviance(mod1), tolerance = .001)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(structure(c(`1` = 0.611790887233831, `2` = 0.940192003231048,
                           `3` = 0.600632506181895, `4` = 0.602923673034407, `5` = 0.885348174589153,
                           `6` = 0.659239566527525), dim = 6L, dimnames = list(c("1", "2",
                                                                                 "3", "4", "5", "6"))),
    head(predict(mod1, type = "response")),
    tolerance = .01
  )

  set.seed(1)
  dat <- mgcv::gamSim(1, scale = .1, verbose = FALSE, dist = "poisson")

  # mod0 <- gamm4::gamm4(y ~ s(x2), data = dat, family = poisson)
  mod1 <- galamm(y ~ s(x2), data = dat, family = poisson)
  expect_equal(558.282159001292, deviance(mod1), tolerance = .001)
  expect_equal(c(`(Intercept)` = 1, `s(x2).1` = 0.994048854607911, `s(x2).2` = 1.03213950780474,
                 `s(x2).3` = 0.740677679091768, `s(x2).4` = 0.411804189512649,
                 `s(x2).5` = 0.398394549735983, `s(x2).6` = -0.160695563809892,
                 `s(x2).7` = 0.198511671928898, `s(x2).8` = 1.47641877183332,
                 `s(x2).9` = 1.00000000000003), mod1$gam$edf, tolerance = .1)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(structure(c(`1` = 0.736415822163393, `2` = 0.805163987477725,
                           `3` = 0.48704824938033, `4` = 0.85948351065318, `5` = 1.19144585169519,
                           `6` = 0.913817291564009), dim = 6L, dimnames = list(c("1", "2",
                                                                                 "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 2.08843675534715, `2` = 2.23706331911341, `3` = 1.62750513359838,
                           `4` = 2.36194046152826, `5` = 3.29183727677599, `6` = 2.49382404030713
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    head(predict(mod1, newdata = nd, type = "response")),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 1.61183826647072, `2` = 1.56227353399864, `3` = 1.6572442185494,
                           `4` = 2.16067370979492, `5` = 3.5830584065625, `6` = 2.41042782687023
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    head(predict(mod1, type = "response")),
    tolerance = .01
  )

  # mod0 <- gamm4::gamm4(y ~ t2(x2, by = x0), data = dat, family = poisson)
  mod1 <- galamm(y ~ t2(x2, by = x0), data = dat, family = poisson)
  expect_equal(590.708731160914, deviance(mod1), tolerance = .001)
  expect_equal(c(`(Intercept)` = 0.999999999999999, `t2(x2):x0.1` = 0.859811531001738,
                 `t2(x2):x0.2` = 0.9587799971987, `t2(x2):x0.3` = 0.992143302480705,
                 `t2(x2):x0.4` = 0.999999999999998, `t2(x2):x0.5` = 1), mod1$gam$edf, tolerance = .1)
  expect_snapshot(print(summary(mod1$gam), digits = 2))
  expect_equal(structure(c(`1` = 0.860750218632325, `2` = 0.800052346079306,
                           `3` = 0.263933079504091, `4` = 0.835765266292293, `5` = 1.21353435558211,
                           `6` = 0.915685459713133), dim = 6L, dimnames = list(c("1", "2",
                                                                                 "3", "4", "5", "6"))),
    predict(mod1, newdata = nd),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 2.36493424608223, `2` = 2.22565742988358, `3` = 1.30204106015189,
                           `4` = 2.30657851984736, `5` = 3.36535802975894, `6` = 2.49848727744805
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    head(predict(mod1, newdata = nd, type = "response")),
    tolerance = .01
  )
  expect_equal(structure(c(`1` = 2.22487072233896, `2` = 1.92890308217768, `3` = 1.63827636859298,
                           `4` = 1.96964009869582, `5` = 2.78250950225448, `6` = 1.97203490713028
  ), dim = 6L, dimnames = list(c("1", "2", "3", "4", "5", "6"))),
    head(predict(mod1, type = "response")),
    tolerance = .01
  )
})

test_that("Basic GAMM with factor structures works", {
  dat <- subset(cognition, domain == 1 & timepoint == 1)

  mod <- galamm(
    formula = y ~ 0 + item + sl(x, factor = "loading"),
    data = dat,
    load.var = "item",
    lambda = list(matrix(c(1, NA, NA), ncol = 1)),
    factor = list("loading")
  )

  expect_equal(
    coef(mod)[1:3],
    c(
      item11 = 0.669888238636741, item12 = 0.938949881157583,
      item13 = 0.19327451279475
    ),
    tolerance = .0001
  )

  expect_equal(
    predict(mod$gam)[3:9],
    structure(c(
      `3` = -0.0468214579631071, `25` = 0.0227632205604528,
      `26` = 0.291824863081295, `27` = -0.453850505281539,
      `49` = -0.00253156792150957,
      `50` = 0.266530074599332, `51` = -0.479145293763501
    ), dim = 7L, dimnames = list(
      c("3", "25", "26", "27", "49", "50", "51")
    )),
    tolerance = .0001
  )

  expect_equal(
    vcov(mod),
    structure(c(
      0.00559985106383491, 6.76254095833776e-16, 8.7167665432172e-17,
      3.59298897606922e-10, 6.76254095833776e-16, 0.0055998510638353,
      1.15498696525652e-16, 4.76077386790633e-10, 8.7167665432172e-17,
      1.15498696525652e-16, 0.00559985106383441, 6.13653297234754e-11,
      3.59298897606922e-10, 4.76077386790633e-10, 6.13653297234754e-11,
      0.0639815815562496
    ), dim = c(4L, 4L)),
    tolerance = .0001
  )

  expect_equal(
    factor_loadings(mod),
    structure(c(
      1, 1.38890559188738, 0.285174466526739, NA, 0.206861514893889,
      0.125655999708447
    ), dim = 3:2, dimnames = list(c(
      "lambda1", "lambda2",
      "lambda3"
    ), c("loading", "SE")))
  )

  mod <- galamm(
    formula = y ~ 0 + item + t2l(x, factor = "loading"),
    data = dat,
    load.var = "item",
    lambda = list(matrix(c(1, NA, NA), ncol = 1)),
    factor = list("loading")
  )

  expect_equal(
    coef(mod)[1:3],
    c(
      item11 = 0.585985043619259, item12 = 0.821880314450475,
      item13 = 0.169255594138125
    ),
    tolerance = .001
  )

  expect_equal(
    predict(mod$gam)[3:9],
    structure(
      c(
        `3` = 0.00564862054158604, `25` = 0.0151596217554541,
        `26` = 0.251054892586669, `27` = -0.401569827725681,
        `49` = -0.00924515260886517,
        `50` = 0.22665011822235, `51` = -0.42597460209
      ),
      dim = 7L, dimnames = list(
        c("3", "25", "26", "27", "49", "50", "51")
      )
    ),
    tolerance = .0001
  )

  expect_equal(
    vcov(mod),
    structure(c(
      0.0083132054251711, 0.00355423354003251, 0.000728096111493452,
      0.00610186781361705, 0.00355423354003251, 0.0107112518833507,
      0.00101403521195143, 0.00959339564679258, 0.000728096111493452,
      0.00101403521195143, 0.0059689273518403, 0.00196479761841795,
      0.00610186781361705, 0.00959339564679258, 0.00196479761841795,
      0.0246253221170968
    ), dim = c(4L, 4L)),
    tolerance = .0001
  )

  expect_equal(
    factor_loadings(mod),
    structure(
      c(
        1, 1.39530180832884, 0.286267927177228, NA, 0.212062073868377,
        0.128493620433266
      ),
      dim = 3:2,
      dimnames = list(
        c("lambda1", "lambda2", "lambda3"),
        c("loading", "SE")
      )
    )
  )
})

test_that("GAMM with factor structures and random effects works", {
  dat <- subset(cognition, domain == 1 & id < 50 & timepoint %in% c(1, 4, 8))

  mod <- galamm(
    formula = y ~ 0 + item + sl(x, factor = "loading", k = 4) +
      (0 + loading | id / timepoint),
    data = dat,
    load.var = "item",
    lambda = list(matrix(c(1, NA, NA), ncol = 1)),
    factor = list("loading")
  )

  expect_equal(
    coef(mod),
    c(
      item11 = 1.15744460299196, item12 = 1.60878967518903,
      item13 = 0.341634130725673,
      `s(x):loadingFx1` = -0.146146004977857
    )
  )

  expect_equal(
    vcov(mod),
    structure(c(
      0.0139704055103376, 0.019404393983382, 0.00416969325689835,
      5.92964962109609e-11, 0.019404393983382, 0.0271611676668362,
      0.00582120760678741, 7.95171728770467e-11, 0.00416969325689835,
      0.00582120760678741, 0.00132204815534548, 1.71344935389941e-11,
      5.92964962109633e-11, 7.95171728772904e-11, 1.7134493539049e-11,
      0.0490470998802075
    ), dim = c(4L, 4L))
  )

  expect_equal(
    factor_loadings(mod),
    structure(c(
      1, 1.39607574182039, 0.29999430086666, NA, 0.0121378075498697,
      0.00737715422888654
    ), dim = 3:2, dimnames = list(c(
      "lambda1",
      "lambda2", "lambda3"
    ), c("loading", "SE")))
  )

  expect_equal(mod$gam$edf,
    c(
      item11 = 1.00000000000001, item12 = 1.00000000000003,
      item13 = 0.999999999999996,
      `s(x):loading.1` = 0.995315567672236,
      `s(x):loading.2` = 0.999757875374097,
      `s(x):loading.3` = 1.00000000000007
    ),
    tolerance = .1
  )

  expect_equal(mod$gam$coefficients,
    c(
      item11 = 1.15744460299196, item12 = 1.60878967518903,
      item13 = 0.341634130725673,
      `s(x):loading.1` = 0.601163762614973, `s(x):loading.2` = 3.02308664034339,
      `s(x):loading.3` = -0.146146004977857
    ),
    tolerance = .01
  )
})

test_that("galamm with by variables and loadings works", {
  dat <- subset(
    cognition,
    domain %in% c(1, 3) & item %in% c("11", "12", "31", "32")
  )
  dat <- cbind(
    dat,
    model.matrix(~ 0 + domain, data = dat)[, c("domain1", "domain3")]
  )
  lmat <- matrix(c(
    1, NA, 0, 0,
    0, 0, 1, NA
  ), ncol = 2)

  mod <- galamm(
    formula = y ~
      domain + sl(x, k = 4, by = domain, factor = c("ability1", "ability3")) +
      (0 + domain1:ability1 + domain3:ability3 | id),
    data = dat,
    load.var = "item",
    lambda = list(lmat),
    factor = list(c("ability1", "ability3")),
    control = galamm_control(
      optim_control = list(maxit = 0), reduced_hessian = TRUE
    )
  )

  expect_equal(
    deviance(mod),
    23732.65,
    tolerance = .001
  )
})
