% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/builder_functions.R
\name{spa_creator}
\alias{spa_creator}
\title{Building \code{pgeometry} objects from a point dataset}
\usage{
spa_creator(tbl, fuzz_policy = "fsp", const_policy = "voronoi", ...)
}
\arguments{
\item{tbl}{A data.frame or tibble with the following format: (x, y, z)}

\item{fuzz_policy}{The fuzzification policy to be employed by the algorithm. See details below.}

\item{const_policy}{The construction policy to be used by the algorithm. See details below.}

\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> Parameters for the chosen policies. See details below.}
}
\value{
A tibble in the format \verb{(class, pgeometry)}, where \code{class} is a character column and \code{pgeometry} is a list of \code{pgeometry} objects.
This means that a spatial plateau object is created for representing a specific class of the point dataset.
}
\description{
This function builds a set of spatial plateau objects from a given point dataset assigned with domain-specific numerical values.
}
\details{
It follows the two-stage construction method described in the research paper of reference.

The input \code{tbl} is a point dataset where each point represents the location of a phenomenon treated by the application.
Further, each point is annotated with numerical data that describe its meaning in the application.
Therefore, \code{tbl} must have three columns: (\emph{x}, \emph{y}, \emph{z}). The columns \emph{x}, \emph{y} are the pair of coordinates, and \emph{z} is the column containing domain-specific numeric values.

\code{fuzz_policy} refers to the method used by the \strong{fuzzification stage}.
This stage aims to assign membership degrees to each point of the dataset.
It accepts two possible values only: either \code{"fsp"} (default) or "\verb{fcp"}.

\code{"fsp"} stands for \emph{fuzzy set policy} and requires two parameters that should be informed in \code{...}:
\itemize{
\item \code{classes}: A character vector containing the name of classes
\item \code{mfs}: A vector of membership functions generated by the function \code{genmf} of \code{FuzzyR} package. Each membership function \emph{i} represents the class \emph{i}, where \emph{i} in \code{length(classes)}
}

\code{"fcp"} stands for \emph{fuzzy clustering policy} and requires the \code{e1071} package. Its possible parameters, informed in \code{...}, are:
\itemize{
\item \code{k}: A numeric value that refers to the number of groups to be created
\item \code{method}: A fuzzy clustering method of the package \code{e1071}, which can be either \code{"cmeans"} (default) or \code{"cshell"}
\item \code{use_coords}: A Boolean value to indicate whether the columns (x, y) should be used in the clustering algorithm (default is \code{FALSE})
\item \code{iter}: A numeric indicating the number of maximum iterations of the clustering algorithm (default is 100)
}

An optional and common parameter for both fuzzification stages is the \code{"digits"}.
This is an integer value that indicates the number of decimal digits of the membership degrees calculated by the fuzzification stage.
That is, it is used to round such membership degrees to a specified number of decimal digits.
Be careful with this optional parameter! If you specify a low value for \code{"digits"} some membership degrees could be rounded to 0 and thus, some components would not be created.

\code{const_policy} refers to the method used by the \strong{construction stage}.
This stage aims to create polygons from the labeled point dataset and use them to build spatial plateau objects.
It accepts two possible values only: either \code{"voronoi"} (default) or "\verb{delaunay"}.

\code{"voronoi"} stands for \emph{Voronoi diagram policy} and has one optional parameter that can be provided in \code{...}:
\itemize{
\item \code{base_poly}: An \code{sfg} object that will be used to clip the generated polygons (optional argument). If this parameter is not provided, the Voronoi is created by using a bounding box (standard behavior of \code{sf}).
}

\code{"delaunay"} stands for \emph{Delaunay triangulation policy}, which accepts the following parameters in \code{...}:
\itemize{
\item \code{base_poly}: An \code{sfg} object that will be used to clip the generated triangles (optional argument).
\item \code{tnorm}: A t-norm used to calculate the membership degree of the triangle. It should be the name of a vector function.
Possible values are \code{"min"} (default), and \code{"prod"}.
Note that it is possible to use your own t-norms. A t-norm should has the following signature: \code{FUN(x)} where
\item \emph{x} is a numeric vector. Such a function should return a single numeric value.
}

\code{"convex_hull"} stands for \emph{Convex hull policy}, which accepts the following parameters in \code{...}:
\itemize{
\item \code{M}: A numeric vector containing the membership degrees that will be used to create the components. The default is defined by \code{seq(0.05, 1, by = 0.05)}.
\item \code{d}: A numeric value representing the tolerance distance to compute the membership degree between the elements of \code{M} and the membership degrees of the points. The default is \code{0.05}.
}
}
\examples{

library(tibble)
library(FuzzyR)

set.seed(7)
tbl = tibble(x = runif(10, min= 0, max = 30), 
             y = runif(10, min = 0, max = 50), 
             z = runif(10, min = 0, max = 100))
classes <- c("cold", "hot")
cold_mf <- genmf("trapmf", c(0, 10, 20, 35))
hot_mf <- genmf("trimf", c(35, 50, 100))

spa_creator(tbl, classes = classes, mfs = c(cold_mf, hot_mf))

spa_creator(tbl, fuzz_policy = "fcp", k = 4)

spa_creator(tbl, fuzz_policy = "fcp", k = 4, digits = 2)

spa_creator(tbl, fuzz_policy = "fcp", k = 3, const_policy = "delaunay")

spa_creator(tbl, fuzz_policy = "fcp", const_policy = "delaunay", k = 3, tnorm = "prod")

spa_creator(tbl, fuzz_policy = "fcp", k = 2, digits = 2, 
            M = seq(0.1, 1, by = 0.1), d = 0.05, const_policy = "convex_hull")
            
spa_creator(tbl, classes = classes, mfs = c(cold_mf, hot_mf), 
            digits = 2, const_policy = "convex_hull")

}
\references{
\href{https://ieeexplore.ieee.org/document/8858878/}{Carniel, A. C.; Schneider, M. A Systematic Approach to Creating Fuzzy Region Objects from Real Spatial Data Sets. In Proceedings of the 2019 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2019), pp. 1-6, 2019.}
}
