% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fourierin.R
\name{fourierin_2d}
\alias{fourierin_2d}
\title{Bivariate Fourier integrals}
\usage{
fourierin_2d(f, a, b, c = NULL, d = NULL, r, s, resol = NULL, w = NULL,
  use_fft = TRUE)
}
\arguments{
\item{f}{function or a matrix of size m1 x m2. If a function is
provided, it must be able to be evaluated in a matrix of two
columns. If a matrix of values is provided instead, such
evaluations must have been obtained on a regular grid matrix
and the Fourier integral is faster is m1 and m2 are powers of
2.}

\item{a}{Lower integration limits.}

\item{b}{Upper integration limits.}

\item{c}{Lower evaluation limits.}

\item{d}{Upper evaluation limits.}

\item{r}{Factor related to adjust definition of Fourier
transform. It is usually 0, -1 or 1.}

\item{s}{Constant to adjust the exponent on the definition of the
Fourier transform. It is usually 1, -1, 2pi or -2pi.}

\item{resol}{A vector of two integers (faster if entries are powers
of two) determining the resolution of the evaluation grid. Not
required if f is a vector.}

\item{w}{An optional two-column matrix where the Fourier integral
will be evaluated. If provided, the FFT will NOT be used.}

\item{use_fft}{Logical value specifying whether the FFT will be
used to compute the Fourier integral.}
}
\value{
If w is given, only the values of the Fourier integral are
    returned, otherwise, a list with three elements
    \item{w1}{Evaluation grid for first entry} \item{w2}{Evaluation
    grid for second entry} \item{values}{m1 x m2 matrix of complex
    numbers, correspoding to the evaluations of the integral}
}
\description{
It computes Fourier integrals for functions of one and two
variables.
}
\examples{
##--- Recovering std. normal from its characteristic function --------
library(fourierin)

##-Parameters of bivariate normal distribution
mu <- c(-1, 1)
sig <- matrix(c(3, -1, -1, 2), 2, 2)

##-Multivariate normal density
##-x is n x d
f <- function(x) {
    ##-Auxiliar values
    d <- ncol(x)
    z <- sweep(x, 2, mu, "-")

    ##-Get numerator and denominator of normal density
    num <- exp(-0.5*rowSums(z * (z \%*\% solve(sig))))
    denom <- sqrt((2*pi)^d*det(sig))

    return(num/denom)
}

##-Characteristic function
##-s is n x d
phi <- function(s) {
    complex(modulus = exp(- 0.5*rowSums(s*(s \%*\% sig))),
            argument = s \%*\% mu)
}

##-Approximate cf using Fourier integrals
eval <- fourierin_2d(f, a = c(-8, -6), b = c(6, 8),
                    c = c(-4, -4), d = c(4, 4),
                    r = 1, s = 1, resol = c(128, 128))
t1 <- eval$w1
t2 <- eval$w2
t <- as.matrix(expand.grid(t1 = t1, t2 = t2))
approx <- eval$values
true <- matrix(phi(t), 128, 128)        # Compute true values

##-This is a section of the characteristic functions
i <- 65
plot(t2, Re(approx[i, ]), type = "l", col = 2,
     ylab = "",
     xlab = expression(t[2]),
     main = expression(paste("Real part section at ",
                             t[1], "= 0")))
lines(t2, Re(true[i, ]), col = 3)
legend("topleft", legend = c("true", "approximation"),
       col = 3:2, lwd = 1)

##-Another section, now of the imaginary part
plot(t1, Im(approx[, i]), type = "l", col = 2,
     ylab = "",
     xlab = expression(t[1]),
     main = expression(paste("Imaginary part section at ",
                             t[2], "= 0")))
lines(t1, Im(true[, i]), col = 3)
legend("topleft", legend = c("true", "approximation"),
       col = 3:2, lwd = 1)
}

