% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assignment.R
\name{run_assignment}
\alias{run_assignment}
\alias{print.flownet}
\title{Run Traffic Assignment}
\usage{
run_assignment(
  graph_df,
  od_matrix_long,
  directed = FALSE,
  cost.column = "cost",
  method = c("PSL", "AoN"),
  beta = 1,
  ...,
  detour.max = 1.5,
  angle.max = 90,
  unique.cost = TRUE,
  npaths.max = Inf,
  dmat.max.size = 10000^2,
  return.extra = NULL,
  verbose = TRUE,
  nthreads = 1L
)

\method{print}{flownet}(x, ...)
}
\arguments{
\item{graph_df}{A data.frame with columns \code{from}, \code{to}, and optionally a cost column.
Represents the network graph with edges between nodes.}

\item{od_matrix_long}{A data.frame with columns \code{from}, \code{to}, and \code{flow}.
Represents the origin-destination matrix in long format with flow values.}

\item{directed}{Logical (default: FALSE). Whether the graph is directed.}

\item{cost.column}{Character string (default: "cost") or numeric vector. Name of the cost column
in \code{graph_df}, or a numeric vector of edge costs with length equal to \code{nrow(graph_df)}.
The cost values are used to compute shortest paths and determine route probabilities.}

\item{method}{Character string (default: "PSL"). Assignment method:
\itemize{
  \item \code{"PSL"}: Path-Sized Logit model considering multiple routes with overlap correction
  \item \code{"AoN"}: All-or-Nothing assignment, assigns all flow to the shortest path (faster but no route alternatives)
}}

\item{beta}{Numeric (default: 1). Path-sized logit parameter (beta_PSL). Only used for PSL method.}

\item{\dots}{Additional arguments (currently ignored).}

\item{detour.max}{Numeric (default: 1.5). Maximum detour factor for alternative routes (applied to shortest paths cost). Only used for PSL method. This is a key parameter controlling the execution time of the algorithm: considering more routes (higher \code{detour.max}) substantially increases computation time.}

\item{angle.max}{Numeric (default: 90). Maximum detour angle (in degrees, two sided). Only used for PSL method. I.e., nodes not within this angle measured against a straight line from origin to destination node will not be considered for detours.}

\item{unique.cost}{Logical (default: TRUE). Deduplicates paths based on the total cost prior to generating them. Only used for PSL method. Since multiple 'intermediate nodes' may be on the same path, there is likely a significant number of duplicate paths which this option removes.}

\item{npaths.max}{Integer (default: Inf). Maximum number of paths to compute per OD-pair. Only used for PSL method. If the number of paths exceeds this number, a random sample will be taken from all but the shortest path.}

\item{dmat.max.size}{Integer (default: 1e4^2). Maximum size of distance matrices (both shortest paths and geodesic) to precompute. If smaller than \code{n_nodes^2}, then the full matrix is precomputed. Otherwise, it is computed in chunks as needed, where each chunk has \code{dmat.max.size} elements. Only used for PSL method.}

\item{return.extra}{Character vector specifying additional results to return.
  Use \code{"all"} to return all available extras for the selected method.

  \tabular{llll}{
    \strong{Option} \tab \strong{PSL} \tab \strong{AoN} \tab \strong{Description} \cr
    \code{"graph"} \tab Yes \tab Yes \tab The igraph graph object \cr
    \code{"paths"} \tab Yes \tab Yes \tab PSL: list of lists of edge indices (multiple routes per OD); AoN: list of edge index vectors (one path per OD) \cr
    \code{"edges"} \tab Yes \tab No \tab List of edge indices used for each OD pair \cr
    \code{"counts"} \tab Yes \tab Yes \tab PSL: list of edge visit counts per OD; AoN: integer vector of global edge traversal counts \cr
    \code{"costs"} \tab Yes \tab Yes \tab PSL: list of path costs per OD; AoN: numeric vector of shortest path costs \cr
    \code{"weights"} \tab Yes \tab No \tab List of path weights (probabilities) for each OD pair \cr
  }}

\item{verbose}{Logical (default: TRUE). Show progress bar and intermediate steps completion status?}

\item{nthreads}{Integer (default: 1L). Number of threads (daemons) to use for parallel processing with \code{\link[mirai]{mirai}}. Should not exceed the number of logical processors.}

\item{x}{An object of class \code{flownet}, typically returned by \code{\link{run_assignment}}.}
}
\value{
A list of class \code{"flownet"} containing:
  \itemize{
    \item \code{call} - The function call
    \item \code{final_flows} - Numeric vector of assigned flows for each edge (same length as \code{nrow(graph_df)})
    \item \code{od_pairs_used} - Indices of OD pairs with valid flows
    \item Additional elements as specified in \code{return.extra}:
      \itemize{
        \item \code{graph} - The igraph graph object
        \item \code{paths} - For PSL: list of lists of edge indices (multiple routes per OD pair); for AoN: list of edge index vectors (one shortest path per OD pair)
        \item \code{edges} - List of edge indices used for each OD pair (PSL only)
        \item \code{edge_counts} - For PSL: list of edge visit counts per OD pair; for AoN: integer vector of global edge traversal counts
        \item \code{path_costs} - For PSL: list of path costs per OD pair; for AoN: numeric vector of shortest path costs
        \item \code{path_weights} - List of path weights (probabilities) for each OD pair (PSL only)
      }
  }
}
\description{
Assign traffic flows to network edges using either Path-Sized Logit (PSL)
  or All-or-Nothing (AoN) assignment methods.
}
\details{
This function performs traffic assignment using one of two methods:
\strong{All-or-Nothing (AoN)} is fast but assigns all flow to a single shortest path;
\strong{Path-Sized Logit (PSL)} considers multiple routes with overlap correction for
more realistic flow distribution.

\subsection{All-or-Nothing (AoN) Method}{
A simple assignment method that assigns all flow from each OD pair to the single shortest path.
This is much faster than PSL but does not consider route alternatives or overlaps.
Parameters \code{detour.max}, \code{angle.max}, \code{unique.cost}, \code{npaths.max},
\code{beta}, and \code{dmat.max.size} are ignored for AoN.
}

\subsection{Path-Sized Logit (PSL) Method}{
A more sophisticated assignment method that considers multiple alternative routes and
accounts for route overlap when assigning flows. The PSL model adjusts choice probabilities
based on how much each route overlaps with other alternatives, preventing overestimation
of flow on shared segments. The \code{beta} parameter controls the sensitivity to overlap.
}

\subsection{PSL Model Formulation}{
The probability \eqn{P_k} of choosing route \eqn{k} from the set of alternatives \eqn{K} is:
\deqn{P_k = \frac{e^{V_k}}{\sum_{j \in K} e^{V_j}}}
where the utility \eqn{V_k} is defined as:
\deqn{V_k = -C_k + \beta_{PSL} \ln(PS_k)}
Here \eqn{C_k} is the generalized cost of route \eqn{k}, \eqn{\beta_{PSL}} is the
path-size parameter (the \code{beta} argument), and \eqn{PS_k} is the path-size factor.

The path-size factor quantifies route uniqueness:
\deqn{PS_k = \frac{1}{C_k} \sum_{a \in \Gamma_k} \frac{c_a}{\delta_a}}
where \eqn{\Gamma_k} is the set of edges in path \eqn{k}, \eqn{c_a} is the cost of
edge \eqn{a}, and \eqn{\delta_a} is the number of alternative routes using edge \eqn{a}.

If a path is unique (\eqn{\delta_a = 1} for all edges), then \eqn{PS_k = 1} and the
model reduces to standard MNL. For overlapping routes, \eqn{PS_k < 1} and
\eqn{\ln(PS_k) < 0}, so a positive \code{beta} penalizes overlap. Higher \code{beta}
values strengthen penalization; \code{beta = 0} gives standard MNL behavior.

For more information about the PSL model consult some of the references below.
}

\subsection{Route Enumeration Algorithm}{
For each origin-destination pair, the algorithm identifies alternative routes as follows:
\enumerate{
  \item Compute the shortest path cost from origin to destination. If \code{sqrt(dmat.max.size) < N.nodes}, the entire shortest-path-distance matrix is precomputed.
  \item For each potential intermediate node, calculate the total cost of going
        origin -> intermediate -> destination (also using the distance matrix).
  \item Keep only routes where total cost is within \code{detour.max} times the
        shortest path cost.
  \item If \code{angle.max} is specified, filter to intermediate nodes that lie
        roughly in the direction of the destination (within the specified angle - see further details below).  If \code{sqrt(dmat.max.size) < N.nodes} a geodesic-distance-matrix is precomputed for speedy calculations using the triangle equation.
  \item If \code{unique.cost = TRUE}, remove duplicate routes based on total cost -
        as multiple intermediate nodes may yield exactly the same route.
  \item (Optionally) use \code{npaths.max} to sample the remaining routes if still too many.
  \item Compute the actual paths and filter out those with duplicate edges
        (where the intermediate node is approached and departed via the same edge).
}
This pre-selection using distance matrices speeds up route enumeration considerably
by avoiding the computation of implausible paths.
}

\subsection{Coordinate-Based Filtering}{
When \code{angle.max} is specified and \code{graph_df} contains coordinate columns
(\code{FX}, \code{FY}, \code{TX}, \code{TY}), the function uses geographic distance
calculations to restrict detours. Only intermediate nodes that are (a) closer to the
origin than the destination is, and (b) within the specified angle from the
origin-destination line are considered. This improves both computational efficiency
and route realism by excluding geographically implausible detours.
}
}
\examples{
library(flownet)
library(collapse)
library(sf)

# Load existing network edges (exclude proposed new links)
africa_net <- africa_network[!africa_network$add, ]

# Convert to graph (use atomic_elem to drop sf geometry, qDF for data.frame)
graph <- atomic_elem(africa_net) |> qDF()
nodes <- nodes_from_graph(graph, sf = TRUE)

# Map cities/ports to nearest network nodes
nearest_nodes <- nodes$node[st_nearest_feature(africa_cities_ports, nodes)]

# Simple gravity-based OD matrix
od_mat <- outer(africa_cities_ports$population, africa_cities_ports$population) / 1e12
dimnames(od_mat) <- list(nearest_nodes, nearest_nodes)
od_matrix_long <- melt_od_matrix(od_mat)

# Run Traffic Assignment (All-or-Nothing method)
result_aon <- run_assignment(graph, od_matrix_long, cost.column = "duration",
                             method = "AoN", return.extra = "all")
print(result_aon)
\donttest{
# Run Traffic Assignment (Path-Sized Logit method)
# Note: PSL is slower but produces more realistic flow distribution
result_psl <- run_assignment(graph, od_matrix_long, cost.column = "duration",
                             method = "PSL", nthreads = 1L,
                             return.extra = c("edges", "counts", "costs", "weights"))
print(result_psl)

# Visualize AoN Results
africa_net$final_flows_log10 <- log10(result_psl$final_flows + 1)
plot(africa_net["final_flows_log10"], main = "PSL Assignment")
}


# --- Trade Flow Disaggregation Example ---
# Disaggregate country-level trade to city-level using population shares

# Compute each city's share of its country's population
city_pop <- africa_cities_ports |> atomic_elem() |> qDF() |>
  fcompute(node = nearest_nodes,
           city = qF(city_country),
           pop_share = fsum(population, iso3, TRA = "/"),
           keep = "iso3")

# Aggregate trade to country-country level and disaggregate to cities
trade_agg <- africa_trade |> collap(quantity ~ iso3_o + iso3_d, fsum)
od_matrix_trade <- trade_agg |>
  join(city_pop |> add_stub("_o", FALSE), multiple = TRUE) |>
  join(city_pop |> add_stub("_d", FALSE), multiple = TRUE) |>
  fmutate(flow = quantity * pop_share_o * pop_share_d) |>
  frename(from = node_o, to = node_d) |>
  fsubset(flow > 0 & from != to)

# Run AoN assignment with trade flows
result_trade_aon <- run_assignment(graph, od_matrix_trade, cost.column = "duration",
                                   method = "AoN", return.extra = "all")
print(result_trade_aon)
\donttest{
# Visualize trade flow results
africa_net$trade_flows_log10 <- log10(result_trade_aon$final_flows + 1)
plot(africa_net["trade_flows_log10"], main = "Trade Flow Assignment (AoN)")

# Run PSL assignment with trade flows (nthreads can be increased for speed)
result_trade_psl <- run_assignment(graph, od_matrix_trade, cost.column = "duration",
                                   method = "PSL", nthreads = 1L,
                                   return.extra = c("edges", "counts", "costs", "weights"))
print(result_trade_psl)

# Compare PSL vs AoN: PSL typically shows more distributed flows
africa_net$trade_flows_psl_log10 <- log10(result_trade_psl$final_flows + 1)
plot(africa_net["trade_flows_psl_log10"], main = "Trade Flow Assignment (PSL)")
}

}
\references{
Ben-Akiva, M., & Bierlaire, M. (1999). Discrete choice methods and their
applications to short term travel decisions. In R. W. Hall (Ed.), \emph{Handbook
of Transportation Science} (pp. 5-33). Springer US. \doi{10.1007/978-1-4615-5203-1_2}

Cascetta, E. (2001). \emph{Transportation systems engineering: Theory and methods}.
Springer.

Ben-Akiva, M., & Lerman, S. R. (1985). \emph{Discrete choice analysis: Theory and
application to travel demand}. The MIT Press.

Ramming, M. S. (2002). \emph{Network knowledge and route choice} (Doctoral
dissertation). Massachusetts Institute of Technology.

Prato, C. G. (2009). Route choice modeling: Past, present and future research
directions. \emph{Journal of Choice Modelling, 2}(1), 65-100. \doi{10.1016/S1755-5345(13)70005-8}

AequilibiaE Python Documentation: https://www.aequilibrae.com/develop/python/route_choice/path_size_logit.html
}
\seealso{
\link{flownet-package}
}
