/*  Routines for manipulating B-splines.  These are intended for use with
 *  S or S-PLUS or R.
 *
 *     Copyright (C) 2015 Michel Grzebyk.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * These functions are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  http://www.r-project.org/Licenses/
 *
 * The routines are loosely based on the pseudo-code in Schumacher (Wiley,
 * 1981) and the CMLIB library DBSPLINES.
 */

#include <R.h>
#include <Rinternals.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(String) dgettext ("splines", String)
#else
#define _(String) (String)
#endif

#ifndef EVAL_SPLINEPARAM
#include "SplineParam.h"
#endif


SEXP predict_spline_basis(SEXP knots, SEXP order, SEXP Matrices, SEXP intercept, SEXP xvals, SEXP outerok)
{
/* predict ie evaluate linear combination of the  B-spline basis functions at xvals, 
using matrices generated by packag::orthogonalsplinbasis */
/* 
 knots : vector of ordered unreplicated INTERIOR knots 
Matrices : a vectorized array of dim order X nbases X number_of_intervales(knots) 
  where nbases is the number of bases of the non integrated, non derived splines
the values in Matrices are scaled by the coef of the spline bases 
order : order of the splines (see package orthogonalsplinbasis
intercept : wehtehr first basis is included
xvals : vector values at which bases are computed
 */
    R_len_t i, k, nknots, theorder, nbases, nintervals, nx, oo;
    R_len_t firstbasis;
    double *rknots, *rMatrices, *rAddMatrices, *rxvals, *rpredict;
    SEXP predict;
    SEXP dims;
    
    double outer_val;
    
    PROTECT(knots = coerceVector(knots, REALSXP));
    PROTECT(order = coerceVector(order, INTSXP));
    PROTECT(intercept = coerceVector(intercept, INTSXP));
    PROTECT(Matrices = coerceVector(Matrices, REALSXP));
    PROTECT(xvals = coerceVector(xvals, REALSXP));
    PROTECT(outerok = coerceVector(outerok, LGLSXP));


    rknots = REAL(knots); 
    nknots = length(knots);
    theorder = INTEGER(order)[0];

    dims = getAttrib(Matrices, R_DimSymbol);
    if( LENGTH(dims) < 3 ){
        error("'Matrices' must be an array with 3 dim");   
    }
    nbases = INTEGER(dims)[1];
    nintervals = INTEGER(dims)[2];
    
    rxvals = REAL(xvals); 
    nx = length(xvals);
        
    
    firstbasis = (INTEGER(intercept)[0]==0);
    rMatrices = REAL(Matrices);
        
    PROTECT(predict = allocVector(REALSXP, nx));
    rpredict = REAL(predict);

    oo = asLogical(outerok);
    if(oo == NA_LOGICAL) {
        error("'outer.ok' must be TRUE or FALSE");    
    } else  if (oo) {
        outer_val = 0.0;
    } else {
        outer_val = R_NaN;
    }

/* first add cols of each matrix */
    rAddMatrices = (double *) R_alloc( theorder * nintervals, sizeof(double));
    for (i = 0; i < nintervals; i++) {
        for (int j = 0; j < theorder ; j++) {
            rAddMatrices[theorder*i + j] = 0.0;
            for (k = firstbasis; k < nbases; k++) {
                rAddMatrices[theorder*i + j] += rMatrices[theorder*nbases*i+ theorder*k + j];
            }
        }
    }

    R_len_t theinterval, mfl;
    double temppredict, U, u;



    for(i = 0; i < nx; i++) {
        PREDIC_one_spline_basis(rxvals[i], rpredict[i])
    }
    
    UNPROTECT(7);
    return(predict);
}






double r_predict_spline_basis(double *rknots, R_len_t nknots, R_len_t theorder, 
                  double *rAddMatrices, double rxval, double outer_val)
{
/* predict (ie evaluate linear combination) of the  B-spline basis functions at a SINGLE value xval, 
using ADDED/SUMMED matrices generated by packag::orthogonalsplinbasis */
/* 
 rknots : vector of ordered unreplicated INTERIOR knots 
rAddMatrices : a vectorized array of dim order X number_of_intervales(knots) 
  where nbases is the number of bases of the non integrated, non derived splines
the values in rAddMatrices are obtained by summing the Matrix from packag::orthogonalsplinbasis,
                                                    previously scaled by the coef of the spline bases  
                                                    over the bases. 
order : order of the splines (see package orthogonalsplinbasis
intercept : wehtehr first basis is included
xval : the single value at which prediction is computed
 */

    R_len_t theinterval, mfl;
    double temppredict, U, u, rpredict;

    PREDIC_one_spline_basis(rxval, rpredict)
    return rpredict;
}

