% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatSample.r
\name{spatSample,GRaster-method}
\alias{spatSample,GRaster-method}
\alias{spatSample}
\alias{spatSample,GVector-method}
\title{Sample random points from a GRaster or GVector}
\usage{
\S4method{spatSample}{GRaster}(
  x,
  size,
  as.points = FALSE,
  values = TRUE,
  cats = TRUE,
  xy = FALSE,
  strata = NULL,
  byStratum = FALSE,
  zlim = NULL,
  seed = NULL,
  verbose = FALSE
)

\S4method{spatSample}{GVector}(
  x,
  size,
  as.points = FALSE,
  values = TRUE,
  xy = FALSE,
  byStratum = FALSE,
  zlim = NULL,
  seed = NULL
)
}
\arguments{
\item{x}{A \code{GRaster} or \code{GVector}.}

\item{size}{Numeric value > 0: Number of points to create.}

\item{as.points}{Logical: If \code{FALSE} (default), the output is a \code{data.frame} or \code{data.table}. If \code{TRUE}, the output is a "points" \code{GVector}.}

\item{values}{Logical: If \code{TRUE} (default), values of the \code{GRaster} at points are returned.}

\item{cats}{Logical: If \code{TRUE} (default) and the \code{GRaster} is categorical, then return the category label of each cell. If \code{values} is also \code{TRUE}, then the cell value will also be returned.}

\item{xy}{Logical: If \code{TRUE}, return the longitude and latitude of each point. Default is \code{FALSE}.}

\item{strata}{Either \code{NULL} (default), or a \code{GVector} defining strata. If supplied, the \code{size} argument will be interpreted as number of points to place per geometry in \code{strata}. Note that using strata can dramatically slow the process.}

\item{byStratum}{Logical: If \code{FALSE} (default), then \code{size} number of points will be placed within the entire area delineated by \code{strata}. If \code{TRUE}, then \code{size} points will be placed within each subgeometry of \code{strata}.}

\item{zlim}{Either \code{NULL} (default), or a vector of two numbers defining the lower and upper altitudinal bounds of coordinates. This cannot be combined with \code{values = TRUE} or \code{cats = TRUE}.}

\item{seed}{Either \code{NULL} (default) or an integer: Random number seed. If this is \code{NULL}, the a seed will be set randomly. Values will be rounded to the nearest integer.}

\item{verbose}{Logical: If \code{TRUE}, display progress. Default is \code{FALSE}.}
}
\value{
A \code{data.frame}, \code{data.table}, or \code{GVector}.
}
\description{
\code{spatSample()} randomly locates points across a \code{GRaster} or \code{GVector}. It can return a \code{GVector}, the coordinates, values associated with the points, or all of these. If you want to generate a raster with randomly-sampled cells, see \code{\link[=sampleRast]{sampleRast()}}.
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

# Example data
madElev <- fastData("madElev") # raster

# Convert to GRasters and GVectors
elev <- fast(madElev)

### spatSample()

# Random points as data.frame or data.table:
randVals <- spatSample(elev, size = 20, values = TRUE)
randVals

# Random points as a points GVector:
randPoints <- spatSample(elev, size = 20, as.points = TRUE)
randPoints
plot(elev)
plot(randPoints, add = TRUE)

# Random points in a select area:
madCoast <- fastData("madCoast4") # vector
coast <- fast(madCoast)
ant <- coast[coast$NAME_4 == "Antanambe"] # subset

restrictedPoints <- spatSample(elev, size = 20, as.points = TRUE,
   strata = ant)

plot(elev)
plot(ant, add = TRUE)
plot(restrictedPoints, add = TRUE) # note 20 points for entire geometry

# Random points, one set per subgeometry:
stratifiedPoints <- spatSample(elev, size = 20, as.points = TRUE,
   strata = ant, byStratum = TRUE)

plot(elev)
plot(ant, add = TRUE)
plot(stratifiedPoints, pch = 21, bg = "red", add = TRUE) # note 20 points per subgeometry

# Random categories:
madCover <- fastData("madCover") # raster
cover <- fast(madCover)

randCover <- spatSample(cover, size = 20, values = TRUE,
     cat = TRUE, xy = TRUE)
randCover

### sampleRast()

# Random cells in non-NA cells:
rand <- sampleRast(elev, 10000)
plot(rand)
nonnacell(rand)

# Use custom values for the mask:
randCustomMask <- sampleRast(elev, 10000, maskvalues = 1:20)
plot(randCustomMask)

# Force selected values to a custom value:
randCustomUpdate <- sampleRast(elev, 10000, updatevalue = 7)
plot(randCustomUpdate)

# Custom values for mask and set selected cells to custom value:
randAll <- sampleRast(elev, 10000, maskvalues = 1:20, updatevalue = 7)
plot(randAll)

}
}
\seealso{
\code{\link[=sampleRast]{sampleRast()}}, \code{\link[terra:sample]{terra::spatSample()}}, tool \code{v.random} in \strong{GRASS} (see \code{grassHelp("v.random")})
}
