% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance.r
\name{distance,GRaster,missing-method}
\alias{distance,GRaster,missing-method}
\alias{distance}
\alias{distance,GRaster,GVector-method}
\alias{distance,GVector,GVector-method}
\alias{distance,GVector,missing-method}
\title{Geographic distance}
\usage{
\S4method{distance}{GRaster,missing}(
  x,
  y,
  target = NA,
  fillNA = TRUE,
  unit = "meters",
  method = ifelse(is.lonlat(x), "geodesic", "Euclidean"),
  minDist = NULL,
  maxDist = NULL
)

\S4method{distance}{GRaster,GVector}(
  x,
  y,
  fillNA = TRUE,
  thick = TRUE,
  unit = "meters",
  method = ifelse(is.lonlat(x), "geodesic", "Euclidean"),
  minDist = NULL,
  maxDist = NULL
)

\S4method{distance}{GVector,GVector}(x, y, unit = "meters", minDist = NULL, maxDist = NULL)

\S4method{distance}{GVector,missing}(x, y, unit = "meters", minDist = NULL, maxDist = NULL)
}
\arguments{
\item{x}{A \code{GRaster} or \code{GVector}.}

\item{y}{Either missing, or a \code{GVector}.}

\item{target}{Numeric: Only applicable for case 1, when \code{x} is a \code{GRaster} and \code{y} is missing.  If this is \code{NA} (default), then cells with \code{NA}s have their values replaced with the distance to the nearest non-\code{NA} cell. If this is another value, then cells with these values have their values replaced with the distance to any other cell (meaning, both \code{NA} and non-\code{NA}, except for cells with this value).}

\item{fillNA}{Logical: Determines which raster cells to fill with distances.
\itemize{
\item Case 1, when \code{x} is a \code{GRaster} and \code{y} is missing: If \code{TRUE} (default), fill values of \code{NA} cells with distances to non-\code{NA} cells. If \code{FALSE}, fill non-\code{NA} cells width distance to \code{NA} cells.
\item Case 2, when \code{x} is a \code{GRaster} and \code{y} is a \code{GVector}: If \code{TRUE} (default), then the returned raster will contain the distance from the cell to the closest feature in the vector. If \code{FALSE}, then cells covered by the vector will have their values replaced with the distance to the nearest cell not covered, and cells that are not covered by the vector will have values of 0.
\item Case 3, when \code{x} is a \code{GVector} and \code{y} is a \code{GVector}: This argument is not used in this case.
}}

\item{unit}{Character: Units of the output. Any of:
\itemize{
\item \code{"meters"}, \code{"metres"}, or \code{"m"} (default)
\item \code{"kilometers"} or \code{"km"}
\item \code{"miles"} or \code{"mi"}
\item \code{"nautical miles"} or \code{"nmi"}
\item \code{"yards"} or \code{"yd"}
\item \code{"feet"} or \code{"ft"} -- international, 1 foot exactly equals 0.3048 meters
}

Partial matching is used and case is ignored.}

\item{method}{Character: The type of distance to calculate. Partial matching is used and capitalization is ignored. Possible values include:
\itemize{
\item \code{Euclidean} (default for projected rasters): Euclidean distance.
\item \code{geodesic} (default for unprojected rasters): Geographic distance. If \code{x} is unprojected (e.g., WGS84 or NAD83), then the \code{method} must be \code{"geodesic"}.
\item \code{squared}: Squared Euclidean distance (faster than just Euclidean distance but same rank--good for cases where only order matters).
\item \code{maximum}: Maximum Euclidean distance.
\item \code{Manhattan}: Manhattan distance (i.e., "taxicab" distance, distance along cells going only north-south and east-west and never along a diagonal).
}}

\item{minDist, maxDist}{Either \code{NULL} (default) or numeric values: Ignore distances less than or greater than these distances.}

\item{thick}{Logical: Only applicable for case 2, when \code{x} is a \code{GRaster} and \code{y} is a \code{GVector}. If \code{TRUE} (default), then the vector will be represented by "thickened" lines (i.e., any cell that the line/boundary touches, not just the ones on the rendering path).}
}
\value{
If \code{x} is a \code{GRaster}, then the output is a \code{GRaster}. If \code{x} is a \code{GVector}, then the output is a numeric vector.
}
\description{
This function produces a raster or a matrix of geographic distances, depending on the input:

\strong{Case 1: Argument \code{x} is a \code{GRaster} and \code{y} is missing:} By default, this function replaces values in all \code{NA} cells with the distance between them and their closest non-\code{NA} cell. Alternatively, all non-\code{NA} cells can have their values replaced by the distance to \code{NA} cells. You can also specify which cells (by value) have their values replaced by distance to other cells.

\strong{Case 2: Argument \code{x} is a \code{GRaster} and \code{y} is a \code{GVector}:} All cells in the raster have their value replaced by the distance to the nearest features in the \code{GVector}. Alternatively, calculate the distance from any cell covered by a vector object and the nearest cell \emph{not} covered by a vector object. Note that the vector is rasterized first.

\strong{Case 3: Argument \code{x} is a \code{GVector} and \code{y} is a \code{GVector}:} A matrix of pairwise distances between all features in one versus the other \code{GVector} is returned.

\strong{Case 4: Argument \code{x} is a \code{GVector} and \code{y} is missing:} A matrix of pairwise distances between all features in the \code{GVector} is returned.
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

# Elevation raster, rivers vector, locations of Dypsis plants
madElev <- fastData("madElev")
madRivers <- fastData("madRivers")
madDypsis <- fastData("madDypsis")

# Convert a SpatRaster to a GRaster, and sf to a GVector
elev <- fast(madElev)
rivers <- fast(madRivers)
dypsis <- fast(madDypsis)

### case 1: GRaster by itself

# Distance between NA cells and nearest non-NA cells
naDist <- distance(elev)
names(naDist) <- "NA Distance"
plot(naDist)

# Distance between non-NA cells and nearest NA cells
nonNaDist <- distance(elev, fillNA = FALSE)
names(nonNaDist) <- "non-NA Distance"
plot(nonNaDist)

# Distance between cells with an elevation of 3 and any other cell that != 3
distFocal3 <- distance(elev, target = 3)
names(distFocal3) <- "Distance from 3"
plot(distFocal3)

# Distance between any cell and cells with a value of 3
distTo3 <- distance(elev, fillNA = FALSE, target = 3)
names(distTo3) <- "Distance to 3"
plot(distTo3)

### Case 2: GRaster and GVector
distToVect <- distance(elev, rivers)

plot(distToVect)
plot(rivers, add = TRUE)

### Case 3: GVector vs GVector
plot(rivers)
plot(dypsis, add = TRUE)

distToRivers <- distance(dypsis, rivers, unit = "yd")
distToPlants <- distance(rivers, dypsis)
distToRivers
distToPlants

### Case 4: GVector vs itself
distToItself <- distance(dypsis)
distToItself

}
}
\seealso{
\code{\link[terra:distance]{terra::distance()}}; \strong{GRASS} modules \code{r.grow.distance} and \code{v.distance}
}
