% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fastbackward.R
\name{fastbackward}
\alias{fastbackward}
\title{Performs Fast Backward Elimination by AIC}
\usage{
fastbackward(
  object,
  scope,
  scale = 0,
  trace = 1,
  keep = NULL,
  steps = 1000,
  k = 2,
  ...
)
}
\arguments{
\item{object}{an object representing a model of an appropriate class. This is
used as the initial model in the stepwise search.}

\item{scope}{defines the range of models examined in the stepwise search. This
should be missing or a single formula. If a formula is included, all of the
components on the right-hand-side of the formula are always included in the model.
If missing, then only the intercept (if included) is always included in the model.}

\item{scale}{used in the definition of the AIC statistic for selecting the models,
currently only for \link{lm} and \link{aov} models (see \link{extractAIC} for details).}

\item{trace}{if positive, information is printed during the running of \code{fastbackward}.
Larger values may give more detailed information. If \code{trace} is greater than 1, then
information about which variables at each step are not considered for removal due
to the bounding algorithm are printed.}

\item{keep}{a filter function whose input is a fitted model object and the associated \code{AIC} statistic,
and whose output is arbitrary. Typically \code{keep} will select a subset of the components
of the object and return them. The default is not to keep anything.}

\item{steps}{the maximum number of steps to be considered. The default is 1000
(essentially as many as required). It is typically used to stop the process early.}

\item{k}{the multiple of the number of degrees of freedom used for the penalty.
Only \code{k = 2} gives the genuine AIC: \code{k = log(n)} is sometimes referred to as BIC or SBC.}

\item{...}{any additional arguments to \link{extractAIC}.}
}
\value{
The stepwise-selected model is returned, with up to two additional components.
There is an "\code{anova}" component corresponding to the steps taken in the search,
as well as a "\code{keep}" component if the \verb{keep=} argument was supplied in the call.
The "\verb{Resid. Dev}" column of the analysis of deviance table refers to a constant
minus twice the maximized log likelihood: it will be a deviance only in cases
where a saturated model is well-defined (thus excluding \code{lm}, \code{aov} and \code{survreg} fits,
for example)
}
\description{
Performs backward elimination by AIC, backward elimination is
performed with a bounding algorithm to make it faster.
}
\details{
The bounding algorithm allows us to avoid fitting models that cannot possibly
provide an improvement in AIC. At a high-level, the algorithm basically
works by identifying important predictors whose removal from the current model
cannot possibly improve upon the current AIC.

Test statistics, p-values, and confidence intervals from the final selected
model are not reliable due to the selection process. Thus, it is not recommended
to use these quantities.

See more details at \link[MASS:stepAIC]{MASS::stepAIC}.
}
\examples{
# Loading fastbackward
library(fastbackward)

# Using examples provided in MASS::stepAIC, but with fastbackward instead
## aov with quine dataset
quine.hi <- aov(log(Days + 2.5) ~ .^4, MASS::quine)
quine.nxt <- update(quine.hi, . ~ . - Eth:Sex:Age:Lrn)
quine.stp <- fastbackward(quine.nxt, trace = FALSE)
quine.stp$anova

## lm with cpus dataset
cpus1 <- MASS::cpus
for(v in names(MASS::cpus)[2:7])
 cpus1[[v]] <- cut(MASS::cpus[[v]], unique(quantile(MASS::cpus[[v]])),
                   include.lowest = TRUE)
cpus0 <- cpus1[, 2:8]  # excludes names, authors' predictions
cpus.samp <- sample(1:209, 100)
cpus.lm <- lm(log10(perf) ~ ., data = cpus1[cpus.samp,2:8])
cpus.lm2 <- fastbackward(cpus.lm, trace = FALSE)
cpus.lm2$anova

## glm with bwt dataset
example(birthwt, package = "MASS")
birthwt.glm <- glm(low ~ ., family = binomial, data = bwt)
birthwt.step <- fastbackward(birthwt.glm, trace = FALSE)
birthwt.step$anova

}
\seealso{
\link[MASS:stepAIC]{MASS::stepAIC}, \link[MASS:dropterm]{MASS::dropterm}, and \link{extractAIC}
}
