% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{rdowndate}
\alias{rdowndate}
\title{Fast downdating of the R matrix}
\usage{
rdowndate(R, k = NULL, m = NULL, U = NULL, fast = NULL, type = NULL)
}
\arguments{
\item{R}{a \eqn{p\times p} upper triangular matrix.}

\item{k}{position where the columns or the rows are removed.}

\item{m}{number of columns or rows to be removed.}

\item{U}{a \eqn{p\times m} matrix of rows to be removed. It should only be provided when rows are being removed.}

\item{fast}{fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.}

\item{type}{either 'row' of 'column', for removing rows or columns.}
}
\value{
R the updated R matrix.
}
\description{
rdowndate provides the update of the thin R matrix of the QR factorization after the deletion of \eqn{m\geq 1} rows or columns to the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The R factorization of the matrix \eqn{X} returns the upper triangular matrix \eqn{R\in\mathbb{R}^{p\times p}} such that \eqn{X^\top X=R^\top R}. See Golub and Van Loan (2013) for further details on the method.
}
\examples{
## Remove one column
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R
R1     <- R[1:p,]

## select the column to be deleted from X and update X
k  <- 2
X1 <- X[, -k]

## downdate the R decomposition
R2 <- fastQR::rdowndate(R = R1, k = k,
                        m = 1, type = "column")

## check
max(abs(crossprod(R2) - crossprod(X1)))

## Remove m columns
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R
R1     <- R[1:p,]

## select the column to be deleted from X and update X
k  <- 2
X1 <- X[, -c(k,k+1)]

## downdate the R decomposition
R2 <- fastQR::rdowndate(R = R1, k = k,
                        m = 2, type = "column")

## check
max(abs(crossprod(R2) - crossprod(X1)))

## Remove one row
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R
R1     <- R[1:p,]

# select the row to be deleted from X and update X
k  <- 5
X1 <- X[-k,]
U  <- as.matrix(X[k,], p, 1)

## downdate the R decomposition
R2 <-  rdowndate(R = R1, k = k, m = 1,
                 U = U, fast = FALSE, type = "row")

## check
max(abs(crossprod(R2) - crossprod(X1)))

## Remove m rows
## create data: n > p
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)
output <- fastQR::qr(X, type = "householder",
                    nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R
R1     <- R[1:p,]

## select the rows to be deleted from X and update X
k  <- 2
m  <- 2
X1 <- X[-c(k,k+m-1),]
U  <- t(X[k:(k+m-1), ])

## downdate the R decomposition
R2 <- rdowndate(R = R1, k = k, m = m,
                U = U, fast = FALSE, type = "row")

## check
max(abs(crossprod(R2) - crossprod(X1)))

}
\references{
\insertRef{golub_van_loan.2013}{fastQR}

\insertRef{bjorck.2015}{fastQR}

\insertRef{bjorck.2024}{fastQR}

\insertRef{bernardi_etal.2024}{fastQR}
}
