#' @title Predict Method for fastNaiveBayes.gaussian fits
#' @description Uses a fastNaiveBayes.gaussian model and a new data set to create the classifications.
#'     This can either be the raw probabilities generated by the fastNaiveBayes.gaussian model or the classes themselves.
#'
#' @param object A fitted object of class "fastNaiveBayes.gaussian".
#' @param newdata A numeric matrix with 1's and 0's to indicate the presence or absence of features. A Sparse dgcMatrix is also accepted.
#'     Note that if newdata contains features that were not encountered in the training data, these are omitted from the prediction.
#'     Furthermore, newdata can contain fewer features than encountered in the training data. In this case, newdata will be padded with
#'     extra columns all filled with 0's.
#' @param type If "raw", the conditional a-posterior probabilities for each class are returned, and the class with maximal probability else.
#' @param sparse Use a sparse Matrix? If true a sparse matrix will be constructed from x, which can give up to a 40\% speed up.
#'     It's possible to directly feed a sparse dgcMatrix as x, which will set this parameter to TRUE
#' @param ... Not used.
#' @return If type = 'class', a factor with classified class levels. If type = 'raw', a matrix with the predicted probabilities of
#'     each class, where each column in the matrix corresponds to a class level.
#'
#' @export
#' @import Matrix
#'
#' @details In the extremely unlikely case that two classes have the exact same estimated probability, the first encountered class
#'     is used as the classification and a warning is issued.
#'
#'     Using a sparse matrix directly can be especially useful if it's necessary to use predict multiple times on the same matrix or
#'     on different subselections of the same initial matrix, see examples for further details.
#' @examples
#' rm(list = ls())
#' require(mlbench)
#' require(Matrix)
#' 
#' # Load BreastCancer data
#' data(BreastCancer)
#' dim(BreastCancer)
#' levels(BreastCancer$Class)
#' head(BreastCancer)
#' 
#' # Gaussian distribution example
#' data_mat <- BreastCancer[, c("Class", "Cl.thickness", "Cell.size", "Cell.shape", "Marg.adhesion")]
#' 
#' y <- data_mat[, "Class"]
#' data_mat <- data_mat[, setdiff(colnames(data_mat), c("Class"))]
#' for (i in 1:ncol(data_mat)) {
#'   data_mat[[i]] <- as.numeric(data_mat[[i]])
#' }
#' 
#' model <- fastNaiveBayes.gaussian(data_mat[1:400, ], y[1:400], laplace = 1, sparse = FALSE)
#' preds <- predict(model, newdata = data_mat[401:nrow(data_mat), ], type = "class")
#' 
#' mean(preds != y[401:length(y)])
predict.fastNaiveBayes.gaussian <- function(object, newdata, type = c("class", "raw", "rawprob"),
                                            sparse = FALSE, ...) {
  type <- match.arg(type)
  if (class(newdata)[1] != "dgCMatrix") {
    if (!is.matrix(newdata)) {
      newdata <- as.matrix(newdata)
    }
    if (sparse) {
      newdata <- Matrix(newdata, sparse = TRUE)
    }
  } else {
    sparse <- TRUE
  }
  names <- object$names

  other_names <- setdiff(names, colnames(newdata))
  if (length(other_names) > 0) {
    if (sparse) {
      other_mat <- Matrix(0L, nrow = nrow(newdata), ncol = length(other_names), sparse = TRUE)
    } else {
      other_mat <- matrix(0L, nrow = nrow(newdata), ncol = length(other_names))
    }
    colnames(other_mat) <- other_names

    newdata <- cbind(newdata, other_mat)
  }
  newdata <- newdata[, names]
  if (length(names) == 1) {
    newdata <- as.matrix(newdata)
  }

  data <- object$probability_table

  probs <- NULL
  for (j in 1:length(data)) {
    level <- data[[j]]
    level_probs <- NULL
    for (i in 1:ncol(newdata)) {
      if (is.null(level_probs)) {
        inter_probs <- stats::dnorm(newdata[, i],
          mean = level$means[[i]],
          sd = level$stddev[[i]]
        )
        level_probs <- log(inter_probs)
      } else {
        inter_probs <- stats::dnorm(newdata[, i],
          mean = level$means[[i]],
          sd = level$stddev[[i]]
        )
        level_probs <- level_probs + log(inter_probs)
      }
    }
    if (is.null(probs)) {
      probs <- as.matrix(level_probs)
    } else {
      probs <- cbind(probs, level_probs)
    }

    colnames(probs)[j] <- level$level
  }

  if (type == "rawprob") {
    return(probs)
  }

  priors <- as.vector(object$priors)
  probs <- exp(probs)
  for (i in 1:length(priors)) {
    probs[, i] <- probs[, i] * priors[i]
  }

  denom <- rowSums(probs)
  probs <- probs / denom

  if (type == "class") {
    if (any(max.col(probs, ties.method = "last") != max.col(probs, ties.method = "first"))) {
      warning("Exact same estimated probabilities occured. First encountered class used as classification")
    }
    class <- names(object$priors)[max.col(probs, ties.method = "first")]
    return(as.factor(class))
  }
  return(probs)
}
