% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PackageOxygen.R
\name{hill.ts}
\alias{hill.ts}
\alias{print.hill.ts}
\title{Compute the extreme quantile procedure on a time dependent data}
\usage{
hill.ts(X, t, Tgrid = seq(min(t), max(t), length = 10), h,
  kernel = TruncGauss.kernel, kpar = NULL, CritVal = 3.6,
  gridlen = 100, initprop = 1/10, r1 = 1/4, r2 = 1/20)

\method{print}{hill.ts}(x, ...)
}
\arguments{
\item{X}{a vector of the observed values.}

\item{t}{a vector of time covariates which should have the same length as X.}

\item{Tgrid}{a grid of time (can be any sequence in the interval \code{[min(t) , max(t)]} ).}

\item{h}{a bandwidth value (vector values are not admitted).}

\item{kernel}{a kernel function used to compute the weights in the time domain, with default the truncated Gaussian kernel.}

\item{kpar}{a value for the kernel function parameter, with no default value.}

\item{CritVal}{a critical value associated to the kernel function given by \code{\link{CriticalValue}}. The default value is 3.6 corresponding to the truncated Gaussian kernel.}

\item{gridlen}{the gridlen parameter used in the function hill.adapt. The length of the grid for which the test will be done.}

\item{initprop}{the initprop parameter used in the function hill.adapt. The initial proportion at which we will begin to test the model.}

\item{r1}{the r1 parameter used in the function hill.adapt. The proportion from the right that we will skip in the test statistic.}

\item{r2}{the r2 parameter used in the function hill.adapt. The proportion from the left that we will skip in the test statistic.}

\item{x}{the result of the hill.ts function}

\item{...}{further arguments to be passed from or to other methods.}
}
\value{
\item{Tgrid}{the given vector \eqn{Tgrid}.}
\item{h}{the given value \eqn{h}.}
\item{Threshold}{the adaptive threshold \eqn{\tau} for each \eqn{t} in \eqn{Tgrid}.}
\item{Theta}{the adaptive estimator of \eqn{\theta} for each \eqn{t} in \eqn{Tgrid}.}
}
\description{
Compute the function hill.adapt on time dependent data.
}
\details{
For a given time serie and kernel function, the function hill.ts will give the results of the adaptive procedure for each \eqn{t}. The adaptive procedure is described in Durrieu et al. (2005).

The kernel implemented in this packages are : Biweight kernel, Epanechnikov kernel, Rectangular kernel, Triangular kernel and the truncated Gaussian kernel.
}
\examples{

theta <- function(t){
   0.5+0.25*sin(2*pi*t)
 }
n <- 5000
t <- 1:n/n
Theta <- theta(t)
Data <- NULL
Tgrid <- seq(0.01, 0.99, 0.01)
#example with fixed bandwidth
\dontrun{ #For computing time purpose
  for(i in 1:n){
     Data[i] <- rparetomix(1, a = 1/Theta[i], b = 5/Theta[i]+5, c = 0.75, precision = 10^(-5))
   }

  #example
  hgrid <- bandwidth.grid(0.009, 0.2, 20, type = "geometric")
  TgridCV <- seq(0.01, 0.99, 0.1)
  hcv <- bandwidth.CV(Data, t, TgridCV, hgrid, pcv = 0.99, TruncGauss.kernel,
                     kpar = c(sigma = 1), CritVal = 3.6, plot = TRUE)

  Tgrid <- seq(0.01, 0.99, 0.01)
  hillTs <- hill.ts(Data, t, Tgrid, h = hcv$h.cv, kernel = TruncGauss.kernel,
             kpar = c(sigma = 1), CritVal = 3.6,gridlen = 100, initprop = 1/10, r1 = 1/4, r2 = 1/20)
  plot(hillTs$Tgrid, hillTs$Theta, xlab = "t", ylab = "Estimator of theta")
  lines(t, Theta, col = "red")

}


}
\references{
Durrieu, G. and Grama, I. and Pham, Q. and Tricot, J.- M (2015). Nonparametric adaptive estimator of extreme conditional tail probabilities quantiles. Extremes, 18, 437-478.
}
\seealso{
\code{\link{hill.adapt}}, \code{\link{Biweight.kernel}}, \code{\link{Epa.kernel}}, \code{\link{Rectangular.kernel}}, \code{\link{Triang.kernel}}, \code{\link{TruncGauss.kernel}}
}
