#' Generate spectrogram panels for a seismic network
#' 
#' The function generates a set of spectrogram (PSD) panels on single to  
#' several hours basis. It depends on seismic files being organised in a 
#' coherent structure as, for example generated by
#' \code{aux_organisecubefiles}. The function is similar to 
#' \code{aux_psdsummary} but arranges PSDs of all stations by time, rather 
#' than creating individual PSDs by time and station.
#' 
#' The function calls a series of other functions, partly with modified 
#' default values, which can be changed by the ...-argument. By default, 
#' the seismic files are imported as eseis objects using 
#' \code{aux_getevent(..., eseis = TRUE)}. The signals are deconvolved with
#' \code{signal_deconvolve()} using the default options, i.e., 
#' \code{sensor = "TC120s"} and \code{logger = "Cube3extBOB"}.
#' Then, the signals are bandpass filtered with \code{signal_filter}, using 
#' \code{f = c(1, 90)}. The PSDs are calculated with \code{signal_spectrogram} 
#' using \code{Welch = TRUE}, \code{window = 30} and \code{window_sub = 15}. 
#' 
#' This and all other aux-functions are primarily written for internal use 
#' in the GFZ Geomorphology Section group members and their usual data 
#' handling scheme. Thus, they may be of limited use when adopted for other 
#' scopes. However, many of these functions are internally consistent in 
#' usage.
#' 
#' @param station \code{Character} value, seismic station ID, which must
#' correspond to the ID in the file name of the data directory structure 
#' (cf. \code{aux_organisecubefiles}).
#' 
#' @param component \code{Character} value, seismic component, which must
#' correspond to the component name in the file name of the data directory  
#' structure (cf. \code{aux_organisecubefiles}). Default is 
#' \code{"BHZ"} (vertical component).
#' 
#' @param period \code{POSIXct} vector of length two, time period to be 
#' processed. 
#' 
#' @param span \code{Numeric} vector, time span per PSD in hours. Value can 
#' range between 1 and 24. For each time span a separate jpeg-file will be 
#' produced. Default is \code{1} hour.
#' 
#' @param input_dir \code{Character} value, path to directory where the 
#' seismic files are stored. 
#' 
#' @param output_dir \code{Character} value, path to directory where PSD
#' image files are saved to.
#' 
#' @param cpu \code{Numeric} value, fraction of CPUs to use for parallel 
#' processing. If omitted, one CPU is used.
#' 
#' @param aggregate \code{Numeric} vector of length two, aggregation factors 
#' for the processed PSD matrics. First entry denotes time aggregation, 
#' second entry frequency aggregation. Default is \code{c(1, 5)}.
#' 
#' @param n_dates \code{Numeric} value, final number of spectra per output 
#' PSD. Default is \code{2000}.
#' 
#' @param jpg_dim \code{Numeric} vector of length four, JPEG image properties 
#' in the form \code{c(width, height, resolution, quality)}. Default is 
#' \code{c(4444, 2500, 300, 90)}.
#' 
#' @param verbose \code{Logical} value, optional screen output of processing 
#' progress. Default is \code{FALSE}.
#' 
#' @param \dots Additional arguments passed to different functions. See 
#' details section for default values.
#' 
#' @return A set of JPEG images wirtten to disk
#' 
#' @author Michael Dietze
#' 
#' @keywords eseis
#' 
#' @examples
#' 
#' \dontrun{
#' 
#' ## PSD generation with minimum input arguments
#' aux_psdpanels(station = stations$ID, 
#'               input_dir = "input/")
#'               
#' }
#' 
#' @export aux_psdpanels
aux_psdpanels <- function(
  
  station,
  component = "BHZ",
  period,
  span = 1,
  input_dir,
  output_dir,
  cpu,
  aggregate = c(1, 5),
  n_dates = 2000,
  jpg_dim = c(4444, 2500, 300, 90),
  verbose = FALSE,
  ...
) {
  
  ## check/set additional arguments -------------------------------------------
  extraArgs <- list(...)
  
  ## set sensor type
  if("sensor" %in% names(extraArgs)) {
    
    sensor <- extraArgs$sensor
  } else {
    
    sensor <- "TC120s"
  }
  
  ## set logger type
  if("logger" %in% names(extraArgs)) {
    
    logger <- extraArgs$logger
  } else {
    
    logger <- "Cube3extBOB"
  }
  
  ## set NA replace option
  if("na.replace" %in% names(extraArgs)) {
    
    na.replace <- extraArgs$na.replace
  } else {
    
    na.replace <- FALSE
  }
  
  ## set filter frequencies
  if("f" %in% names(extraArgs)) {
    
    f <- extraArgs$f
  } else {
    
    f <- c(1, 90)
  }
  
  ## set filter type
  if("type" %in% names(extraArgs)) {
    
    type <- extraArgs$type
  } else {
    
    type <- c("BP")
  }
  
  ## set Welch option for PSD
  if("Welch" %in% names(extraArgs)) {
    
    Welch <- extraArgs$Welch
  } else {
    
    Welch <- TRUE
  }
  
  ## set window for PSD
  if("window" %in% names(extraArgs)) {
    
    window <- extraArgs$window
  } else {
    
    window <- 30
  }
  
  ## set sub window for PSD
  if("window_sub" %in% names(extraArgs)) {
    
    window_sub <- extraArgs$window_sub
  } else {
    
    window_sub <- 15
  }
  
  ## set multitaper option for PSD
  if("multitaper" %in% names(extraArgs)) {
    
    multitaper_psd <- extraArgs$multitaper
  } else {
    
    multitaper_psd <- FALSE
  }
  
  ## set overlap option for PSD
  if("overlap" %in% names(extraArgs)) {
    
    overlap_psd <- extraArgs$overlap
  } else {
    
    overlap_psd <- FALSE
  }
  
  ## set overlap option for PSD
  if("overlap_sub" %in% names(extraArgs)) {
    
    overlap_sub_psd <- extraArgs$overlap_sub
  } else {
    
    overlap_sub_psd <- FALSE
  }
  
  ## set z-limits
  if("zlim" %in% names(extraArgs)) {
    
    zlim_psd <- extraArgs$zlim
  } else {
    
    zlim_psd <- c(-180, -100)
  }
  
  ## set legend
  if("legend" %in% names(extraArgs)) {
    
    legend <- extraArgs$legend
  } else {
    
    legend <- TRUE
  }
  
  ## remove keywords from plot arguments
  keywords <- c("zlim", "legend", "f", "sensor", "logger", 
                "type", "Welch", "window", "window_sub", 
                "na.replace", "multitaper", "overlap", "overlap_sub")
  
  extraArgs <- extraArgs[!names(extraArgs)%in%keywords]
  
  ## check/set fraction of CPUs to use
  if(missing(cpu) == TRUE) {
    
    cpu <- NA
  }
  
  ## create directory structure -----------------------------------------------
  
  ## check/set output directory
  if(missing(output_dir) == TRUE) {
    
    output_dir <- file.path(tempdir(), "output")
    print(paste("Output will be written to", output_dir))
  }
  
  if(dir.exists(output_dir) == FALSE) {
    
    warning("Output directory did not exist. Created by function")
    dir.create(output_dir)
  }
  
  ## GENERATE PROCESSING PERIOD -----------------------------------------------
  
  ## make time snippets vector
  t_get <- seq(from = period[1], 
               to = period[2], 
               by = span * 60 * 60)
  
  ## detect and adjust number of cores to use
  cores <- parallel::detectCores()
  
  if(is.na(cpu) == FALSE) {
    
    n_cpu <- floor(cores * cpu)
    cores <- ifelse(cores < n_cpu, cores, n_cpu)
  } else {
    
    cores <- 1
  }
  
  ## initiate cluster
  cl <- parallel::makeCluster(getOption("mc.cores", cores))
  
  invisible(parallel::parLapply(
    cl = cl, 
    X = t_get, 
    fun = function(t_get, span, station, component, input_dir, sensor, 
                   logger, f, type, window, Welch, window_sub, 
                   multitaper_psd, overlap_psd, overlap_sub_psd, 
                   aggregate, output_dir, jpg_dim) {
      
      ## read input files
      s <- try(eseis::aux_getevent(start = t_get - 600, 
                                   duration = span * 60 * 60 + 600, 
                                   station = station, 
                                   component = component, 
                                   dir = input_dir,
                                   eseis = TRUE), 
               silent = TRUE)
      
      ## reorganise data structure
      s <- lapply(X = s, FUN = function(s) {
        
        s[[1]]
      })
      
      ## deconvolve data
      s <- try(eseis::signal_deconvolve(data = s, 
                                        sensor = sensor, 
                                        logger = logger,
                                        na.replace = na.replace),
               silent = TRUE)
      
      ## detrend data
      s <- try(eseis::signal_detrend(data = s),
               silent = TRUE)
      
      ## filter data
      s <- try(eseis::signal_filter(data = s, 
                                    f = f,
                                    type = type),
               silent = TRUE)
      
      ## taper data
      s <- try(eseis::signal_taper(data = s,
                                   n = 1 / s[[1]]$meta$dt * 600 / 2),
               silent = TRUE)
      
      ## create PSD
      P <- try(eseis::signal_spectrogram(data = s, 
                                         window = window, 
                                         Welch = Welch, 
                                         window_sub = window_sub,
                                         multitaper = multitaper_psd,
                                         overlap = overlap_psd,
                                         overlap_sub = overlap_sub_psd),
               silent = TRUE)
      
      ## check/replace PSD if unsuccessfully calculated
      if(class(P)[1] == "try-error") {
        
        dt <- try(s$meta$dt, 
                  silent = TRUE)
        
        if(class(dt)[1] == "try-error" | is.null(dt) == TRUE) {
          
          dt <- 1 / 200
        }
        
        t_null <- seq(from = t_get, 
                      to = t_get + span * 60 * 60, 
                      by = 3600)
        
        f_null <- seq(from = 0, 
                      to = 0.5 / dt, 
                      length.out = 0.5 / dt * 
                        ifelse(Welch == TRUE, window_sub, window))
        
        P_null <- matrix(data = rep(-500, length(t_null) * length(f_null)),
                         ncol = length(t_null), 
                         nrow = length(f_null))
        
        P_plot <- lapply(X = 1:length(station), 
                         FUN = function(i, P_null, t_null, f_null) {
                           
                           list(S = P_null,
                                t = t_null,
                                f = f_null)
                         }, P_null, t_null, f_null)
        
      } else {
        
        P_plot <- lapply(X = P, FUN = function(P) {
          
          P$PSD
        })
      }
      
      ## clip and aggregate PSDs
      P_agg <- lapply(X = P_plot, 
                      FUN = function(P_plot, t_get, span, aggregate) {
                        
                        ## clip PSD
                        P_clip <- P_plot
                        
                        P_clip$S <- P_clip$S[,P_clip$t >= t_get & 
                                               P_clip$t <= t_get + 
                                               span * 60 * 60]
                        
                        P_clip$t <- P_clip$t[P_clip$t >= t_get & 
                                               P_clip$t <= t_get + 
                                               span * 60 * 60]
                        
                        ## aggregate PSD
                        P_agg <- P_clip
                        
                        t_agg <- seq(from = 1, 
                                     by = aggregate[1], 
                                     to = length(P_agg$t))
                        
                        f_agg <- seq(from = 1, 
                                     by = aggregate[2], 
                                     to = length(P_agg$f))
                        
                        P_agg$S <- P_agg$S[f_agg, t_agg]
                        P_agg$t <- P_agg$t[t_agg]
                        P_agg$f <- P_agg$f[f_agg]
                        
                        return(P_agg)
                        
                      }, t_get, span, aggregate)
      
      ## prepare output device
      jpeg(filename = paste(output_dir,
                            "PSD_",
                            format(x = t_get, 
                                   format = "%Y_%m_%d_%H_%M"),
                            ".jpg", 
                            sep = ""), 
           width = jpg_dim[1], 
           height = jpg_dim[2], 
           res = jpg_dim[3],
           quality = jpg_dim[4])
      
      ## create plot panels
      graphics::par(mfcol = c(length(station), 1),
                    mar = c(2, 4, 1, 1))
      
      ## generate plot
      for(j in 1:length(P_agg)) {
        
        do.call(what = eseis::plot_spectrogram, 
                args = c(list(data = P_agg[[j]],
                              main = paste(station[j], 
                                           " | (f = ", 
                                           f[1],
                                           "-",
                                           f[2],
                                           ")",
                                           sep = ""), 
                              zlim = zlim_psd),
                         extraArgs))
      }
      
      ## close plot device
      dev.off()
      
    }, span, station, component, input_dir, sensor, 
    logger, f, type, window, Welch, window_sub, 
    multitaper_psd, overlap_psd, overlap_sub_psd, 
    aggregate, output_dir, jpg_dim))
 
  ## stop cluster
  parallel::stopCluster(cl = cl)
}