% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dsaddle.R
\name{dsaddle}
\alias{dsaddle}
\title{Evaluating the Extended Empirical Saddlepoint (EES) density}
\usage{
dsaddle(y, X, decay, deriv = FALSE, log = FALSE, normalize = FALSE,
  fastInit = FALSE, control = list(), multicore = !is.null(cluster),
  ncores = detectCores() - 1, cluster = NULL)
}
\arguments{
\item{y}{points at which the EES is evaluated (d dimensional vector) or an n by d matrix, each row indicating
a different position.}

\item{X}{n by d matrix containing the data.}

\item{decay}{rate at which the EES falls back on a normal density approximation, fitted to \code{X}. 
It must be a positive number, and it is inversely proportional to the complexity of the fit.
Setting it to \code{Inf} leads to a Gaussian fit.}

\item{deriv}{If TRUE also the gradient of the log-saddlepoint density is returned.}

\item{log}{If TRUE the log of the saddlepoint density is returned.}

\item{normalize}{If TRUE the normalizing constant of the EES density will be computed. FALSE by 
default.}

\item{fastInit}{If TRUE a smart initialization is used to start the solution of the saddlepoint equation 
corresponding to each row of y. It can lead to faster computation if n >> d and d < 5. FALSE by default.}

\item{control}{A list of control parameters with entries:
\itemize{
\item{ \code{method} }{the method used to calculate the normalizing constant. 
                       Either "LAP" (laplace approximation) or "IS" (importance sampling).}
\item{ \code{nNorm} }{if control$method == "IS", this is the number of importance samples used.}
\item{ \code{tol} }{the tolerance used to assess the convergence of the solution to the saddlepoint equation.
                    The default is 1e-6.}
\item{ \code{maxit} }{maximal number of iterations used to solve the saddlepoint equation.
                      The default is 100;}
\item{ \code{ml} }{Relevant only if \code{control$method=="IS"}. n random variables are generated from 
                   a Gaussian importance density with covariance matrix \code{ml*cov(X)}. 
                   By default the inflation factor is \code{ml=2}.}
}}

\item{multicore}{if TRUE the empirical saddlepoint density at each row of y will be evaluated in parallel.}

\item{ncores}{number of cores to be used.}

\item{cluster}{an object of class \code{c("SOCKcluster", "cluster")}. This allowes the user to pass her own cluster,
which will be used if \code{multicore == TRUE}. The user has to remember to stop the cluster.}
}
\value{
A list with entries:
        \itemize{
        \item{ \code{llk} }{the value of the EES log-density at each location y;}
        \item{ \code{mix} }{for each location y, the fraction of saddlepoint used: 
                            1 means that only ESS is used and 0 means that only a Gaussian fit is used;}
        \item{ \code{iter} }{for each location y, the number of iteration needed to solve the 
                             saddlepoint equation;}
        \item{ \code{lambda} }{an n by d matrix, where the i-th row is the solution of the saddlepoint 
                               equation corresponding to the i-th row of y;}
        \item{ \code{grad} }{the gradient of the log-density at y (optional);}
        \item{ \code{logNorm} }{the estimated log normalizing constant (optional);}
        }
}
\description{
Gives a pointwise evaluation of the EES density (and optionally of its gradient) at one or more 
             locations.
}
\examples{
library(esaddle)

### Simple univariate example
set.seed(4141)
x <- rgamma(1000, 2, 1)

# Evaluating EES at several point
xSeq <- seq(-2, 8, length.out = 200)
tmp <- dsaddle(y = xSeq, X = x, decay = 0.05, log = TRUE)  # Un-normalized EES
tmp2 <- dsaddle(y = xSeq, X = x, decay = 0.05,             # EES normalized by importance sampling
                normalize = TRUE, control = list("method" = "IS", nNorm = 500), log = TRUE)

# Plotting true density, EES and normal approximation
plot(xSeq, exp(tmp$llk), type = 'l', ylab = "Density", xlab = "x")
lines(xSeq, dgamma(xSeq, 2, 1), col = 3)
lines(xSeq, dnorm(xSeq, mean(x), sd(x)), col = 2)
lines(xSeq, exp(tmp2$llk), col = 4)
suppressWarnings( rug(x) )
legend("topright", c("EES un-norm", "EES normalized", "Truth", "Gaussian"), 
        col = c(1, 4, 3, 2), lty = 1)
}
\references{
Fasiolo, M., Wood, S. N., Hartig, F. and Bravington, M. V. (2016). 
            An Extended Empirical Saddlepoint Approximation for Intractable Likelihoods. ArXiv http://arxiv.org/abs/1601.01849.
}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com> and Simon N. Wood.
}
