% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/competition.R
\name{competition}
\alias{competition}
\title{Interspecific competition under the influence of temperature trend adapted from the IPCC
projection (RCP2.6 or RCP8.5 scenarios)}
\usage{
competition(
  y_ini = c(N1 = 400, N1 = 400, N1 = 400, N2 = 200, N2 = 200, N2 = 200),
  temp_ini = rep(25, 3),
  temp_cmin = rep(18, 3),
  temp_cmax = c(25, 28, 35),
  ro = rep(0.7, 3),
  r2 = rep(0.7, 3),
  lambda1 = rep(5e-05, 3),
  K2 = rep(5e-05, 3),
  alpha = rep(0.002, 3),
  beta = rep(0.03, 3),
  RCP = 2.6,
  time_start = 2005,
  time_end = 2100,
  leap = 1/50
)
}
\arguments{
\item{y_ini}{Initial population values (must be written with its name: N).}

\item{temp_ini}{Initial temperature.}

\item{temp_cmin}{Minimum critical temperature.}

\item{temp_cmax}{Maximum critical temperature.}

\item{ro}{Population growth rate at optimal temperature of species-1.}

\item{r2}{Population growth rate of species-2.}

\item{lambda1}{Marginal loss a by non-thermodependent intraspecific competition factor of species-1.}

\item{K2}{Carrying capacity of species-2.}

\item{alpha}{Competition coefficient that quantifies the per capita effect of species-2 on species-1.}

\item{beta}{Per capita competition coefficient that quantifies the per capita effect of species-1 on species-2.}

\item{RCP}{Representative concentration trajectories (RCP2.6 and RCP8.5 scenarios).}

\item{time_start}{Start of time sequence.}

\item{time_end}{End of time sequence.}

\item{leap}{Time sequence step.}
}
\value{
(1) A data.frame with columns having the simulated trends.

(2) A four-panel figure where (a), (b), and (c) show the abundance curves of the populations for each
           simulation, where the brown curve corresponds to the abundance of the ectotherm species and
           the green curve to the species not affected by temperature. Panel (d) shows the temperature
           trend curves, as they may differ for each simulation, these will be displayed by the colors
           green, blue, and black respectively.
}
\description{
This function allows simulating the effect of temperature trends according
to RCP2.6 or RCP8.5 scenarios (2014) on the abundances of two competing species, where one of
them is ectothermic.
}
\details{
The function allows simulating simultaneously three potential outcomes for the interaction of
        two competing populations where one is an ectothermic species. The temperature trends that can
        be specified corresponds to IPCC projections under the RCP2.6 or RCP8.5 scenarios.
}
\examples{

#######################################################################
  #Example 1: Different thermal tolerance ranges (scenario RCP2.6).
#######################################################################

temp_cmin <- 18

# Temperature that occurs before the minimum simulation time.
temp_i <- 22

time_end <- 2100

# Temperature that occurs in the maximum time of the simulation.
temp_max <- get_RCP2.6(time_end)+temp_i

# Simulation thermal range.
RS <- temp_max-temp_cmin

temp_cmax1 <- 4/3*RS+temp_cmin
temp_cmax2 <- 2/3*RS+temp_cmin
temp_cmax3 <- 1/3*RS+temp_cmin
temp_ini <- (temp_cmin+temp_cmax3)/2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 300, N2 = 300, N2 = 300),
           temp_ini = rep(temp_ini,3),
           temp_cmin = rep(temp_cmin,3),
           temp_cmax = c(temp_cmax1,temp_cmax2,temp_cmax3),
           ro = rep(0.7,3),
           r2 = rep(0.7,3),
           lambda1 = rep(0.0005,3),
           K2 = rep(1400,3),
           alpha = rep(0.02,3),
           beta = rep(0.3,3),
           RCP = 2.6,
           time_start = 2005,
           time_end = time_end,
           leap = 1/50)
\donttest{
#######################################################################
  #Example 2: Different thermal tolerance ranges (scenario RCP8.5).
#######################################################################

temp_cmin <- 18

# Temperature that occurs before the minimum simulation time.
temp_i <- 22

time_end <- 2100

# Temperature that occurs in the maximum time of the simulation.
temp_max <- get_RCP8.5(time_end)+temp_i

# Simulation thermal range.
RS <- temp_max-temp_cmin

temp_cmax1 <- 4/3*RS+temp_cmin
temp_cmax2 <- 2/3*RS+temp_cmin
temp_cmax3 <- 1/3*RS+temp_cmin
temp_ini <- (temp_cmin+temp_cmax3)/2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 300, N2 = 300, N2 = 300),
           temp_ini = rep(temp_ini,3),
           temp_cmin = rep(temp_cmin ,3),
           temp_cmax = c(temp_cmax1,temp_cmax2,temp_cmax3),
           ro = rep(0.7,3),
           r2 = rep(0.7,3),
           lambda1 = rep(0.0005,3),
           K2 = rep(1400,3),
           alpha = rep(0.02,3),
           beta = rep(0.3,3),
           RCP = 8.5,
           time_start = 2005,
           time_end = time_end,
           leap = 1/50)

#######################################################################
  #Example 3: Different marginal losses by a non-thermodependent
  #           component of intraspecific competition for species-1
  #           (scenario RCP2.6).
#######################################################################

lambda3 <- 0.002
lambda2 <- 1/2*lambda3
lambda1 <- 1/2*lambda2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 200, N2 = 200, N2 = 200),
           temp_ini = rep(25,3),
           temp_cmin = rep(20,3),
           temp_cmax = rep(30,3),
           ro = rep(0.5,3),
           r2 = rep(0.4,3),
           lambda1 = c(lambda1,lambda2,lambda3),
           K2 = rep(1200,3),
           alpha = rep(0.02,3),
           beta = rep(0.3,3),
           RCP = 2.6,
           time_start = 2005,
           time_end = 2100,
           leap = 1/50)

#'#######################################################################
  #Example 4: Different marginal losses by a non-thermodependent
  #           component of intraspecific competition for species-1
  #           (scenario RCP8.5).
#######################################################################

lambda3 <- 0.002
lambda2 <- 1/2*lambda3
lambda1 <- 1/2*lambda2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 200, N2 = 200, N2 = 200),
           temp_ini = rep(25,3),
           temp_cmin = rep(20,3),
           temp_cmax = rep(30,3),
           ro = rep(0.5,3),
           r2 = rep(0.4,3),
           lambda1 = c(lambda1,lambda2,lambda3),
           K2 = rep(1200,3),
           alpha = rep(0.02,3),
           beta = rep(0.3,3),
           RCP = 8.5,
           time_start = 2005,
           time_end = 2100,
           leap = 1/50)

#######################################################################
  #Example 5: Different competition coefficients (scenario RCP2.6).
#######################################################################

alpha1 <- 0.02
alpha2 <- 2*alpha1
alpha3 <- 2*alpha2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 200, N2 = 200, N2 = 200),
           temp_ini = rep(25,3),
           temp_cmin = rep(20,3),
           temp_cmax = rep(30,3),
           ro = rep(0.5,3),
           r2 = rep(0.4,3),
           lambda1 = rep(0.0005,3),
           K2 = rep(1200,3),
           alpha = c(alpha1,alpha2,alpha3),
           beta = rep(0.3,3),
           RCP = 2.6,
           time_start = 2005,
           time_end = 2100,
           leap = 1/50)

#######################################################################
  #Example 6: Different competition coefficients (scenario RCP8.5).
#######################################################################

alpha1 <- 0.02
alpha2 <- 2*alpha1
alpha3 <- 2*alpha2

competition(y_ini = c(N1 = 400, N1 = 400, N1 = 400,
                     N2 = 200, N2 = 200, N2 = 200),
           temp_ini = rep(25,3),
           temp_cmin = rep(20,3),
           temp_cmax = rep(30,3),
           ro = rep(0.5,3),
           r2 = rep(0.4,3),
           lambda1 = rep(0.0005,3),
           K2 = rep(1200,3),
           alpha = c(alpha1,alpha2,alpha3),
           beta = rep(0.3,3),
           RCP = 8.5,
           time_start = 2005,
           time_end = 2100,
           leap = 1/50)
}
}
\references{
IPCC. (2014): Climate Change 2014: Synthesis Report. Contribution of Working Groups I,
           II and III to the Fifth Assessment Report of the Intergovernmental Panel on Climate
           Change [Core Writing Team, R.K. Pachauri and L.A. Meyer (eds.)]. IPCC, Geneva,
           Switzerland, 151 pp.
}
