setClassUnion("matrixORnumeric", c("matrix", "numeric"))
setClassUnion("functionORNULL",members=c("function", "NULL"))
setClassUnion("listORNULL",members=c("list", "NULL"))
setClassUnion("matrixORnumericORfunction", c("matrix", "numeric", "function"))

#' An S4 class to represent an exponential marked Hawkes model
#'
#' This class represents a specification of a marked Hawkes model with exponential kernel.
#' The intensity of the ground process is defined by:
#' \deqn{\lambda(t) = \mu + \int ( \alpha + \Psi ) * exp( -\beta (t-u)) d N(t).}
#' For more details, please see the vignettes.
#'
#' \eqn{\mu} is base intensity.
#' This is generally a constant vector but can be extended to stochastic processes.
#' Currently, piecewise constant mu is also possible. mu is left continous.
#'
#' \eqn{\alpha} is a constant matrix which represents impacts on intensities after events.
#' It is represented by slot \code{mu}.
#'
#' \eqn{\Psi} is for non-constant parts of the impact.
#' It may depend on any information generated by \eqn{N}, \eqn{\lambda}, \eqn{k} and so on.
#' It is represented by slot \code{impact}.
#'
#' \eqn{\beta} is a constant matrix for exponential decay rates.
#' It is represetned by slot \code{beta}.
#'
#' \eqn{k} is mark and represented by slot \code{rmark}.
#'
#' \code{mu}, \code{alpha} and \code{beta} are required slots for every exponential Hawkes model.
#' \code{rmark} and \code{impact} are additional slots.
#'
#' @slot mu numeric value or matrix or function, if numeric, automatically converted to matrix
#' @slot alpha numeric value or matrix or function, if numeric, automatically converted to matrix, exciting term
#' @slot beta numeric value or matrix or function, if numeric,, automatically converted to matrix, exponential decay
#' @slot dimens dimension of the model
#' @slot rmark a function that generates mark for counting process, for simulation
#' @slot dmark a density function for mark, for estimation
#' @slot impact a function that describe the after impact of mark to lambda
#' @slot type_col_map used for multiple kernel
#'
#' @examples
#' MU <- matrix(c(0.2), nrow = 2)
#' ALPHA <- matrix(c(0.75, 0.92, 0.92, 0.75), nrow = 2, byrow=TRUE)
#' BETA <- matrix(c(2.25, 2.25, 2.25, 2.25), nrow = 2, byrow=TRUE)
#' mhspec2 <- new("hspec", mu=MU, alpha=ALPHA, beta=BETA)
#'
#' @export
setClass("hspec",
  slots = list(
    mu = "matrixORnumericORfunction",
    alpha = "matrixORnumericORfunction",
    beta = "matrixORnumericORfunction",
    dimens = "numeric",
    rmark = "functionORNULL",
    dmark = "functionORNULL",
    impact = "functionORNULL",
    type_col_map = "listORNULL"
  )
)

# Initialize the hspec object
#
#
# \code{mu}, \code{alpha} and \code{beta} are required slots for every exponential Hawkes model.
# \code{mark} and \code{impact} are additional slots.
#
# @param .Object hspec
# @param mu numeric value or matrix or function, if numeric, automatically converted to matrix
# @param alpha numeric value or matrix or function, if numeric, automatically converted to matrix, exciting term
# @param beta numeric value or matrix or function, if numeric,, automatically converted to matrix, exponential decay
# @param dimens dimesion of the model
# @param rmark a function that generate mark
# @param impact a function that describe the mark impact
# @param stability_check check the spectral radius
setMethod(
  "initialize",
  "hspec",
  function(.Object, mu, alpha, beta, impact=NULL, type_col_map = NULL, dimens=NULL,
           rmark=NULL, dmark = NULL, stability_check=FALSE){

    # If rmark is not provided, then rmark is constant 1.
    if (is.null(rmark)) rmark <- function(...) 1

    # check the number of arguments
    if(length(formals(rmark)) == 0){
      .Object@rmark <- function(...) rmark()
    } else {
      .Object@rmark <- rmark
    }


    if( !is.function(mu) & !is.matrix(mu) ){
      .Object@mu <- as.matrix(mu)
    } else{
      .Object@mu <- mu
    }
    if( !is.function(alpha) & !is.matrix(alpha) ){
      .Object@alpha <- as.matrix(alpha)
    } else{
      .Object@alpha <- alpha
    }
    if( !is.function(beta) & !is.matrix(beta) ){
      .Object@beta <-  as.matrix(beta)
    } else {
      .Object@beta <- beta
    }
#
#     if( !is.list(type_col_map)){
#       .Object@type_col_map <-  list(type_col_map)
#     } else {
#       .Object@type_col_map <- type_col_map
#     }

    .Object@type_col_map <- type_col_map
    # set dimens
    if( is.null(dimens)){
      if( is.matrix(.Object@mu) ){
        .Object@dimens <- length(.Object@mu)
      } else if ( is.matrix(.Object@alpha) ) {
        .Object@dimens <- nrow(.Object@alpha)
      } else if ( is.matrix(.Object@beta) ) {
        .Object@dimens <- nrow(.Object@beta)
      } else if ( is.function(.Object@alpha) ){
        .Object@dimens <- nrow(evalf(.Object@alpha))
      } else if (is.function(.Object@beta) ){
        .Object@dimens <- nrow(evalf(.Object@beta))
      } else {
        stop("argument \"dimens\" is missing, with no default")
      }
    }


    .Object@dmark <- dmark
    .Object@impact <- impact


    # Check spectral radius, only works for non marked model
    if ( stability_check==TRUE && max(abs(eigen(alpha/beta)$values)) > 1)
      warning("This model may not satisfy the stability condition.")

    methods::callNextMethod()

    .Object


  }
)


