% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectoryMetrics.R
\encoding{UTF-8}
\name{trajectorymetrics}
\alias{trajectorymetrics}
\alias{segmentDistances}
\alias{trajectoryDistances}
\alias{trajectoryLengths}
\alias{trajectoryLengths2D}
\alias{trajectoryAngles}
\alias{trajectoryAngles2D}
\alias{trajectoryProjection}
\alias{trajectoryConvergence}
\alias{trajectoryDirectionality}
\title{Metrics for Ecological Trajectory Analysis}
\usage{
segmentDistances(
  d,
  sites,
  surveys = NULL,
  distance.type = "directed-segment",
  add = TRUE,
  verbose = FALSE
)

trajectoryDistances(
  d,
  sites,
  surveys = NULL,
  distance.type = "DSPD",
  symmetrization = "mean",
  add = TRUE,
  verbose = FALSE
)

trajectoryLengths(
  d,
  sites,
  surveys = NULL,
  relativeToInitial = FALSE,
  all = FALSE,
  verbose = FALSE
)

trajectoryLengths2D(
  xy,
  sites,
  surveys,
  relativeToInitial = FALSE,
  all = FALSE,
  verbose = FALSE
)

trajectoryAngles(
  d,
  sites,
  surveys = NULL,
  all = FALSE,
  relativeToInitial = FALSE,
  stats = TRUE,
  add = TRUE,
  verbose = FALSE
)

trajectoryAngles2D(
  xy,
  sites,
  surveys,
  relativeToInitial = FALSE,
  betweenSegments = TRUE
)

trajectoryProjection(d, target, trajectory, tol = 1e-06, add = TRUE)

trajectoryConvergence(
  d,
  sites,
  surveys = NULL,
  symmetric = FALSE,
  add = TRUE,
  verbose = FALSE
)

trajectoryDirectionality(d, sites, surveys = NULL, add = TRUE, verbose = FALSE)
}
\arguments{
\item{d}{A symmetric \code{\link{matrix}} or an object of class \code{\link{dist}} containing the distance values between pairs of ecosystem states (see details).}

\item{sites}{A vector indicating the site corresponding to each ecosystem state.}

\item{surveys}{A vector indicating the survey corresponding to each ecosystem state (only necessary when surveys are not in order).}

\item{distance.type}{The type of distance index to be calculated (Besse et al. 2016; De Cáceres et al. submitted). For \code{segmentDistances} the available indices are:
\itemize{
\item{\code{Hausdorff}: Hausdorff distance between two segments.}
\item{\code{directed-segment}: Directed segment distance (default).}
\item{\code{PPA}: Perpendicular-parallel-angle distance.}
}
whereas for \code{trajectoryDistances} the available indices are:
\itemize{
\item{\code{Hausdorff}: Hausdorff distance between two trajectories.}
\item{\code{SPD}: Segment path distance.}
\item{\code{DSPD}: Directed segment path distance (default).}
}}

\item{add}{Flag to indicate that constant values should be added (local transformation) to correct triplets of distance values that do not fulfill the triangle inequality.}

\item{verbose}{Provides console output informing about process (useful for large dataset).}

\item{symmetrization}{Function used to obtain a symmetric distance, so that DSPD(T1,T2) = DSPD(T2,T1) (e.g., \code{mean} or \code{min}). If \code{symmetrization = NULL} then the symmetrization is not conducted and the output dissimilarity matrix is not symmetric.}

\item{relativeToInitial}{Flag to indicate that lengths or angles should be calculated with respect to initial survey.}

\item{all}{A flag to indicate that angles are desired for all triangles (i.e. all pairs of segments) in the trajectory. If FALSE, angles are calculated for consecutive segments only.}

\item{xy}{Matrix with 2D coordinates in a Cartesian space (typically an ordination of ecosystem states).}

\item{stats}{A flag to indicate that circular statistics are desired (mean, standard deviation and mean resultant length, i.e. rho)}

\item{betweenSegments}{Flag to indicate that angles should be calculated between trajectory segments or with respect to X axis.}

\item{target}{An integer vector of the ecosystem states to be projected.}

\item{trajectory}{An integer vector of the trajectory onto which target states are to be projected.}

\item{tol}{Numerical tolerance value to determine that projection of a point lies within the trajectory.}

\item{symmetric}{A logical flag to indicate a symmetric convergence comparison of trajectories.}
}
\value{
Function \code{trajectoryDistances} returns an object of class \code{\link{dist}} containing the distances between trajectories (if \code{symmetrization = NULL} then the object returned is of class \code{matrix}).

Function \code{trajectorySegments} returns a list with the following elements:
\itemize{
\item{\code{Dseg}: Distance matrix between segments.}
\item{\code{Dini}: Distance matrix between initial points of segments.}
\item{\code{Dfin}: Distance matrix between final points of segments.}
\item{\code{Dinifin}: Distance matrix between initial points of one segment and the final point of the other.}
\item{\code{Dfinini}: Distance matrix between final points of one segment and the initial point of the other.}
}

Function \code{trajectoryLengths} returns a data frame with the length of each segment on each trajectory and the total length of all trajectories.
If \code{relativeToInitial = TRUE} lengths are calculated between the initial survey and all the other surveys.
If \code{all = TRUE} lengths are calculated for all segments.

Function \code{trajectoryLengths2D} returns a data frame with the length of each segment on each trajectory and the total length of all trajectories.
If \code{relativeToInitial = TRUE} lengths are calculated between the initial survey and all the other surveys.
If \code{all = TRUE} lengths are calculated for all segments.

Function \code{trajectoryAngles} returns a data frame with angle values on each trajectory. If \code{stats=TRUE}, then the mean, standard deviation and mean resultant length of those angles are also returned.

Function \code{trajectoryAngles2D} returns a data frame with angle values on each trajectory. If \code{betweenSegments=TRUE}, then angles are calculated between trajectory segments, alternatively, If \code{betweenSegments=FALSE}, angles are calculated considering Y axis as the North (0°).

Function \code{trajectoryProjection} returns a data frame with the following columns:
\itemize{
\item{\code{distanceToTrajectory}: Distances to the trajectory, i.e. rejection (\code{NA} for target points whose projection is outside the trajectory).}
\item{\code{segment}: Segment that includes the projected point (\code{NA} for target points whose projection is outside the trajectory).}
\item{\code{relativePosition}: Relative position of the projected point within the trajectory, i.e. values from 0 to 1 with 0 representing the start of the trajectory and 1 representing the end (\code{NA} for target points whose projection is outside the trajectory).}
}

Function \code{trajectoryConvergence} returns a list with two elements:
\itemize{
\item{\code{tau}: A matrix with the statistic (Mann-Kendall's tau) of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the statistic of the test of the row trajectory approaching the column trajectory.}
\item{\code{p.value}: A matrix with the p-value of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the p-value indicates the test of the row trajectory approaching the column trajectory.}
}

Function \code{trajectoryDirectionality} returns a vector with directionality values (one per trajectory).
}
\description{
Ecological Trajectory Analysis (ETA) is a framework to analyze dynamics of ecosystems described as trajectories in a chosen space of multivariate resemblance (De \enc{Cáceres}{Caceres} et al. 2019).
ETA takes trajectories as objects to be analyzed and compared geometrically.
}
\details{
Given a distance matrix between ecosystem states, the set of functions that provide ETA metrics are:
\itemize{
\item{Functions \code{segmentDistances} and \code{trajectoryDistances} calculate the distance between pairs of directed segments and ecosystem trajectories, respectively.}
\item{Function \code{trajectoryLengths} calculates lengths of directed segments and total path lengths of trajectories.}
\item{Function \code{trajectoryLengths2D} calculates lengths of directed segments and total path lengths of trajectories from 2D coordinates given as input.}
\item{Function \code{trajectoryAngles} calculates the angle between consecutive pairs of directed segments or between segments of ordered triplets of points.}
\item{Function \code{trajectoryAngles2D} calculates the angle between consecutive pairs of directed segments or between segments of ordered triplets of points.}
\item{Function \code{trajectoryProjection} projects a set of target points onto a specified trajectory and returns the distance to the trajectory (i.e. rejection) and the relative position of the projection point within the trajectory.}
\item{Function \code{trajectoryConvergence} performs the Mann-Kendall trend test on the distances between trajectories (symmetric test) or the distance between points of one trajectory to the other.}
\item{Function \code{trajectoryDirectionality} returns (for each trajectory) a statistic that measures directionality of the whole trajectory.}
}

Details of calculations are given in De \enc{Cáceres}{Caceres} et al (2019).
The input distance matrix \code{d} should ideally be metric. That is, all subsets of distance triplets should fulfill the triangle inequality (see utility function \code{\link{is.metric}}).
All ETA functions that require metricity include a parameter '\code{add}', which by default is TRUE, meaning that whenever the triangle inequality is broken the minimum constant required to fulfill it is added to the three distances.
If such local (an hence, inconsistent across triplets) corrections are not desired, users should find another way modify \code{d} to achieve metricity, such as PCoA, metric MDS or non-metric MDS (see vignette 'Introduction to Ecological Trajectory Analysis').
If parameter '\code{add}' is set to FALSE and problems of triangle inequality exist, ETA functions may provide missing values in some cases where they should not.

The resemblance between trajectories is done by adapting concepts and procedures used for the analysis of trajectories in space (i.e. movement data) (Besse et al. 2016).

Function \code{trajectoryAngles} calculates angles between consecutive segments in degrees. For each pair of segments, the angle between the two is defined on the plane that contains the two segments, and measures the change in direction (in degrees) from one segment to the other.
Angles are always positive, with zero values indicating segments that are in a straight line, and values equal to 180 degrees for segments that are in opposite directions. If \code{all = TRUE}
angles are calculated between the segments corresponding to all ordered triplets. Alternatively, if \code{relativeToInitial = TRUE} angles are calculated for each segment with respect to the initial survey.

Function \code{trajectoryAngles2D} calculates angles between consecutive segments in degrees from 2D coordinates given as input. For each pair of segments, the angle between the two is defined on the plane that contains the two segments, and measures the change in direction (in degrees) from one segment to the other.
Angles are always positive (O to 360), with zero values indicating segments that are in a straight line, and values equal to 180 degrees for segments that are in opposite directions.
If \code{all = TRUE} angles are calculated between the segments corresponding to all ordered triplets. Alternatively, if \code{relativeToInitial = TRUE} angles are calculated for each segment with respect to the initial survey.
If \code{betweenSegments = TRUE} angles are calculated between segments of trajectory, otherwise, If \code{betweenSegments = FALSE}, angles are calculated considering Y axis as the North (0°).
}
\examples{
#Description of sites and surveys
sites = c(1,1,1,2,2,2)
surveys=c(1,2,3,1,2,3)
  
#Raw data table
xy<-matrix(0, nrow=6, ncol=2)
xy[2,2]<-1
xy[3,2]<-2
xy[4:6,1] <- 0.5
xy[4:6,2] <- xy[1:3,2]
xy[6,1]<-1
  
#Draw trajectories
trajectoryPlot(xy, sites, surveys, 
               traj.colors = c("black","red"), lwd = 2)

#Distance matrix
d = dist(xy)
d
  
trajectoryLengths(d, sites, surveys)
trajectoryLengths2D(xy, sites, surveys)
trajectoryAngles(d, sites, surveys)
trajectoryAngles2D(xy, sites, surveys, betweenSegments = TRUE)
trajectoryAngles2D(xy, sites, surveys, betweenSegments = FALSE)
segmentDistances(d, sites, surveys)$Dseg
trajectoryDistances(d, sites, surveys, distance.type = "Hausdorff")
trajectoryDistances(d, sites, surveys, distance.type = "DSPD")
  
  
#Should give the same results if surveys are not in order 
#(here we switch surveys for site 2)
temp = xy[5,]
xy[5,] = xy[6,]
xy[6,] = temp
surveys[5] = 3
surveys[6] = 2
  
trajectoryPlot(xy, sites, surveys, 
               traj.colors = c("black","red"), lwd = 2)   
trajectoryLengths(dist(xy), sites, surveys)
trajectoryLengths2D(xy, sites, surveys)
segmentDistances(dist(xy), sites, surveys)$Dseg
trajectoryAngles(dist(xy), sites, surveys)
trajectoryAngles2D(xy, sites, surveys, betweenSegments = TRUE)
trajectoryAngles2D(xy, sites, surveys, betweenSegments = FALSE)
trajectoryDistances(dist(xy), sites, surveys, distance.type = "Hausdorff")
trajectoryDistances(dist(xy), sites, surveys, distance.type = "DSPD")
 
}
\references{
Besse, P., Guillouet, B., Loubes, J.-M. & François, R. (2016). Review and perspective for distance based trajectory clustering. IEEE Trans. Intell. Transp. Syst., 17, 3306–3317.

De \enc{Cáceres}{Caceres} M, Coll L, Legendre P, Allen RB, Wiser SK, Fortin MJ, Condit R & Hubbell S. (2019).
Trajectory analysis in community ecology. Ecological Monographs 89, e01350.
}
\seealso{
\code{\link{trajectoryplots}}, \code{\link{trajectoryutils}}
}
\author{
Miquel De \enc{Cáceres}{Caceres}, CREAF

Anthony Sturbois, Vivarmor nature, Réserve Naturelle nationale de la Baie de Saint-Brieuc
}
