% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ebTobit.R
\name{ebTobit}
\alias{ebTobit}
\title{Empirical Bayes Matrix Estimation under a Tobit Likelihood}
\usage{
ebTobit(
  L,
  R = L,
  gr = (R + L)/2,
  s1 = 1,
  algorithm = "EM",
  pos_lik = TRUE,
  ...
)
}
\arguments{
\item{L}{n x p matrix of lower bounds on observations}

\item{R}{n x p matrix of upper bounds on observations}

\item{gr}{m x p matrix of grid points}

\item{s1}{a single numeric standard deviation or an n x p matrix of standard
deviations}

\item{algorithm}{method to fit prior, either a function or function name}

\item{pos_lik}{boolean indicating whether to lower-bound the likelihood
matrix with \code{.Machine$double.xmin} (default: TRUE); helps avoid possible
divide-by-zero errors in \code{algorithm}}

\item{...}{further arguments passed into fitting method such as \code{rtol}
and \code{maxiter}, see for example \code{\link{EM}}}
}
\value{
a fitted \code{ebTobit} object containing at least the prior weights,
corresponding grid/support points, and likelihood matrix relating the grid to
the observations
}
\description{
Fit and estimate the nonparametric maximum likelihood estimator in R^p
(p >= 1) when the likelihood is Gaussian and possibly interval censored. If
p = 1, then \code{L}, \code{R}, and \code{gr} may be vectors (they are
immediately converted into matrices internally).
}
\details{
Each observation is stored in a pair of matrices, \code{L} and \code{R}. If
L_ij = R_ij then a direct measurement X_ij ~ N(theta, s1^2) is made; 
if L_ij < R_ij then the measurement is censored so that L_ij < X_ij < R_ij.

To use a custom fitting algorithm, define a function \code{MyAlg} that
takes in an n x m likelihood matrix: P_ij = P(L_i, R_i | theta = t_j) and
returns a vector of estimated prior weights for t_j. Once \code{MyAlg} is
defined, fit the prior by using \code{algorithm = "MyAlg"} or use the
function itself \code{algorithm = MyAlg}.

Alternative fitting algorithms "ConvexPrimal"and "ConvexDual" have been
(wrappers of \code{REBayes::KWPrimal} and \code{REBayes::KWDual},
respectively) included and can be used if MOSEK and \code{REBayes} are
properly installed.
}
\examples{
set.seed(1)
n <- 100
p <- 5
r <- 2
U.true <- matrix(stats::rexp(n*r), n, r)
V.true <- matrix(sample(x = c(1,4,7), 
                        size = p*r, 
                        replace = TRUE, 
                        prob = c(0.7, 0.2, 0.1)), 
                 p, r)
TH <- tcrossprod(U.true, V.true)
X <- TH + matrix(stats::rnorm(n*p), n, p)

# fit uncensored method
fit1 <- ebTobit(X)

# fit left-censored method
ldl <- 1 # lower and upper detection limits
udl <- Inf
L <- ifelse(X < ldl, 0, ifelse(X <= udl, X, udl))
R <- ifelse(X < ldl, ldl, ifelse(X <= udl, X, Inf))
fit2 <- ebTobit(L, R)
}
