% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynrRecipe.R
\name{prep.formulaDynamics}
\alias{prep.formulaDynamics}
\title{Recipe function for specifying dynamic functions using formulas}
\usage{
prep.formulaDynamics(formula, startval = numeric(0),
  isContinuousTime = FALSE, jacobian)
}
\arguments{
\item{formula}{a list of formulas specifying the drift or state-transition 
equations for the latent variables in continuous or discrete time, respectively.}

\item{startval}{a named vector of starting values of the parameters in the 
formulas for estimation with parameter names as its name. If there are no free parameters in 
the dynamic functions, leave startval as the default \code{numeric(0)}.}

\item{isContinuousTime}{if True, the left hand side of the formulas represent 
the first-order derivatives of the specified variables; if False, the left hand 
side of the formulas represent the current state of the specified variable while 
the same variable on the righ hand side is its previous state.}

\item{jacobian}{(optional) a list of formulas specifying the analytic jacobian matrices 
containing the analytic differentiation function of the dynamic functions with respect to
the latent variables. If this is not provided, dynr will invoke an automatic differentiation
procedure to compute the jacobian functions.}
}
\description{
Recipe function for specifying dynamic functions using formulas
}
\details{
This function defines the dynamic functions of the model either in discrete time or in continuous time.
The function can be either linear or nonlinear, with free or fixed parameters, numerical constants, 
covariates, and other mathematical functions that define the dynamics of the latent variables.
Every latent variable in the model needs to be defined by a differential (for continuous time model), or
difference (for discrete time model) equation.  The names of the latent variables should match 
the specification in prep.measurement().
For nonlinear models, the estimation algorithm generally needs a Jacobian matrix that contains
elements of first differentiations of the dynamic functions with respect to the latent variables
in the model. For most nonlinear models, such differentiations can be handled automatically by
dynr. However, in some cases, such as when the absolute function (abs) is used, the automatic
differentiation would fail and the user may need to provide his/her own Jacobian functions.
}
\examples{
# In this example, we present how to define the dynamics of a bivariate dual change score model
# (McArdle, 2009). This is a linear model and the user does not need to worry about 
# providing any jacobian function (the default). 
 
# We start by creating a list of formula that describes the model. In this model, we have four 
# latent variables, which are "readLevel", "readSlope", "mathLevel", and "math Slope".  The right-
# hand side of each formula gives a function that defines the dynamics.   
 
 formula =list(
          list(readLevel~ (1+beta.read)*readLevel + readSlope + gamma.read*mathLevel,
          readSlope~ readSlope,
          mathLevel~ (1+beta.math)*mathLevel + mathSlope + gamma.math*readLevel, 
          mathSlope~ mathSlope
          ))

# Then we use prep.formulaDynamics() to define the formula, starting value of the parameters in
# the model, and state the model is in discrete time by setting isContinuousTime=FALSE.
 
dynm  <- prep.formulaDynamics(formula=formula,
                             startval=c(beta.read = -.5, beta.math = -.5, 
                                        gamma.read = .3, gamma.math = .03
                             ), isContinuousTime=FALSE)


# For a full demo example of regime switching nonlinear discrete time model, you
# may refer to a tutorial on 
# \\url{https://quantdev.ssri.psu.edu/tutorials/dynr-rsnonlineardiscreteexample}

#Not run: 
#For a full demo example that uses user-supplied analytic jacobian functions see:
#demo(RSNonlinearDiscrete, package="dynr")
formula <- list(
    list(
      x1 ~ a1*x1,
      x2 ~ a2*x2),
    list(
      x1 ~ a1*x1 + c12*(exp(abs(x2)))/(1+exp(abs(x2)))*x2,
      x2 ~ a2*x2 + c21*(exp(abs(x1)))/(1+exp(abs(x1)))*x1)
  )
jacob <- list(
  list(x1~x1~a1,
      x2~x2~a2),
  list(x1~x1~a1,
      x1~x2~c12*(exp(abs(x2))/(exp(abs(x2))+1)+x2*sign(x2)*exp(abs(x2))/(1+exp(abs(x2))^2)),
      x2~x2~a2,
      x2~x1~c21*(exp(abs(x1))/(exp(abs(x1))+1)+x1*sign(x1)*exp(abs(x1))/(1+exp(abs(x1))^2))))
dynm <- prep.formulaDynamics(formula=formula, startval=c( a1=.3, a2=.4, c12=-.5, c21=-.5),
                             isContinuousTime=FALSE, jacobian=jacob)

#For a full demo example that uses automatic jacobian functions (the default) see:
#demo(RSNonlinearODE , package="dynr")
formula=list(prey ~ a*prey - b*prey*predator, predator ~ -c*predator + d*prey*predator)
dynm <- prep.formulaDynamics(formula=formula,
                          startval=c(a = 2.1, c = 0.8, b = 1.9, d = 1.1),
                          isContinuousTime=TRUE)
}

