#' @title Function \code{plot_graph}
#' @description Plot the dependency structure of your workflow
#' @export
#' @param plan workflow plan data frame, same as for function 
#' \code{\link{make}()}.
#' @param targets names of targets to bulid, same as for function
#' \code{\link{make}()}.
#' @param envir environment to import from, same as for function
#' \code{\link{make}()}.
#' @param verbose logical, whether to output messages to the console.
plot_graph = function(plan, targets = drake::possible_targets(plan), 
  envir = parent.frame(), verbose = TRUE){
  force(envir)
  build_graph(plan = plan, targets = targets, envir = envir,
    verbose = verbose) %>% plot.igraph
}

#' @title Function \code{build_graph}
#' @description Make a graph of the dependency structure of your workflow.
#' @details This function returns an igraph object representing how
#' the targets in your workflow depend on each other. 
#' (\code{help(package = "igraph")}). To plot the graph, call
#' to \code{\link{plot.igraph}()} on your graph, or just use 
#' \code{\link{plot_graph}()} from the start.
#' @export
#' @param plan workflow plan data frame, same as for function
#' \code{\link{make}()}.
#' @param targets names of targets to bulid, same as for function
#' \code{\link{make}()}.
#' @param envir environment to import from, same as for function
#' \code{\link{make}()}.
#' @param verbose logical, whether to output messages to the console.
build_graph = function(plan, targets = drake::possible_targets(plan), 
  envir = parent.frame(), verbose = TRUE){
  force(envir)
  plan = sanitize_plan(plan)
  targets = sanitize_targets(plan, targets)
  imports = as.list(envir)
  assert_unique_names(imports = names(imports), targets = plan$target,
    envir = envir, verbose = verbose)
  import_deps = lapply(imports, import_dependencies)
  command_deps = lapply(plan$command, command_dependencies)
  names(command_deps) = plan$target
  dependency_list = c(command_deps, import_deps)
  keys = names(dependency_list)
  vertices = c(keys, unlist(dependency_list)) %>% unique
  graph = make_empty_graph() + vertices(vertices)
  for(key in keys)
    for(dependency in dependency_list[[key]])
      graph = graph + edge(dependency, key)
  ignore = lapply(targets, function(vertex)
    subcomponent(graph = graph, v = vertex, mode = "in")$name
  ) %>% unlist %>% unique %>% setdiff(x = vertices)
  graph = delete_vertices(graph, v = ignore)
  if(!is_dag(graph)) 
    stop("Workflow is circular (chicken and egg dilemma).")
  graph
}

#' @title Function \code{tracked}
#' @description Print out which objects, functions, files, targets, etc.
#' are reproducibly tracked.
#' @export
#' @param plan workflow plan data frame, same as for function
#' \code{\link{make}()}.
#' @param targets names of targets to bulid, same as for function
#' \code{\link{make}()}.
#' @param envir environment to import from, same as for function
#' \code{\link{make}()}.
tracked = function(plan, targets = drake::possible_targets(plan),
  envir = parent.frame()){
  force(envir)
  graph = build_graph(plan = plan, targets = targets, envir = envir)
  V(graph)$name
}

assert_unique_names = function(imports, targets, envir, verbose){
  if(anyDuplicated(targets)){
    duplicated = which(table(targets) > 1) %>% names
    stop("Duplicate targets in workflow plan:\n", 
      multiline_message(duplicated))
  }
  common = intersect(imports, targets)
  if(verbose & length(common))
    cat("Unloading targets from environment:\n",
     multiline_message(common), "\n", sep = "")
  remove(list = common, envir = envir)
}
