% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expectreg.ipc.R
\name{expectreg.ipc}
\alias{expectreg.ipc}
\title{Expectile regression for right-censored data}
\usage{
expectreg.ipc(
  formula,
  data = NULL,
  smooth = c("schall", "ocv", "aic", "bic", "cvgrid", "lcurve", "fixed"),
  lambda = 1,
  expectiles = NA,
  LAWSmaxCores = 1,
  IPC_weights = c("IPCRR", "IPCKM"),
  KMweights = NULL,
  ci = FALSE,
  hat1 = FALSE
)
}
\arguments{
\item{formula}{A formula object, with the response on the left of the `~'
operator, and the terms on the right.  The response must be a
\code{\link[survival]{Surv}} object as returned by the \code{\link[survival]{Surv}} 
function. Only right censored data are allowed. Splines can be specified through the function \code{\link[expectreg]{rb}}.}

\item{data}{Optional data frame containing the variables used in the model, if the data is not explicitly given in the formula.}

\item{smooth}{The smoothing method that shall be used.
There are different smoothing algorithms that should prevent overfitting. The '\code{schall}' algorithm balances 
variance of errors and contrasts. Ordinary cross- validation '\code{ocv}' minimizes a score-function using \code{\link[stats]{nlminb}} or with a grid search by '\code{cvgrid}'
or the function uses a fixed penalty. The numerical minimizatioin is also possible with AIC or BIC as score. 
The L-curve is an experimental grid search by Frasso and Eilers.}

\item{lambda}{The fixed penalty can be adjusted. Also serves as starting value for the smoothing algorithms.}

\item{expectiles}{In default setting, the expectiles (0.01,0.02,0.05,0.1,0.2,0.5,0.8,0.9,0.95,0.98,0.99) are 
calculated. You may specify your own set of expectiles in a vector.}

\item{LAWSmaxCores}{How many cores should maximally be used by parallelization. Currently only implemented for Unix-like OS.}

\item{IPC_weights}{Denotes the kind of IPC weights to use. \code{IPCRR} weights differ from \code{IPCKM} weights by modifying the weights for the last observation if it is censored.}

\item{KMweights}{Custom IPC weights can be supplied here. This argument is used by \code{\link{modreg}}.}

\item{ci}{If TRUE, calculates the covariance matrix}

\item{hat1}{If TRUE, the hat matrix for the last asymetry level is calculated. This argument is mainly used by \code{\link{modreg}}.}
}
\value{
A list with the following elements.
\item{lambda}{The final smoothing parameters for all expectiles and for all effects in a list.}
\item{intercepts}{The intercept for each expectile.}
\item{coefficients}{A matrix of all the coefficients, for each base element a row and for each expectile a column.}
\item{values}{The fitted values for each observation and all expectiles, separately in a list for each effect in the model, sorted in order of ascending covariate values.}
\item{response}{Vector of the response variable.}
\item{covariates}{List with the values of the covariates.}
\item{formula}{The formula object that was given to the function.}
\item{asymmetries}{Vector of fitted expectile asymmetries as given by argument \code{expectiles}.}
\item{effects}{List of characters giving the types of covariates.}
\item{helper}{List of additional parameters like neighbourhood structure for spatial effects or \eqn{\phi} for kriging.}
\item{design}{Complete design matrix.}
\item{bases}{Bases components of each covariate.}
\item{fitted}{Fitted values.}
\item{covmat}{Covariance matrix.}
\item{diag.hatma}{Diagonal of the hat matrix. Used for model selection criteria.}
\item{data}{Original data.}
\item{smooth_orig}{Unchanged original type of smoothing.}
\item{KMweights}{Vector with IPC weights used in fitting.}
\item{aic}{Area under the AIC, approximated with a Riemannian sum.}
\item{hat}{The hat matrix for the last asymmetry level. This is used by \code{\link{modreg}}.}
}
\description{
This function extends expectile regression with inverse probability of censoring (IPC) weights to right-censored data.
}
\details{
Fits least asymmetrically weighted squares (LAWS) for each expectile. This function is intended
for right-censored data. For uncensored data, \code{\link[expectreg]{expectreg.ls}} should be used instead.
This function modifies \code{\link[expectreg]{expectreg.ls}} by adding IPC weights. See Seipp et al. (2021) for details on
the IPC weights. P-splines can be used with \code{\link[expectreg]{rb}}. The Schall algorithm is used for choosing the penalty.
}
\examples{

data(colcancer)

# linear effect
expreg <- expectreg.ipc(Surv(logfollowup, death) ~ sex + age, data = colcancer, 
                        expectiles = c(0.05, 0.2, 0.5, 0.8, 0.95))
coef(expreg)

\donttest{
# with p-splines, smoothing parameter selection with schall algorithm
expreg2 <- expectreg.ipc(Surv(logfollowup, death) ~ sex + rb(age), data = colcancer)
# smoothing parameter selection with AIC
expreg3 <- expectreg.ipc(Surv(logfollowup, death) ~ sex + rb(age), data = colcancer, smooth = "aic")
# manually selected smoothing parameter
expreg4 <- expectreg.ipc(Surv(logfollowup, death) ~ sex + rb(age), data = colcancer, 
                         smooth = "fixed", lambda = 2)

plot(expreg2)
plot(expreg3)
plot(expreg4)
}



}
\references{
Seipp, A, Uslar, V, Weyhe, D, Timmer, A, Otto-Sobotka, F. Weighted expectile regression for right-censored data. Statistics in Medicine. 2021; 40(25): 5501- 5520. https://doi.org/10.1002/sim.9137
}
