% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/space_filling.R
\name{grid_space_filling}
\alias{grid_space_filling}
\alias{grid_space_filling.parameters}
\alias{grid_space_filling.list}
\alias{grid_space_filling.param}
\title{Space-filling parameter grids}
\usage{
grid_space_filling(x, ..., size = 5, type = "any", original = TRUE)

\method{grid_space_filling}{parameters}(
  x,
  ...,
  size = 5,
  type = "any",
  variogram_range = 0.5,
  iter = 1000,
  original = TRUE
)

\method{grid_space_filling}{list}(
  x,
  ...,
  size = 5,
  type = "any",
  variogram_range = 0.5,
  iter = 1000,
  original = TRUE
)

\method{grid_space_filling}{param}(
  x,
  ...,
  size = 5,
  variogram_range = 0.5,
  iter = 1000,
  type = "any",
  original = TRUE
)
}
\arguments{
\item{x}{A \code{param} object, list, or \code{parameters}.}

\item{...}{One or more \code{param} objects (such as \code{\link[=mtry]{mtry()}} or
\code{\link[=penalty]{penalty()}}). None of the objects can have \code{unknown()} values in
the parameter ranges or values.}

\item{size}{A single integer for the maximum number of parameter value
combinations returned. If duplicate combinations are
generated from this size, the smaller, unique set is returned.}

\item{type}{A character string with possible values: \code{"any"},
\code{"audze_eglais"}, \code{"max_min_l1"}, \code{"max_min_l2"}, \code{"uniform"},
\code{"max_entropy"}, or \code{"latin_hypercube"}. A value of \code{"any"} will choose the
first design available (in the order listed above, excluding
\code{"latin_hypercube"}). For a single-point design, a random grid is created.}

\item{original}{A logical: should the parameters be in the original units or
in the transformed space (if any)?}

\item{variogram_range}{A numeric value greater than zero. Larger values
reduce the likelihood of empty regions in the parameter space. Only used
for \code{type = "max_entropy"}.}

\item{iter}{An integer for the maximum number of iterations used to find
a good design. Only used for \code{type = "max_entropy"}.}
}
\description{
Experimental designs for computer experiments are used to construct parameter
grids that try to cover the parameter space such that any portion of the
space has does not have an observed combination that is unnecessarily close
to any other point.
}
\details{
The types of designs supported here are latin hypercube designs of
different types. The simple designs produced by
\code{\link[=grid_latin_hypercube]{grid_latin_hypercube()}} are space-filling but
don’t guarantee or optimize any other properties.
\code{\link[=grid_space_filling]{grid_space_filling()}} might be able to produce
designs that discourage grid points from being close to one another.
There are a lot of methods for doing this, such as maximizing the
minimum distance between points (see Husslage \emph{et al} 2001).
\code{\link[=grid_max_entropy]{grid_max_entropy()}} attempts to maximize the
determinant of the spatial correlation matrix between coordinates.

Latin hypercube and maximum entropy designs use random numbers to make
the designs.

By default, \code{\link[=grid_space_filling]{grid_space_filling()}} will try to
use a pre-optimized space-filling design from
\url{https://www.spacefillingdesigns.nl/}
(see Husslage \emph{et al}, 2011) or using a uniform design. If no pre-made
design is available, then a maximum entropy design is created.

Also note that there may a difference in grids depending on how the
function is called. If the call uses the parameter objects directly the
possible ranges come from the objects in \code{dials}. For example:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mixture()
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## Proportion of Lasso Penalty (quantitative)
## Range: [0, 1]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(283)
mix_grid_1 <- grid_latin_hypercube(mixture(), size = 1000)
range(mix_grid_1$mixture)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] 0.0001530482 0.9999530388
}\if{html}{\out{</div>}}

However, in some cases, the \code{parsnip} and \code{recipe} packages overrides
the default ranges for specific models and preprocessing steps. If the
grid function uses a \code{parameters} object created from a model or recipe,
the ranges may have different defaults (specific to those models). Using
the example above, the \code{mixture} argument above is different for
\code{glmnet} models:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(parsnip)
library(tune)

# When used with glmnet, the range is [0.05, 1.00]
glmn_mod <-
  linear_reg(mixture = tune()) |>
  set_engine("glmnet")

set.seed(283)
mix_grid_2 <-
  glmn_mod |> 
  extract_parameter_set_dials() |> 
  grid_latin_hypercube(size = 1000)
range(mix_grid_2$mixture)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] 0.0501454 0.9999554
}\if{html}{\out{</div>}}
}
\examples{
grid_space_filling(
  hidden_units(),
  penalty(),
  epochs(),
  activation(),
  learn_rate(c(0, 1), trans = scales::transform_log()),
  size = 10,
  original = FALSE
)
# ------------------------------------------------------------------------------
# comparing methods

if (rlang::is_installed("ggplot2")) {

  library(dplyr)
  library(ggplot2)

  set.seed(383)
  parameters(trees(), mixture()) |>
    grid_space_filling(size = 25, type = "latin_hypercube") |>
    ggplot(aes(trees, mixture)) +
    geom_point() +
    lims(y = 0:1, x = c(1, 2000)) +
    ggtitle("latin hypercube")

  set.seed(383)
  parameters(trees(), mixture()) |>
    grid_space_filling(size = 25, type = "max_entropy") |>
    ggplot(aes(trees, mixture)) +
    geom_point() +
    lims(y = 0:1, x = c(1, 2000)) +
    ggtitle("maximum entropy")

  parameters(trees(), mixture()) |>
    grid_space_filling(size = 25, type = "audze_eglais") |>
    ggplot(aes(trees, mixture)) +
    geom_point() +
    lims(y = 0:1, x = c(1, 2000)) +
    ggtitle("Audze-Eglais")

  parameters(trees(), mixture()) |>
    grid_space_filling(size = 25, type = "uniform") |>
    ggplot(aes(trees, mixture)) +
    geom_point() +
    lims(y = 0:1, x = c(1, 2000)) +
    ggtitle("uniform")
}

}
\references{
Sacks, Jerome & Welch, William & J. Mitchell, Toby, and Wynn, Henry.
(1989). Design and analysis of computer experiments. With comments and a
rejoinder by the authors. Statistical Science. 4. 10.1214/ss/1177012413.

Santner, Thomas, Williams, Brian, and Notz, William. (2003). The Design and
Analysis of Computer Experiments. Springer.

Dupuy, D., Helbert, C., and Franco, J. (2015). DiceDesign and DiceEval: Two R
packages for design and analysis of computer experiments. Journal of
Statistical Software, 65(11)

Husslage, B. G., Rennen, G., Van Dam, E. R., & Den Hertog, D. (2011).
Space-filling Latin hypercube designs for computer experiments. \emph{Optimization
and Engineering}, 12, 611-630.

Fang, K. T., Lin, D. K., Winker, P., & Zhang, Y. (2000). Uniform design:
Theory and application. _Technometric_s, 42(3), 237-248
}
