#' ctStanFit
#'
#' Fits a ctsem model specified via \code{\link{ctModel}} with type either 'stanct' or 'standt', using Bayseian inference software
#' Stan. 
#' 
#' @param datalong long format data containing columns for subject id (numeric values, 1 to max subjects), manifest variables, 
#' any time dependent (i.e. varying within subject) predictors, 
#' and any time independent (not varying within subject) predictors.
#' @param ctstanmodel model object as generated by \code{\link{ctModel}} with type='stanct' or 'standt', for continuous or discrete time
#' models respectively.
#' @param stanmodeltext already specified Stan model character string, generally leave NA unless modifying Stan model directly.
#' (Possible after modification of output from fit=FALSE)
#' @param kalman logical indicating whether or not to integrate over latent states using a Kalman filter. 
#' Generally recommended to set TRUE unless using non-gaussian measurement model. If not using Kalman filter, experience
#' suggests that some models / datasets require a relatively high amount of very fast iterations before the sampler is
#' in the high density region. This can make it difficult to determine the number of iterations needed a priori - in such cases 
#' setting initwithoptim=TRUE may be helpful.
#' @param binomial logical indicating the use of binomial rather than Gaussian data, as with IRT analyses.
#' @param esthyper Logical indicating whether to explictly estimate distributions for any individually varying
#' parameters, or to fix the distributions to maximum likelihood estimates conditional on subject parameters.
#' @param fit If TRUE, fit specified model using Stan, if FALSE, return stan model object without fitting.
#' @param plot if TRUE, a Shiny program is launched upon fitting to interactively plot samples. 
#' May struggle with many (e.g., > 5000) parameters, and may leave sample files in working directory if sampling is terminated.
#' @param diffusionindices vector of integers denoting which latent variables are involved in covariance calculations.
#' latents involved only in deterministic trends or input effects can be removed from matrices, speeding up calculations. 
#' If unsure, leave default of 'all' ! Ignored if kalman=FALSE.
#' @param asymdiffusion if TRUE, increases fitting speed at cost of model flexibility - T0VAR in the model specification is ignored, the DIFFUSION matrix specification is used as the 
#' asymptotic DIFFUSION matrix (Q*_inf in the vignette / paper) (making it difficult if not impossible to properly specify
#' higher order processes). The speed increases come about because the internal Kalman filter routine has many steps removed, and the
#' asymptotic diffusion parameters are less dependent on the DRIFT matrix.
#' @param optimize if TRUE, use Stan's optimizer for maximum a posteriori estimates. 
#' Setting this also sets \code{esthyper=FALSE}
#' @param nopriors logical. If TRUE, any priors are disabled, can be desirable for optimization.
#' @param vb if TRUE, use Stan's variational approximation. Rudimentary testing suggests it is not accurate 
#' for many ctsem models at present.
#' @param iter number of iterations, half of which will be devoted to warmup by default when sampling.
#' When optimizing, this is the maximum number of iterations to allow -- convergence hopefully occurs before this!
#' @param inits vector of parameter start values, as returned by the rstan function \code{\link{unconstrain_pars}} for instance. 
#' @param initwithoptim Logical. If TRUE, the model, with population standard deviations fixed to 1 
#' (so approx 65% of the population mean prior), is first fit with penalised maximum likelihood to determine starting values
#' for the chains. This can help speed convergence and avoid problematic regions for certain problems.
#' @param chains number of chains to sample.
#' @param cores number of cpu cores to use. Either 'maxneeded' to use as many as available,
#' up to the number of chains, or an integer.
#' @param control List of arguments sent to \code{\link[rstan]{stan}} control argument, 
#' regarding warmup / sampling behaviour.
#' @param verbose Logical. If TRUE, prints log probability at each iteration.
#' @param stationary Logical. If TRUE, T0VAR and T0MEANS input matrices are ignored, 
#' the parameters are instead fixed to long run expectations. More control over this can be achieved
#' by instead setting parameter names of T0MEANS and T0VAR matrices in the input model to 'stationary', for
#' elements that should be fixed to stationarity.
#' @param ... additional arguments to pass to \code{\link[rstan]{stan}} function.
#' @examples
#' \dontrun{
#' #test data with 2 manifest indicators measuring 1 latent process each, 
#' # 1 time dependent predictor, 3 time independent predictors
#' head(ctstantestdat) 
#' 
#' #generate a ctStanModel
#' model<-ctModel(type='stanct',
#' n.latent=2, latentNames=c('eta1','eta2'),
#' n.manifest=2, manifestNames=c('Y1','Y2'),
#' n.TDpred=1, TDpredNames='TD1', 
#' n.TIpred=3, TIpredNames=c('TI1','TI2','TI3'),
#' LAMBDA=diag(2))
#' 
#' #set all parameters except manifest means to be fixed across subjects
#' model$pars$indvarying[-c(19,20)] <- FALSE
#' 
#' #fit model to data (takes a few minutes - but insufficient 
#' # iterations and max_treedepth for inference!)
#' fit<-ctStanFit(ctstantestdat, model, iter=200, chains=2, 
#' control=list(max_treedepth=6))
#' 
#' #output functions
#' summary(fit) 
#' 
#' plot(fit)
#' 
#' }
#' @importFrom Rcpp evalCpp
#' @export
ctStanFit<-function(datalong, ctstanmodel, stanmodeltext=NA, iter=1000, kalman=TRUE, binomial=FALSE,
  esthyper=TRUE, fit=TRUE, stationary=FALSE,plot=FALSE,  diffusionindices='all',
  asymdiffusion=FALSE,optimize=FALSE, nopriors=FALSE, vb=FALSE, chains=1,cores='maxneeded', inits=NULL,initwithoptim=FALSE,
  control=list(adapt_delta=.8, adapt_init_buffer=2, adapt_window=2,
    max_treedepth=8,stepsize=.001),verbose=FALSE,...){
  
  if(class(ctstanmodel) != 'ctStanModel') stop('not a ctStanModel object')
  
  args=match.call()
  
  noncentered=FALSE
  nonexplosive=FALSE
  fixedkalman=FALSE
  if(optimize) {
    if(esthyper==TRUE) message('Setting esthyper=FALSE for optimization')
    esthyper=FALSE
  }
  
  checkvarying<-function(matrixnames,yesoutput,nooutput=''){#checks if a matrix is set to individually vary in ctspec
    check<-0
    out<-nooutput
    if('T0VAR' %in% matrixnames & nt0varstationary > 0) matrixnames <- c(matrixnames,'DRIFT','DIFFUSION')
    if('T0MEANS' %in% matrixnames & nrow(t0meansstationary) > 0) matrixnames <- c(matrixnames,'DRIFT','CINT')
    for( matname in matrixnames){
      if(any(c(ctspec$indvarying,ctspecduplicates$indvarying)[c(ctspec$matrix,ctspecduplicates$matrix) %in% matrixnames])) check<-c(check,1)
    }
    if(sum(check)==length(matrixnames))  out<-yesoutput
    return(out)
  }
  
  
  #read in ctmodel values
  ctspec<-ctstanmodel$pars
  
  if(binomial) {
    ctspec<-ctspec[ctspec$matrix != 'MANIFESTVAR',]
    message(paste0('MANIFESTVAR matrix is ignored when binomial=TRUE'))
  }
  
  #clean ctspec structure
  found=FALSE
  ctspec$indvarying=as.logical(ctspec$indvarying)
  ctspec$value=as.numeric(ctspec$value)
  ctspec$transform=as.character(ctspec$transform)
  ctspec$param=as.character(ctspec$param)
  comparison=c(NA,NA,'FALSE')
  replacement=c(NA,NA,FALSE)
  # names(comparison)=c('param','transform','indvarying')
  for(rowi in 1:nrow(ctspec)){
    if( !is.na(ctspec$value[rowi])) {
      if(!identical(as.character(unlist(ctspec[rowi,c('param','transform','indvarying')])),comparison)) {
        found<-TRUE
        ctspec[rowi,c('param','transform','indvarying')]=replacement
      }
    }
  }
  if(found) message('Minor inconsistencies in model found - removing param name, transform and indvarying from any parameters with a value specified')
  
  #remove T0VAR and T0MEANS if stationary argument
  if(stationary) {
    ctspec=ctspec[ctspec$matrix %in% c('T0VAR','T0MEANS')==FALSE,,drop=FALSE]
    message('removing T0VAR and T0MEANS from parameter matrices because stationary=TRUE')
  }
  
  if(1==99){ #not sure this is a great approach
  #set t0var to stationary where t0means is indvarying
      t0varspec <- ctspec[ctspec$matrix %in% 'T0VAR',]
      t0meansindvarying <- ctspec$indvarying[ctspec$matrix %in% 'T0MEANS']
      t0varstationarymessage <- FALSE
      for(rowi in unique(t0varspec$row)){
        for(coli in unique(t0varspec$row)){
          if(rowi %in% which(t0meansindvarying)){ #if this row is indvarying t0means
          if(coli %in% which(t0meansindvarying)){ #and this column is indvarying t0means
            if(is.na(t0varspec$value[t0varspec$row == rowi & t0varspec$col ==coli])){ #and the t0var parameter is free
              if(!(t0varspec$param[t0varspec$row == rowi & t0varspec$col ==coli] %in% 'stationary')){ #and the t0var parameter is not stationary

            t0varspec$param[t0varspec$row==rowi & t0varspec$col==coli] <-'stationary'
            t0varstationarymessage <- TRUE
        }}}}}}
      if(t0varstationarymessage) {
        message('T0VAR parameters fixed to stationary where T0MEANS are individually varying')
        ctspec[ctspec$matrix %in% 'T0VAR',] <- t0varspec
      }
}
  
  
  ctstanmodel$pars <- ctspec #updating because we save the model later
  

  
  #collect individual stationary elements and update ctspec
  t0varstationary <- as.matrix(rbind(ctspec[ctspec$param %in% 'stationary' & ctspec$matrix %in% 'T0VAR',c('row','col')]))
  if(nrow(t0varstationary) > 0){ #ensure upper tri is consistent with lower
    for(i in 1:nrow(t0varstationary)){
      if(t0varstationary[i,1] != t0varstationary[i,2]) t0varstationary <- rbind(t0varstationary,t0varstationary[i,c(2,1)])
    }}
  
  
  t0meansstationary <- as.matrix(rbind(ctspec[ctspec$param[ctspec$matrix %in% 'T0MEANS'] %in% 'stationary',c('row','col')]))
  ctspec$value[ctspec$param %in% 'stationary'] <- 0
  ctspec$indvarying[ctspec$param %in% 'stationary'] <- FALSE
  ctspec$transform[ctspec$param %in% 'stationary'] <- NA
  ctspec$param[ctspec$param %in% 'stationary'] <- NA
  
  nt0varstationary <- nrow(t0varstationary)
  # if(nt0varstationary ==0) t0varstationary <- matrix(-99,ncol=2)
  nt0meansstationary <- nrow(t0meansstationary)
  # if(nt0meansstationary ==0) t0meansstationary <- matrix(-99,ncol=2)

  
  #split ctspec into unique and non-unique components
  ctspecduplicates <- ctspec[duplicated(ctspec$param)&!is.na(ctspec$param),]
  hypermeanduplicates<-c()
  
  if(any(duplicated(ctspec$param)&!is.na(ctspec$param))){
    for(i in 1:nrow(ctspecduplicates)){
      hypermeanduplicates[i] = paste0('hypermeans[',which(ctspecduplicates$param[i] %in% ctspec$param),']')
    }
  }
  
  ctspec <- ctspec[!duplicated(ctspec$param) | is.na(ctspec$param),]
  ctspecduplicates=cbind(ctspecduplicates,hypermeanduplicates)
  
  driftdiagonly <- ifelse(all(!is.na(ctspec$value[ctspec$matrix == 'DRIFT' & ctspec$row != ctspec$col]) &
     all(ctspec$value[ctspec$matrix == 'DRIFT' & ctspec$row != ctspec$col] == 0) ), 1, 0)
  
  n.latent<-ctstanmodel$n.latent
  n.manifest<-ctstanmodel$n.manifest
  n.TDpred<-ctstanmodel$n.TDpred
  n.TIpred<-ctstanmodel$n.TIpred
  
  manifestNames<-ctstanmodel$manifestNames
  latentNames<-ctstanmodel$latentNames
  TDpredNames<-ctstanmodel$TDpredNames
  TIpredNames<-ctstanmodel$TIpredNames
  id<-ctstanmodel$subjectIDname
  timeName<-ctstanmodel$timeName
  continuoustime<-ctstanmodel$continuoustime
  indvarying<-c(ctspec$indvarying,ctspecduplicates$indvarying)
  nindvarying<-sum(ctspec$indvarying)
  nparams<-sum(is.na(ctspec$value))
  
  
  #data checks
  if (!(id %in% colnames(datalong))) stop(paste('id column', omxQuotes(id), "not found in data"))
  if(any(is.na(as.numeric(datalong[,id])))) stop('id column may not contain NA\'s or character strings!')
  
  #fit spec checks
  if(binomial & any(kalman)) stop('Binomial observations only possible with kalman=FALSE')
  
  
  
  
  
  T0check<-1
  for(i in 2:nrow(datalong)){
    T0check<-c(T0check, ifelse(datalong[,'id'][i] != datalong[,'id'][i-1], 1, 0))
  }
  
  if (!(timeName %in% colnames(datalong))) stop(paste('time column', omxQuotes(timeName), "not found in data"))
  if(any(is.na(datalong[,timeName]))) stop('Missing "time" column!')
  
  #check id and calculate intervals, discrete matrix indices
  driftindex<-rep(0,nrow(datalong))
  diffusionindex<-driftindex
  cintindex<-driftindex
  oldsubi<-0
  dT<-rep(-1,length(datalong[,timeName]))
  # intervalChange<-dT
  for(rowi in 1:length(datalong[,timeName])) {
    subi<-datalong[rowi,id]
    if(rowi==1 && subi!=1) stop('subject id column must ascend from 1 to total subjects without gaps')
    if(oldsubi!=subi && subi-oldsubi!=1) stop('subject id column must ascend from 1 to total subjects without gaps')
    if(subi - oldsubi == 1) {
      dT[rowi]<-0
      subistartrow<-rowi
    }
    if(subi - oldsubi == 0) {
      if(continuoustime) dT[rowi]<-round(datalong[rowi,timeName] - datalong[rowi-1,timeName],8)
      if(!continuoustime) dT[rowi]<-1
      if(dT[rowi] <=0) stop(paste0('A time interval of ', dT[rowi],' was found at row ',rowi))
      # if(subi!=oldsubi) intervalChange[rowi] <-  0
      # if(subi==oldsubi && dT[rowi] != dT[rowi-1]) intervalChange[rowi] <- 1
      # if(subi==oldsubi && dT[rowi] == dT[rowi-1]) intervalChange[rowi] <- 0
      if(dT[rowi] %in% dT[1:(rowi-1)]) dTinwhole<-TRUE else dTinwhole<-FALSE
      if(dT[rowi] %in% dT[subistartrow:(rowi-1)]) dTinsub<-TRUE else dTinsub<-FALSE
      
      if(checkvarying('DRIFT',1,0)==0 & dTinwhole==FALSE) driftindex[rowi] <- max(driftindex)+1
      if(checkvarying('DRIFT',1,0)==1 & dTinsub==FALSE) driftindex[rowi] <- max(driftindex)+1
      if(checkvarying('DRIFT',1,0)==0 & dTinwhole==TRUE) driftindex[rowi] <- driftindex[match(dT[rowi],dT)]
      if(checkvarying('DRIFT',1,0)==1 & dTinsub==TRUE) driftindex[rowi] <- driftindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
      
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==0 & dTinwhole==FALSE) diffusionindex[rowi] <- max(diffusionindex)+1
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==1 & dTinsub==FALSE) diffusionindex[rowi] <- max(diffusionindex)+1
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==0 & dTinwhole==TRUE) diffusionindex[rowi] <- diffusionindex[match(dT[rowi],dT)]
      if(checkvarying(c('DIFFUSION','DRIFT'),1,0)==1 & dTinsub==TRUE) diffusionindex[rowi] <- diffusionindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
      
      if(checkvarying(c('CINT','DRIFT'),1,0)==0 & dTinwhole==FALSE) cintindex[rowi] <- max(cintindex)+1
      if(checkvarying(c('CINT','DRIFT'),1,0)==1 & dTinsub==FALSE) cintindex[rowi] <- max(cintindex)+1
      if(checkvarying(c('CINT','DRIFT'),1,0)==0 & dTinwhole==TRUE) cintindex[rowi] <- cintindex[match(dT[rowi],dT)]
      if(checkvarying(c('CINT','DRIFT'),1,0)==1 & dTinsub==TRUE) cintindex[rowi] <- cintindex[subistartrow:rowi][match(dT[rowi],dT[subistartrow:rowi])]
    }
    oldsubi<-subi
  }
  
  message('Unique discreteDRIFT calculations per step required = ', length(unique(driftindex))-1)
  message('Unique discreteCINT calculations per step required = ', length(unique(cintindex))-1)
  message('Unique discreteDIFFUSION calculations per step required = ', length(unique(diffusionindex))-1)
  
  # datalong[sort(c(which(dT > 5),which(dT > 5)+1,which(dT > 5)-1)),1:2]
  
  if(n.TDpred > 0) {
    tdpreds <- datalong[,TDpredNames,drop=FALSE]
    if(any(is.na(tdpreds))) message('Missingness in TDpreds! Replaced by zeroes...')
    tdpreds[is.na(tdpreds)] <-0 ## rough fix for missingness
  }
  if(n.TIpred > 0) {
    tipreds <- datalong[match(unique(datalong[,id]),datalong[,id]),TIpredNames,drop=FALSE]
    if(any(is.na(tipreds))) {
      message(paste0('Missingness in TIpreds - imputing ', sum(is.na(tipreds)),' values'))
      tipreds[is.na(tipreds)] = 99999
    }
  }
  
  datalong[is.na(datalong)]<-99999 #missing data
  
  nsubjects <- length(unique(datalong[, 'id'])) 
  
  indvaryingindex = array( #used to select hypermeans appropriately when generating subject params
    c(which(ctspec$indvarying[is.na(ctspec$value)]))) #unique hypermeans
      # match(ctspecduplicates$param[which(ctspecduplicates$indvarying[is.na(ctspecduplicates$value)])], ctspec$param[is.na(ctspec$value)] ) #duplicated hypermeans
      # )) 

  
  #
  # indvaryingcorindex= which(ctspec$param[ctspec$indvarying] %in% ctspec$param[is.na(ctspec$value) & 
  #     ctspec$indvarying & ctspec$row != ctspec$col & ctspec$matrix %in% c('T0VAR','DIFFUSION')])
  
  # DIFFUSION<-matrix(1,n.latent,n.latent)
  # for(rowi in ctspec$row[ctspec$matrix=='DIFFUSION' ]){
  #  DIFFUSION[
  
  #check diffusion indices input by user - which latents are involved in covariance
  if(diffusionindices=='all' || kalman==FALSE) diffusionindices = 1:n.latent
  diffusionindices = as.integer(diffusionindices)
  if(any(diffusionindices > n.latent)) stop('diffusionindices > than n.latent found!')
  if(length(diffusionindices) > n.latent) stop('diffusionindices vector cannot be longer than n.latent!')
  if(length(unique(diffusionindices)) < length(diffusionindices)) stop('diffusionindices vector cannot contain duplicates or!')
  ndiffusion=length(diffusionindices)
  message(paste(ndiffusion ,'/',n.latent,'latent variables required for covariance calculations'))
  
  
  
  
  # (vector[] Y, int ndatapoints, int[] nobs_y,int[,] whichobs_y, int[] subject, 
  # int nlatent, int nmanifest, 
  # int ntdpred,int ndiffusion, int[] T0check,vector dT,
  # matrix[] DRIFT, matrix[] LAMBDA, matrix[] DIFFUSION, matrix[] asymDIFFUSION, vector[] CINT, vector[] T0MEANS, 
  # matrix[] T0VAR, matrix[] MANIFESTVAR, vector[] MANIFESTMEANS, matrix IIlatent,
  # ',if(n.TDpred >0) 'matrix[] TDPREDEFFECT, vector[] tdpreds,','
  # int[] driftindex,int[] diffusionindex,int[] diffusionindices, int[] cintindex)
  
  verbosefilter<-function(){
    all <- c('etaprior[rowi]','etapriorcov[rowi]','Ypredcov_filt','Ypredcov_filt_chol',
      'K_filt','err','etapost[rowi]','etapostcov[rowi]')
    if(kalman) all <- c(all,'errscales[obscount:(obscount+nobsi-1)]','errtrans[obscount:(obscount+nobsi-1)]')
      
      t1plus<-c('discreteDRIFT[driftindex[rowi]]','discreteCINT[cintindex[rowi]]','discreteDIFFUSION[diffusionindex[rowi]]')
      out<-paste0('print("',all,' = ",', all,');')
      out<-c(out,paste0('if(T0check[rowi]==0) print("',t1plus,' = ",', t1plus,');'))
      return(paste0(out,collapse='\n '))
  }
  
  filteringfunc<-function(ppchecking){
    out<-paste0('
      {
      int subjecti;
      int counter;
      matrix[nlatent,nlatent] discreteDRIFT[',max(driftindex),']; 
      vector[nlatent] discreteCINT[',max(cintindex),'];
      ',if(!fixedkalman) paste0('matrix[ndiffusion,ndiffusion] discreteDIFFUSION[',max(diffusionindex),'];',
        if(!kalman) paste0('matrix[ndiffusion,ndiffusion] discreteDIFFUSIONchol[',max(diffusionindex),'];')),'
      vector[nlatent] etaprior[ndatapoints]; //prior for latent states
      ',if(kalman) paste0('matrix[ndiffusion, ndiffusion] etapriorcov[ndatapoints]; //prior for covariance of latent states
        vector[nlatent] etapost[ndatapoints]; //posterior for latent states'),'
      ',if(kalman) 'matrix[ndiffusion, ndiffusion] etapostcov[ndatapoints]; //posterior for covariance of latent states','
      ',if(!kalman) 'vector[nlatent] etapost[ndatapoints]; \n','
      
      vector[sum(nobs_y)] errtrans; // collection of prediction errors transformed to standard normal
      vector[sum(nobs_y)] errscales; // collection of prediction error scaling factors
      int obscount; // counter summing over number of non missing observations in each row
      int nobsi; 
      
      // pre-calculate necessary discrete time matrices      
      counter=0;
      for(rowi in 1:ndatapoints) {
      if(T0check[rowi]==0 && (rowi==1 || driftindex[rowi] > counter)) { 
      discreteDRIFT[driftindex[rowi]] = ',
      if(!continuoustime) paste0('DRIFT',checkvarying('DRIFT','[subject[rowi]]','[1]'),';'),
      if(continuoustime & !driftdiagonly) paste0('matrix_exp(DRIFT',checkvarying('DRIFT','[subject[rowi]]','[1]'),' * dT[rowi]);'),
      if(continuoustime & driftdiagonly) paste0('matrix_diagexp(DRIFT',checkvarying('DRIFT','[subject[rowi]]','[1]'),' * dT[rowi]);'),'
      counter=counter+1;
      }
      }
      counter=0;

      for(rowi in 1:ndatapoints) {
      if(T0check[rowi]==0 && (rowi==1 || diffusionindex[rowi] > counter)){ 
      discreteDIFFUSION[diffusionindex[rowi]] = ',
      if(!continuoustime) paste0('DIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[subject[rowi]]','[1]'),';'),
      if(continuoustime & !asymdiffusion) paste0('asymDIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[subject[rowi]]','[1]'),' - 
        quad_form_sym(asymDIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[subject[rowi]]','[1]'),' , discreteDRIFT[driftindex[rowi]][diffusionindices,diffusionindices]\');'),
      if(continuoustime & asymdiffusion) paste0('DIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[subject[rowi]]','[1]'),'[diffusionindices,diffusionindices] - 
        makesym(quad_form(DIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[subject[rowi]]','[1]'),'[diffusionindices,diffusionindices] , discreteDRIFT[driftindex[rowi]][diffusionindices,diffusionindices]\'));'),'
      counter=counter+1;
      ',if(!kalman) 'discreteDIFFUSIONchol[diffusionindex[rowi]] = cholesky_decompose(discreteDIFFUSION[diffusionindex[rowi]]);','
      }
      }
      counter=0;
      
      for(rowi in 1:ndatapoints) {
      if(T0check[rowi]==0 && (rowi==1 || cintindex[rowi] > counter)) { 
      discreteCINT[cintindex[rowi]] = ',
      if(!continuoustime) paste0('CINT',checkvarying('CINT','[subject[rowi]]','[1]'),';'),
      if(continuoustime) paste0('DRIFT',checkvarying('DRIFT','[subject[rowi]]','[1]'),' \\ (discreteDRIFT[driftindex[rowi]] - IIlatent) * CINT',
        checkvarying('CINT','[subject[rowi]]','[1]'),';'),'

      counter=counter+1;
      }
      }
      
      
      
      obscount=1; //running total of observed indicators
      for(rowi in 1:ndatapoints){
      int whichobs[nobs_y[rowi]];
      whichobs = whichobs_y[rowi][1:nobs_y[rowi]]; //which are not missing in this row
      subjecti=subject[rowi];
      nobsi = nobs_y[rowi]; //number of obs this row
      
      if(rowi!=1) obscount=obscount+nobs_y[rowi-1]; // number of non missing observations until now
      
      if(T0check[rowi] == 1) { // calculate initial matrices if this is first row for subjecti
      etaprior[rowi] = T0MEANS',checkvarying('T0MEANS','[subjecti]','[1]'),'; //prior for initial latent state
      ',if(n.TDpred > 0) paste0('etaprior[rowi] =TDPREDEFFECT',checkvarying('TDPREDEFFECT','[subjecti]','[1]'),' * tdpreds[rowi] + etaprior[rowi];'),'
      ',if(kalman) paste0('etapriorcov[rowi] =  T0VAR',checkvarying('T0VAR','[subjecti]','[1]'),'[diffusionindices,diffusionindices];'),'
      etapost[rowi] = etaprior[rowi];
      }
      
      if(T0check[rowi]==0){
      etaprior[rowi] = discreteCINT[cintindex[rowi]]  + discreteDRIFT[driftindex[rowi]] * etapost[rowi-1]; //prior for latent state of this row
      ',if(n.TDpred > 0) paste0('etaprior[rowi] =TDPREDEFFECT',checkvarying('TDPREDEFFECT','[subjecti]','[1]'),' * tdpreds[rowi] + etaprior[rowi];'),'
      ',if(kalman) paste0('etapriorcov[rowi] =  makesym(quad_form(etapostcov[rowi-1], discreteDRIFT[driftindex[rowi]][diffusionindices,diffusionindices]\')  + discreteDIFFUSION[diffusionindex[rowi]]);'),'
      ',if(!kalman) 'etapost[rowi] = etaprior[rowi] +  discreteDIFFUSIONchol[diffusionindex[rowi]] * etapostbase[(1+(rowi-1-subjecti)*nlatent):((rowi-subjecti)*nlatent)];','
      }
      
      ',if(kalman) 'etapost[rowi] = etaprior[rowi];
        etapostcov[rowi] = etapriorcov[rowi];','
      
      if (nobsi > 0) {  // if some observations create right size matrices for missingness and calculate...
      
      matrix[nobsi, nlatent] LAMBDA_filt;
      ',if(!binomial) 'vector[nobsi] err;','
      ',ifelse(binomial, 'int Y_filt[nobsi];','vector[nobsi] Y_filt;'),'
      
      ',if(kalman) paste0('
        matrix[nobsi, nobsi] Ypredcov_filt;
        //matrix[nobsi, nobsi] invYpredcov_filt_chol;
        matrix[ndiffusion, nobsi] K_filt; // kalman gain
        '),'
      
      ',if(!binomial) 'matrix[nobsi, nobsi] Ypredcov_filt_chol; \n','
      
      ',paste0('LAMBDA_filt = LAMBDA',checkvarying('LAMBDA','[subjecti]','[1]'),'[whichobs]; // and loading matrix'),'
      

      ',if(kalman) paste0(
        'Ypredcov_filt = makesym(quad_form(etapriorcov[rowi], LAMBDA_filt[,diffusionindices]\')) + MANIFESTVAR',checkvarying('MANIFESTVAR','[subjecti]','[1]'),'[whichobs,whichobs];
    
Ypredcov_filt_chol=cholesky_decompose(Ypredcov_filt); 

       K_filt = mdivide_right_spd(etapriorcov[rowi] * LAMBDA_filt[,diffusionindices]\', Ypredcov_filt); 

       etapostcov[rowi] = (IIlatent[diffusionindices,diffusionindices] - K_filt * LAMBDA_filt[,diffusionindices]) * etapriorcov[rowi];
        '),'
      
      ',if(!kalman & !binomial) paste0('Ypredcov_filt_chol = diag_matrix( sqrt(diagonal(MANIFESTVAR',checkvarying('MANIFESTVAR','[subjecti]','[1]'),'[whichobs,whichobs])));'),'
      
      ',if(!ppchecking) 'Y_filt = Y[rowi][whichobs];','
      
      ',if(kalman & ppchecking) paste0('Y_filt = multi_normal_rng( MANIFESTMEANS',
        checkvarying('MANIFESTMEANS','[subjecti]','[1]'),'[whichobs] + LAMBDA_filt * etaprior[rowi], Ypredcov_filt);'),'
      
      ',if(!kalman & !binomial & ppchecking) paste0('Y_filt = multi_normal_rng( MANIFESTMEANS',
        checkvarying('MANIFESTMEANS','[subjecti]','[1]'),'[whichobs] + LAMBDA_filt * etaprior[rowi], MANIFESTVAR',
        checkvarying('MANIFESTVAR','[subjecti]','[1]'),'[whichobs,whichobs]);'),'
      
      ',if(binomial & ppchecking) paste0('for(obsi in 1:nobsi) Y_filt[obsi] = bernoulli_rng(inv_logit(LAMBDA_filt * etapost[rowi] + ', 
        'MANIFESTMEANS',checkvarying('MANIFESTMEANS','[subjecti]','[1]'),'[whichobs])[obsi]);'),'
      
      ',if(kalman) paste0('err = Y_filt - ( MANIFESTMEANS',
        checkvarying('MANIFESTMEANS','[subjecti]','[1]'),'[whichobs] + LAMBDA_filt * etaprior[rowi] ); // prediction error'),
      if(!kalman & !binomial) paste0('err = Y_filt - ( MANIFESTMEANS',
        checkvarying('MANIFESTMEANS','[subjecti]','[1]'),
        '[whichobs] + LAMBDA_filt * etapost[rowi] ); // measurement error'),'


      ',if(kalman) 'etapost[rowi,diffusionindices] = etaprior[rowi][diffusionindices] + K_filt * err;','
      
      ',if(!binomial) '
      errtrans[obscount:(obscount+nobsi-1)]=mdivide_left_tri_low(Ypredcov_filt_chol, err); //transform pred errors to standard normal dist and collect
      errscales[obscount:(obscount+nobsi-1)]=log(diagonal(Ypredcov_filt_chol)); //account for transformation of scale in loglik ','
      
      ',if(binomial) paste0('ll =  ll + bernoulli_logit_lpmf(Y_filt | LAMBDA_filt * etapost[rowi] + ', 
        'MANIFESTMEANS',checkvarying('MANIFESTMEANS','[subjecti]','[1]'),'[whichobs]);'),'

    ',if(verbose) verbosefilter() ,'
      }
      }

      ',if(!binomial) 'll = ll+normal_lpdf(errtrans|0,1); 
      ll= ll - sum(errscales);','
      }
      ')
    return(out)
      }

  writemodel<-function(init=FALSE,nopriors=FALSE){
    stanmodel <- paste0('
      functions{

matrix matrix_diagexp(matrix in){
matrix[rows(in),rows(in)] out;
for(i in 1:rows(in)){
for(j in 1:rows(in)){
if(i==j) out[i,i] = exp(in[i,i]);
if(i!=j) out[i,j] = 0;
}}
return out;
}

matrix trisolve(matrix l, int unitdiag){ //inverse of triangular matrix
  matrix[rows(l),rows(l)] b ;
  matrix[rows(l),rows(l)] x;
  x = diag_matrix(rep_vector(1,rows(l)));
  b = x;

    for(j in 1:cols(x)){
      if(unitdiag != 1) x[j,j] = b[j,j] / l[1,1];
      
     if(rows(x)>1){
      for(m in 2:rows(x)){
      x[m,j]  =  (b[m,j] - (l[m,1:(m-1)] * x[1:(m-1),j]) ) / l[m,m];
      }
     }
    }
  return x;
}

matrix inverse_sym(matrix a){ // use ldlt decomposition to decompose symmetric non-definite matrices
  matrix[rows(a),rows(a)] d;
  matrix[rows(a),rows(a)] l;

  d=diag_matrix(rep_vector(0,rows(a)));
  l=diag_matrix(rep_vector(1,rows(a)));
  
  for(j in 1:rows(a)){
    for(i in (min(j+1,rows(a))):rows(a)){
      
      if(j==1) {
        d[j,j]=a[j,j];
        l[i,j]=1/d[j,j] * a[i,j];
      }
      if(j>1) {
        d[j,j]=a[j,j]- sum( square(l[j, 1:(j-1)]) * d[1:(j-1), 1:(j-1)]);
        if(i > j) l[i,j]=1/d[j,j] * ( a[i,j] - sum(l[i,1:(j-1)] .* l[j,1:(j-1)] * d[1:(j-1),1:(j-1)]));
      }
    }
  }
  l = trisolve(l,0);
  l = l\' * diag_matrix(rep_vector(1,rows(d)) ./ diagonal(d)) * l;
  return l;
}


 matrix covchol2corchol(matrix mat, int invert){ //converts from lower partial sd matrix to cor
      matrix[rows(mat),cols(mat)] o;
      vector[rows(mat)] s;
    o=mat;

    for(i in 1:rows(o)){ //set upper tri to lower
for(j in min(i+1,rows(mat)):rows(mat)){
o[j,i] = inv_logit(o[j,i])*2-1;  // can change cor prior here
o[i,j] = o[j,i];
}
      o[i,i]=1; // change to adjust prior for correlations
    }

if(invert==1) o = inverse_sym(o);


  for(i in 1:rows(o)){
      s[i] = inv_sqrt(o[i,] * o[,i]);
    if(is_inf(s[i])) s[i]=0;
    }
      o= diag_pre_multiply(s,o);
return o;
 }

      matrix sdcovchol2cov(matrix mat, int cholesky){ //converts from lower partial sd and diag sd to cov or cholesky cov
      matrix[rows(mat),rows(mat)] out;
      int invert;

invert = 0; //change integer for marginal or partial prior
if(rows(mat) > 1){
      out=covchol2corchol(mat,invert); 
      out= diag_pre_multiply(diagonal(mat), out);
}
if(rows(mat)==1) out[1,1] = mat[1,1];

      if(cholesky==0) out = out * out\';
      return(out);
      }
      

      matrix kron_prod(matrix mata, matrix matb){
      int m;
      int p;
      int n;
      int q;
      matrix[rows(mata)*rows(matb),cols(mata)*cols(matb)] C;
      m=rows(mata);
      p=rows(matb);
      n=cols(mata);
      q=cols(matb);
      for (i in 1:m){
      for (j in 1:n){
      for (k in 1:p){
      for (l in 1:q){
      C[p*(i-1)+k,q*(j-1)+l] = mata[i,j]*matb[k,l];
      }
      }
      }
      }
      return C;
      }
      
      matrix makesym(matrix mat){
      matrix[rows(mat),cols(mat)] out;
      out=mat;
      for(rowi in 1:rows(mat)){
      for(coli in 1:cols(mat)){
      if(coli > rowi) {
out[rowi,coli]=mat[coli,rowi] *.5 + mat[rowi,coli] *.5;
out[coli,rowi] = out[rowi,coli];

}
      }
      }
      return out;
      }
      
      matrix cov(vector[] mat,int nrows,int ncols){
      vector[ncols] means;
      matrix[nrows,ncols] centered;
      matrix[ncols,ncols] cov;
      for (coli in 1:ncols){
      means[coli] = mean(mat[,coli]);
      for (rowi in 1:nrows)  {
      centered[rowi,coli] = mat[rowi,coli] - means[coli];
      }
      }
      cov = centered\' * centered / (nrows-1);
      return cov; 
      }
      
      real inttoreal(int mynum){
      real out;
      out=mynum;
      return out;
      }


      }
      
      data {
      int<lower=0> ndatapoints;
      int<lower=1> nmanifest;
      int<lower=1> nlatent;
      int<lower=1> nsubjects;
      
      ',if(n.TIpred > 0) paste0('int<lower=0> ntipred; // number of time independent covariates
        matrix[nsubjects,ntipred] tipredsdata;
        int nmissingtipreds;'),'
      
      ',if(!binomial) 'vector[nmanifest] Y[ndatapoints]; \n',
      if(binomial) 'int Y[ndatapoints,nmanifest]; \n','
      int<lower=0> ntdpred; // number of time dependent covariates
      
      ',if(n.TDpred > 0) paste0('vector[ntdpred] tdpreds[ndatapoints];'),'
      
      vector[ndatapoints] dT; // time intervals
      int driftindex[ndatapoints]; //which discreteDRIFT matrix to use for each time point
      int diffusionindex[ndatapoints]; //which discreteDIFFUSION matrix to use for each time point
      int cintindex[ndatapoints]; //which discreteCINT matrix to use for each time point
      int subject[ndatapoints];
      int<lower=0> nparams;
      int T0check[ndatapoints]; // logical indicating which rows are the first for each subject
      int continuoustime; // logical indicating whether to incorporate timing information
      int nindvarying; // number of subject level parameters that are varying across subjects
      int nindvaryingoffdiagonals; //number of off diagonal parameters needed for hypercov matrix
      int notindvaryingindex[nparams-nindvarying];
      ',if(nindvarying>0) paste0('int indvaryingindex[nindvarying];
        vector[nindvarying] sdscale;'),'
      
      ',if(!is.na(ctstanmodel$stationarymeanprior)) 'vector[nlatent] stationarymeanprior; // prior std dev for difference between process asymptotic mean and initial mean','
      ',if(!is.na(ctstanmodel$stationaryvarprior)) 'vector[nlatent] stationaryvarprior; // prior std dev for difference between process asymptotic variance and initial variance','

      int nt0varstationary;
      int nt0meansstationary;
      int t0varstationary [nt0varstationary, 2];
      int t0meansstationary [nt0meansstationary, 2];
      
      int<lower = 0, upper = nmanifest> nobs_y[ndatapoints];  // number of observed variables per observation
      int<lower = 0, upper = nmanifest> whichobs_y[ndatapoints, nmanifest]; // index of which variables are observed per observation
      int<lower=0,upper=nlatent> ndiffusion; //number of latents involved in covariance calcs
      int<lower=0,upper=nlatent> diffusionindices[ndiffusion]; //index of which latent variables are involved in covariance calculations
      }
      
      transformed data{
      matrix[nlatent,nlatent] IIlatent;
      matrix[nlatent*nlatent,nlatent*nlatent] IIlatent2;
      IIlatent = diag_matrix(rep_vector(1,nlatent));
      IIlatent2 = diag_matrix(rep_vector(1,nlatent*nlatent));
      }
      
      parameters {
      vector[nparams] hypermeans',if(!esthyper) 'base','; // population level means \n','
      
      ',if(any(indvarying)) paste0(
        'vector',if(!is.na(ctstanmodel$rawhypersdlowerbound)) paste0('<lower=',ctstanmodel$rawhypersdlowerbound[1],'>'),'[nindvarying] rawhypersd',if(!esthyper) 'base','; //population level std dev
        //cholesky_factor_corr[nindvarying] hypercorrchol; // population level cholesky correlation
        //cov_matrix[nindvarying] wishmat;
        vector[nindvaryingoffdiagonals] sqrtpcov;
        vector[nindvarying*nsubjects] indparamsbase; //subject level parameters
        '),'
      
      ',if(n.TIpred > 0) paste0('
        vector[',sum(unlist(ctspec[,paste0(TIpredNames,'_effect')])),'] tipredeffectparams; // effects of time independent covariates
        vector[nmissingtipreds] tipredsimputed; \n'),'
      
      ',if(!kalman)  'vector[nlatent*ndatapoints] etapostbase; //sampled latent states posterior','
      
      
      }
      
      transformed parameters{
      ',if(!esthyper) 'vector[nparams] hypermeans;','

',if(nonexplosive) paste0('
        vector[',checkvarying(c('DRIFT'),'nsubjects','1'),'] dets;
        real detpenalty;
'),'

      ',if(nindvarying > 0) paste0('vector[nindvarying] indparams[nsubjects]; 
       vector[nindvarying] hypersd',if(!esthyper) 'base','; //population level std dev
        matrix[nindvarying,nindvarying] sqrtpcovmat;
        matrix[nindvarying,nindvarying] hypercorrchol;
        matrix[nindvarying,nindvarying] hypercovchol; 
        ',if(!esthyper) paste0('matrix[nindvarying,nindvarying] mlcov;
          vector[nindvarying] hypersd;')),'
      
      matrix[nlatent,nlatent] DIFFUSION',checkvarying('DIFFUSION','[nsubjects]','[1]'),'; //additive latent process variance
      matrix[nlatent,nlatent] T0VAR',checkvarying(if(!stationary & nt0varstationary ==0) 'T0VAR' else(c('T0VAR','DRIFT','DIFFUSION')),'[nsubjects]','[1]'),'; //initial latent process variance
      ',if(!kalman)  paste0('matrix[nlatent,nlatent] T0VARchol',checkvarying('T0VAR','[nsubjects]','[1]'),'; //initial latent process variance'),'
      matrix[nlatent,nlatent] DRIFT',checkvarying('DRIFT','[nsubjects]','[1]'),'; //dynamic relationships of processes
      ',if(!binomial) paste0('matrix[nmanifest,nmanifest] MANIFESTVAR',checkvarying('MANIFESTVAR','[nsubjects]','[1]'),'; // manifest error variance'),'
      vector[nmanifest] MANIFESTMEANS',checkvarying('MANIFESTMEANS','[nsubjects]','[1]'),';
      vector[nlatent] T0MEANS',checkvarying(if(!stationary & nt0meansstationary ==0) 'T0MEANS' else c('T0MEANS','DRIFT','CINT'),'[nsubjects]','[1]'),'; // initial (T0) latent states
      matrix[nmanifest,nlatent] LAMBDA',checkvarying('LAMBDA','[nsubjects]','[1]'),'; // loading matrix
      vector[nlatent] CINT',checkvarying('CINT','[nsubjects]','[1]'),'; // latent process intercept
      ',if(nt0meansstationary > 0) paste0('vector[nlatent] asymCINT',checkvarying(c('DRIFT','CINT'),'[nsubjects]','[1]'),'; // latent process asymptotic level'),'
      ',if(!asymdiffusion) paste0('matrix[ndiffusion,ndiffusion] asymDIFFUSION',checkvarying(c('DIFFUSION','DRIFT'),'[nsubjects]','[1]'),'; //latent process variance as time interval goes to inf'),'
      
      ',if(n.TDpred > 0) paste0('matrix[nlatent,ntdpred] TDPREDEFFECT',checkvarying('TDPREDEFFECT','[nsubjects]','[1]'),'; // effect of time dependent predictors'),'
      ',if(n.TIpred > 0) paste0('
        matrix[',nindvarying,',',n.TIpred,'] tipredeffect; //design matrix of individual time independent predictor effects
        matrix[nsubjects,ntipred] tipreds; //tipred values to fill from data and, when needed, imputation vector'),'
      
      ',if(n.TIpred > 0) paste0('
    {
      int counter;
      counter = 0;
      for(rowi in 1:rows(tipreds)){
        for(coli in 1:cols(tipreds)){
          if(tipredsdata[rowi,coli]==99999) {
            counter = counter + 1;
            tipreds[rowi,coli] = tipredsimputed[counter];
          } else tipreds[rowi,coli] = tipredsdata[rowi,coli];
        }
      }
    }
      '),

if(n.TIpred > 0) paste0(unlist(lapply(1,function(x){ ## collects all the time independent predictors effects into the design matrix
        count<-0
        tirow<-0
        out<-c()
        for(rowi in 1:nrow(ctspec[,])){
          if(is.na(ctspec$value[rowi]) & ctspec$indvarying[rowi]) {
            tirow<-tirow+1
            for(predi in 1:n.TIpred){
              if(ctspec[rowi, paste0(TIpredNames[predi],'_effect')] == FALSE) out<-c(out, ' tipredeffect[',tirow,', ', predi,'] = 0; \n')
              if(ctspec[rowi, paste0(TIpredNames[predi],'_effect')] == TRUE) {
                count<-count+1
                out<-c(out, ' tipredeffect[',tirow,', ', predi,'] = tipredeffectparams[',count,']; \n ')
              }
            }
          }}
        return(out)
      })),collapse=''),'

  

      ',if(!esthyper) 'hypermeans = hypermeansbase; \n','
      ',if(any(indvarying)) paste0('
        hypersd = ',ctstanmodel$hypersdtransform, ';

',if(1==1) '
if(nindvarying > 0){
{
int counter;
sqrtpcovmat=diag_matrix(rep_vector(-99,nindvarying));
if(nindvarying > 1){
counter=0;
        for(j in 1:(nindvarying-1)){
        for(i in (j+1):nindvarying){
counter=counter+1;
        sqrtpcovmat[i,j]=sqrtpcov[counter];
        }}
}
}

if(nindvarying > 1) hypercorrchol = covchol2corchol(sqrtpcovmat,0); //change int to change from partial to marginal prior
}
','

',if(1==99) '
if(nindvarying >1)hypercorrchol = cholesky_decompose(quad_form_diag(wishmat, inv_sqrt( diagonal(wishmat))));
','

if(nindvarying ==1) hypercorrchol = diag_matrix(rep_vector(1,1));


        hypercovchol= diag_pre_multiply(hypersd',if(!esthyper) 'base',
        ', hypercorrchol);  
        
        for(subi in 1:nsubjects) {
        indparams[subi]= 
        hypercovchol * indparamsbase[(1+(subi-1)*nindvarying):(subi*nindvarying)] +',
        'hypermeans',
        if(!esthyper) 'base', 
        '[indvaryingindex]',
        if(n.TIpred>0) ' + tipredeffect * tipreds[subi]\' ',
        ';  
        }
        ',if(!esthyper) paste0('
          mlcov = cov(indparams,nsubjects,nindvarying) + diag_matrix(rep_vector(.000001,nindvarying));
          hypersd=sqrt(diagonal(mlcov));
          hypermeans=hypermeansbase;
          for(pari in 1:nindvarying){
          hypermeans[indvaryingindex[pari]]=mean(indparams[,pari]);
          }
          '),'
        '),'
      
      {
      vector[ndiffusion*ndiffusion] asymDIFFUSIONvec',checkvarying(c('DIFFUSION','DRIFT'),'[nsubjects]','[1]'),';
      ',if(continuoustime & !asymdiffusion) paste0('matrix[ndiffusion*ndiffusion,ndiffusion*ndiffusion] DRIFTHATCH',checkvarying(c('DRIFT'),'[nsubjects]','[1]'),';'),'
      
      // create subject specific parameter matrices from fixed and transformed free effects 
      ',paste0(unlist(lapply(1:nrow(ctspec),function(rowi) {
        
        x<-paste0( #left side - which matrix element
          checkvarying(ctspec[rowi,'matrix'],'for(subi in 1:nsubjects) '),
          ctspec[rowi,'matrix'], checkvarying(ctspec[rowi,'matrix'],'[subi]','[1]'),'[', ctspec[rowi,'row'], 
          if(ctspec[rowi,'matrix'] %in% c('LAMBDA','DRIFT','DIFFUSION',
            'MANIFESTVAR', 'TDPREDEFFECT', 'T0VAR')) paste0(' , ', ctspec[rowi,'col']),
          ']') 
        
        y<- paste0('(', #right side - using hypermean or indparam? and which?
          if(is.na(ctspec[rowi,'value']) & (!ctspec$indvarying[rowi])) paste0('hypermeans[',which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),']'),
          if(ctspec[rowi,'indvarying']) paste0('indparams[subi][',which(ctspec[ctspec$indvarying,'param']==ctspec[rowi,'param']),']')
          ,')')
        
        #right side if fixed value
        if(!is.na(ctspec[rowi,'value'])) out<-paste0(x,' = ',ctspec[rowi,'value'],'; \n')
        
        #transform right side if appropriate
        if(!is.na(ctspec$transform[rowi]) & is.na(ctspec$value[rowi])) out<-paste0(
          x, ' = ', gsub('param',y,ctspec$transform[rowi]),'; \n')
        return(out)
      })),collapse=''),
      
      '
      // create subject specific parameter matrices from duplicated transformed free effects 
      ',if(nrow(ctspecduplicates)>0) paste0(unlist(lapply(1:nrow(ctspecduplicates),function(rowi) {
        
        x<-paste0( #left side - which matrix element
          checkvarying(ctspecduplicates[rowi,'matrix'],'for(subi in 1:nsubjects) '),
          ctspecduplicates[rowi,'matrix'], checkvarying(ctspecduplicates[rowi,'matrix'],'[subi]','[1]'),'[', ctspecduplicates[rowi,'row'], 
          if(ctspecduplicates[rowi,'matrix'] %in% c('LAMBDA','DRIFT','DIFFUSION',
            'MANIFESTVAR', 'TDPREDEFFECT', 'T0VAR')) paste0(' , ', ctspecduplicates[rowi,'col']),
          ']') 
        
        y<- paste0('(', #right side - using hypermean or indparam? and which?
          if(is.na(ctspecduplicates[rowi,'value']) & (!ctspecduplicates$indvarying[rowi])) ctspecduplicates$hypermeanduplicates[rowi],
          if(ctspecduplicates[rowi,'indvarying']) paste0('indparams[subi][',
            which(
              ctspec[ctspec$indvarying,'param']==ctspecduplicates[rowi,'param'] ),
            ']')
          ,')')
        
        #transform right side if appropriate
        if(!is.na(ctspecduplicates$transform[rowi]) & is.na(ctspecduplicates$value[rowi])) out<-paste0(
          x, ' = ', gsub('param',y,ctspecduplicates$transform[rowi]),'; \n')
        return(out)
        })),collapse=''),
      '
      
      
      // perform any whole matrix transformations 


      for(individual in 1:',checkvarying('DIFFUSION','nsubjects','1'),') DIFFUSION[individual] = sdcovchol2cov(DIFFUSION[individual],0);

      ',if(continuoustime) paste0('
        for(individual in 1:',checkvarying('DRIFT','nsubjects','1'),') {
        ',if(!asymdiffusion) paste0(
          'DRIFTHATCH[individual] = kron_prod(DRIFT[individual][diffusionindices,diffusionindices],diag_matrix(rep_vector(1, ndiffusion))) +  ',
          'kron_prod(diag_matrix(rep_vector(1, ndiffusion)),DRIFT[individual][diffusionindices,diffusionindices]);
          '),'
        }
        
        ',if(!asymdiffusion) paste0('
          for(individual in 1:',checkvarying(c('DIFFUSION','DRIFT'),'nsubjects','1'),'){
          asymDIFFUSIONvec[individual] =  -(DRIFTHATCH',checkvarying(c('DRIFT'),'[individual]','[1]'),
          ' \\ to_vector(DIFFUSION',checkvarying(c('DIFFUSION'),'[individual]','[1]'),
          '[diffusionindices,diffusionindices]));

          for(drowi in 1:ndiffusion) {
          for(dcoli in 1:ndiffusion){
          asymDIFFUSION[individual][drowi,dcoli] =  asymDIFFUSIONvec[individual][drowi+(dcoli-1)*ndiffusion];
          }}
          asymDIFFUSION[individual] = makesym(asymDIFFUSION[individual]);
          }
          ')),
      
      if(!continuoustime & !asymdiffusion) paste0('
        for(individual in 1:',checkvarying(c('DIFFUSION','DRIFT'),'nsubjects','1'),'){
        asymDIFFUSIONvec[individual] = (IIlatent2 - kron_prod(DRIFT',checkvarying(c('DRIFT'),'[individual]','[1]'),', DRIFT',checkvarying(c('DRIFT'),'[individual]','[1]'),')) * 
        to_vector(DIFFUSION',checkvarying(c('DIFFUSION'),'[individual]','[1]'),');
        for(drowi in 1:nlatent) {
        for(dcoli in 1:nlatent){
        asymDIFFUSION[individual][drowi,dcoli] =  asymDIFFUSIONvec[individual][drowi+(dcoli-1)*nlatent];
        }}
        }
        '),'


      ',if(nt0meansstationary > 0 & continuoustime) paste0('
          for(individual in 1:',checkvarying(c('CINT','DRIFT'),'nsubjects','1'),'){
        asymCINT[individual] =  -DRIFT',checkvarying(c('DRIFT'),'[individual]','[1]'),
        ' \\ CINT',checkvarying(c('CINT'),'[individual]','[1]'),';
        }
        '),

      if(nt0meansstationary > 0 & !continuoustime) paste0('
        for(individual in 1:',checkvarying(c('CINT','DRIFT'),'nsubjects','1'),'){
           asymCINT[individual] =  (latentII - DRIFT',checkvarying(c('DRIFT'),'[individual]','[1]'),
          ') \\ CINT',checkvarying(c('CINT'),'[individual]','[1]'),';
        }
      '),'

      
      ',if(!binomial) paste0(
        'for(individual in 1:',checkvarying('MANIFESTVAR','nsubjects','1'),') {
          MANIFESTVAR[individual] = square(MANIFESTVAR[individual]);
          //MANIFESTVAR[individual] = asymDIFFUSION[individual] .* MANIFESTVAR[individual] ;
}
        '),'

      
      for(individual in 1:',
      checkvarying(if(!stationary & nt0varstationary==0) 'T0VAR' else(c('T0VAR','DRIFT','DIFFUSION')),'nsubjects','1'),') {
      T0VAR[individual] = ',
      if(!stationary) paste0('sdcovchol2cov(T0VAR[individual],0);'),
      if(stationary) paste0(if(!asymdiffusion) 'asym', 'DIFFUSION',
        checkvarying(c('DRIFT','DIFFUSION'),'[individual]','[1]'),';'),'
      ',if(!kalman) 'T0VARchol[individual] = cholesky_decompose(T0VAR[individual]);',
      'if(nt0varstationary > 0) for(rowi in 1:nt0varstationary){
          T0VAR[individual][t0varstationary[rowi,1],t0varstationary[rowi,2] ] = 
            asymDIFFUSION[individual][t0varstationary[rowi,1],t0varstationary[rowi,2] ];
          }
      }
      
     for(individual in 1:',
checkvarying(if(!stationary & nt0meansstationary==0) 'T0MEANS' else(c('T0MEANS','DRIFT','CINT')),'nsubjects','1'),') {
',if(stationary) paste0('T0MEANS[individual] = -DRIFT', checkvarying('DRIFT','[individual]','[1]'),
        ' \\ CINT',checkvarying('CINT','[individual]','[1]'),'; // prior for initial latent states is stationary mean'),'
        ',if(nt0meansstationary > 0) paste0('if(nt0meansstationary > 0) for(rowi in 1:nt0meansstationary){
          T0MEANS[individual][t0meansstationary[rowi,1]] = 
            asymCINT',checkvarying(c('DRIFT','CINT'),'[individual]','[1]'),'[t0meansstationary[rowi,1]];
        }'),'
      }
      
      }

',if(nonexplosive) paste0('
      // non explosive priors
              for(individual in 1:',checkvarying(c('DRIFT','DIFFUSION'),'nsubjects','1'),'){
      dets[individual]= determinant(asymDIFFUSION[individual]);
              }
detpenalty = sum(-log( 1-(inv_logit(-dets) .* (dets))+1)) / ',checkvarying(c('DRIFT'),'nsubjects','1'),';
'),'

      }
      
      model{
      real ll;
      ',if(!nopriors) paste0('target += normal_lpdf(hypermeans|0,1);'),'
      
      ',if(n.TIpred > 0 & !nopriors) paste0('
        tipredeffectparams ~ ',ctstanmodel$tipredeffectprior, '; 
        tipredsimputed ~ ',ctstanmodel$tipredsimputedprior,';\n '),' 
      
      ',if(any(ctspec$indvarying) & !nopriors) paste0(
        '//hypercorrchol ~ lkj_corr_cholesky(.1); 
        if(nindvarying >1) sqrtpcov ~ normal(0,1);
        //wishmat ~ inv_wishart(nindvarying,diag_matrix(rep_vector(1,nindvarying)));
        indparamsbase ~ normal(0,1); 
        rawhypersd',if(!esthyper) 'base', ' ~ ',ctstanmodel$rawhypersd,';',
        if(!esthyper) 'target += -log(determinant(mlcov)); // /2*nsubjects; 
        // target += multi_normal_lpdf(indparams| hypermeans[indvaryingindex], mlcov);','
        '),'
      
      ',if(!kalman) 'etapostbase ~ normal(0,1); \n','
      
      ll = 0;

   ',if(!binomial & 99==1) paste0(
     'for(individual in 1:',checkvarying(c('MANIFESTVAR','DIFFUSION','DRIFT'),'nsubjects','1'),') {
    diagonal(asymDIFFUSION[individual]) ./ (diagonal(MANIFESTVAR[individual]) + diagonal(asymDIFFUSION[individual])) ~  beta(2,2); 
  }'),'
      
      // adjust partial correlation probabilities 
      ',if(!nopriors) paste0(unlist(lapply(1:nrow(ctspec),function(rowi) {
        out<-''
        if(ctspec$matrix[rowi] %in% c('T0VAR','DIFFUSION') & ctspec$row[rowi] > ctspec$col[rowi] & is.na(ctspec$value[rowi])) {
          out=paste0('//target += beta_lpdf(inv_logit(',
            'hypermeans[',which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),']',
            ')| 1.5 + (inttoreal(nlatent)-1)/2 - .6 * ',ctspec$col[rowi],', 1.5 + (inttoreal(nlatent)-1)/2 - .6 * ',ctspec$col[rowi],'); \n ')
        }
        return(out)
      })),collapse=''),'

',if(nonexplosive) '
    // non explosive priors - converging to stationary
target +=  detpenalty;
','

      
      ',if(!is.na(ctstanmodel$stationaryvarprior)) paste0('
        // stationarity priors
        for(individual in 1:nsubjects) {
        (diagonal(',
        if(!asymdiffusion) 'asym', 'DIFFUSION[',
        checkvarying(c('DIFFUSION','DRIFT'),'individual','1'),']) - diagonal(T0VAR[',
        checkvarying('T0VAR','individual','1'),'][diffusionindices,diffusionindices])) ~ normal(0,stationaryvarprior); // variance stationarity prior
        }
        '),'
      
      ',if(!is.na(ctstanmodel$stationarymeanprior)) paste0('
        for(individual in 1:nsubjects) {
        T0MEANS[',checkvarying('T0MEANS','individual','1'),'] - ',
        '( DRIFT[',checkvarying('DRIFT','individual','1'),'] \\ CINT[',checkvarying('CINT','individual','1'),'] )',
        ' ~ normal(0,stationarymeanprior); // mean stationarity prior
        }
        '),'
      
      
      ',filteringfunc(ppchecking=FALSE),'
      target += ll;
      
      
      ',if(verbose) paste0('
        print("lp = ", target());
        '),'
      
        }
      generated quantities{
      
      ',paste0('real hmean_',ctspec$param[is.na(ctspec$value)],'; \n',collapse=''),'
      
      ',if(nindvarying > 0) paste0(unlist(lapply(1:nrow(ctspec),function(rowi){
        if(ctspec$indvarying[rowi]) paste0('real hsd_',ctspec$param[rowi],'; \n')
      })),collapse=''),'

      ',if(n.TIpred > 0) paste0(unlist(lapply(1:n.TIpred,function(tip){
        paste0(unlist(lapply(1:nrow(ctspec),function(rowi){
          if(ctspec$indvarying[rowi] & 
              ctspec[,paste0(TIpredNames[tip],'_effect')][rowi]) paste0('real tipred_',
                TIpredNames[tip], '_on_', ctspec$param[rowi],'; \n'
              )
        })),collapse='')
        })),collapse=''),'

      ',paste0(unlist(lapply(1:nrow(ctspec),function(rowi){
        if(is.na(ctspec$value[rowi])) paste0('hmean_',ctspec$param[rowi],' = ',
          gsub('param',
            paste0('hypermeans[',
              which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),
              ']'),
            ctspec$transform[rowi]),'; \n')
      })),collapse=''),'

      
      ',paste0(unlist(lapply(1:nrow(ctspec),function(rowi){
        if(ctspec$indvarying[rowi]) paste0('hsd_',ctspec$param[rowi],' = ',
          'hypersd[',which(ctspec$param[ctspec$indvarying] == ctspec$param[rowi]),']; \n',
          if(!is.na(ctspec$transform[rowi])) paste0(
            'hsd_',ctspec$param[rowi],' = fabs
            ((', 
            gsub('param', paste0('(hypermeans[',
              which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),
              '] + hsd_',
              ctspec$param[rowi],')'),ctspec$transform[rowi]), ') - (',
            gsub('param', 
              paste0('(hypermeans[',which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),
                '] -  hsd_',
                ctspec$param[rowi],')'),ctspec$transform[rowi]),'))/2 ; \n')
            )
      })),collapse=''),'

      
      ',if(n.TIpred > 0) paste0(unlist(lapply(1:n.TIpred,function(tip){
        paste0(unlist(lapply(1:nrow(ctspec),function(rowi){
          if(ctspec$indvarying[rowi] & ctspec[,paste0(TIpredNames[tip],'_effect')][rowi]) paste0('
            tipred_',TIpredNames[tip], '_on_', ctspec$param[rowi],' = ',
            'tipredeffect[',which(ctspec$param[ctspec$indvarying] == ctspec$param[rowi]),',',tip,']; \n',
            if(!is.na(ctspec$transform[rowi])) paste0('tipred_', TIpredNames[tip], '_on_', ctspec$param[rowi],' = ((', 
              gsub('param', 
                paste0('hypermeans[',which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),'] + tipredeffect[',which(ctspec$param[ctspec$indvarying] == ctspec$param[rowi]),',',tip,']'),
                ctspec$transform[rowi]), 
              ') - (',
              gsub('param', 
                paste0('hypermeans[',which(ctspec$param[is.na(ctspec$value)] == ctspec$param[rowi]),'] - tipredeffect[',which(ctspec$param[ctspec$indvarying] == ctspec$param[rowi]),',',tip,']'),
                ctspec$transform[rowi]),'))/2; \n')
          )
        })),collapse='')
      })),collapse=''),'

      }')
    }
  
  if(is.na(stanmodeltext)) stanmodeltext<-writemodel(init=initwithoptim,nopriors= nopriors)

  # out<-list(stanmodeltext=stanmodeltext)
  
  standata<-list(
    Y=cbind(as.matrix(datalong[,manifestNames])),
    subject=datalong[,'id'],
    nsubjects=nsubjects,
    nmanifest=n.manifest,
    T0check=T0check,
    indvaryingindex=indvaryingindex,
    notindvaryingindex=array(which(ctspec$indvarying[is.na(ctspec$value)] == FALSE),dim=nparams-nindvarying),
    continuoustime=sum(continuoustime),
    nlatent=n.latent,
    ntipred=n.TIpred,
    ntdpred=n.TDpred,
    nparams=nparams,
    nindvarying=nindvarying,
    nindvaryingoffdiagonals = (nindvarying*nindvarying-nindvarying)/2,
    sdscale=array(ctspec$sdscale[ctspec$indvarying]),
    IIparams = diag(nparams),
    ndatapoints=nrow(datalong),
    dT=dT,
    time=datalong[,timeName],
    driftindex=driftindex,
    cintindex=cintindex,
    nt0varstationary=nt0varstationary,
    nt0meansstationary=nt0meansstationary,
    t0varstationary=matrix(as.numeric(t0varstationary),ncol=2),
    t0meansstationary=matrix(as.numeric(t0meansstationary),ncol=2),
    diffusionindex=diffusionindex,
    diffusionindices=array(diffusionindices,dim=ndiffusion),
    ndiffusion=ndiffusion,
    nobs_y=array(apply(datalong[,manifestNames,drop=FALSE],1,function(x) length(x[x!=99999])),dim=nrow(datalong)),
    whichobs_y=matrix(t(apply(datalong[,manifestNames,drop=FALSE],1,function(x) {
      out<-as.numeric(which(x!=99999))
      if(length(out)==0) out<-rep(0,n.manifest)
      if(length(out)<n.manifest) out<-c(out,rep(0,n.manifest-length(out)))
      out
    }) ),nrow=c(nrow(datalong),ncol=n.manifest)))

  if(!is.na(ctstanmodel$stationarymeanprior)) standata$stationarymeanprior=array(ctstanmodel$stationarymeanprior,dim=n.latent)
  if(!is.na(ctstanmodel$stationaryvarprior)) standata$stationaryvarprior=array(ctstanmodel$stationaryvarprior,dim=n.latent)
  
  if(n.TIpred > 0) {
    standata$tipredsdata <- as.matrix(tipreds)
    standata$nmissingtipreds <- length(tipreds[tipreds== 99999])
}
  
  if(n.TDpred > 0) standata<-c(standata,list(tdpreds=array(tdpreds,dim=c(nrow(tdpreds),ncol(tdpreds)))))
  
  if(fit==TRUE){
    
    message('Compiling model, ignore forthcoming warning re number of chains...')
    sm <- stan(model_code = c(stanmodeltext),
      data = standata, chains = 0, iter=1, control=list(max_treedepth=1))
    
    #control arguments for rstan
    if(is.null(control$adapt_term_buffer)) control$adapt_term_buffer <- min(c(iter/10,max(iter-20,75)))
    if(is.null(control$adapt_delta)) control$adapt_delta <- .9
    if(is.null(control$adapt_window)) control$adapt_window <- 5
    if(is.null(control$max_treedepth)) control$max_treedepth <- 10
    if(is.null(control$adapt_init_buffer)) adapt_init_buffer=10
    
    
    if(initwithoptim & chains > 0){#optimize with bfgs for initial values
      
      npars=get_num_upars(sm)
      
      if(any(ctspec$indvarying)) hypersdindex=(nparams+1):(nparams+ sum(ctspec$indvarying)) else hypersdindex<-NULL
      
      lp<-function(parm) {
        parm[hypersdindex]<-0
        out<-try(log_prob(sm,upars=parm))
        if(class(out)=='try-error') {
          out=-1e20
        }
        return(out)
      }
      
      grf<-function(parm) {
        parm[hypersdindex]<-0
        out=grad_log_prob(sm,upars=parm)
        out[hypersdindex]=0
        return(out)
      }
      
      message('Optimizing to get inits...')
      optimfit <- stats::optim(stats::rnorm(npars,0,.001), lp, gr=grf, 
        control = list(fnscale = -1,trace=0,parscale=rep(.00001,npars),maxit=2000,factr=1e-12,lmm=100), 
        method='L-BFGS-B',hessian = FALSE)
      parsout=optimfit$par
      parsout[hypersdindex]=0
      
      inits=constrain_pars(sm,parsout)
      message('Got inits.')
    }
    
    if(!is.null(inits)){
      staninits=list(inits)
      if(chains > 0){
        for(i in 2:chains){
          staninits[[i]]<-inits
        }
      }
    }
    
    
    if(is.null(inits)){
      staninits=list()
      if(chains > 0){
        for(i in 1:(chains)){
          staninits[[i]]=list(etapost=array(stats::rnorm(nrow(datalong)*n.latent,0,.1),dim=c(nrow(datalong),n.latent)),
            rawhypersd = array(exp(rnorm(nindvarying,-3,1)),dim=c(nindvarying)))
        }
      }
    }
    
    if(cores=='maxneeded') cores=min(c(chains,parallel::detectCores()))
    
    if(!optimize & !vb) {
      message('Sampling...')
      
      stanargs <- list(fit = sm, 
        # enable_random_init=TRUE,
        init_r=.05,
        init=staninits,
        refresh=20,
        iter=iter,
        data = standata, chains = ifelse(optimize==FALSE & vb==FALSE,chains,0), control=control,
        cores=cores,
        ...) 
      
      if(plot==TRUE) stanfit <- do.call(stanWplot,stanargs) else stanfit <- do.call(stan,stanargs)
    }
    
    if(optimize==TRUE && fit==TRUE) {
      
      
      # stanfit <- optimizing(object = stanfit@stanmodel, 
      #   init=0,
      #   # algorithm='BFGS',
      #   as_vector=F,
      #   history_size=6,
      #   init_alpha=.00001,
      #   tol_obj=1e-12, tol_grad=1e-12,tol_param=1e-12,tol_rel_grad=0, tol_rel_obj=0,
      #   data = standata, iter=iter)
      
      npars=get_num_upars(sm)
      
      if(any(ctspec$indvarying)) hypersdindex=(nparams+1):(nparams+ sum(ctspec$indvarying)) else hypersdindex<-NULL
      
      lp<-function(parm) {
        out<-try(log_prob(sm,upars=parm))
        if(class(out)=='try-error') out=-1e20
        return(-out)
      }
      
      grf<-function(parm) {
        out=try(grad_log_prob(sm,upars=parm))
        if(class(out)=='try-error') out=rnorm(length(parm))
        return(-out)
      }
      
      message('Optimizing...')
      convergence=1
      iteri=0
      optimstarts=stats::rnorm(npars,0,.001)
      while(convergence==1){
        if(iteri > 0) {
          message(paste0('Iteration ', iteri * 500, ', Log prob = ', optimfit$value))
          optimstarts=optimfit$par
        }
        iteri=iteri+1
        optimfit <- stats::optim(optimstarts, lp, gr=grf, 
          control = list(trace=0,maxit=500,factr=1e5), 
          method='L-BFGS-B',hessian = TRUE)
        convergence=optimfit$convergence
      }
      
      est=optimfit$par
      sds=sqrt(diag(solve(optimfit$hessian)))
      lest= est - 1.96 * sds
      uest= est + 1.96 * sds
      
      transformedpars=cbind(unlist(constrain_pars(sm,lest)),
        unlist(constrain_pars(sm,est)),
        unlist(constrain_pars(sm,uest)))
      colnames(transformedpars)=c('2.5%','mean','97.5%')
      
      stanfit=list(optimfit=optimfit,transformedpars=transformedpars)
    }
    
    if(vb==TRUE && fit==TRUE) {
      stanfit <- vb(object = sm@stanmodel, 
        iter=iter,
        eta=1e-6,
        data = standata,...)
      
    }
    
    #convert missings back to NA's for data output
    standataout<-unlist(standata)
    standataout[standataout==99999] <- NA
    standataout <- utils::relist(standataout,skeleton=standata)
    
    
    out <- list(args=args,stanmodeltext=stanmodeltext, data=standataout, ctstanmodel=ctstanmodel,stanfit=stanfit)
    class(out) <- 'ctStanFit'
    
  } # end if fit==TRUE
  
  if(!fit) out=list(stanmodeltext=stanmodeltext,data=standata, ctstanmodel=ctstanmodel)
  
  
  return(out)
  }

