#' ctStanKalman 
#'
#' Outputs predicted, updated, and smoothed estimates of manifest indicators and latent states, 
#' with covariances, for specific subjects from data fit with \code{\link{ctStanFit}}, 
#' based on medians of parameter distribution.
#' 
#' @param ctstanfitobj fit object as generated by \code{\link{ctStanFit}}.
#' @param datalong Optional long format data object as used by \code{\link{ctStanFit}}. 
#' If not included, data from ctstanfitobj will used. 
#' @param timerange Either 'asdata' to just use the observed data range, or a numeric vector of length 2 denoting start and end of time range, 
#' allowing for estimates outside the range of observed data.
#' @param timestep Either 'asdata' to just use the observed data 
#' (which also requires 'asdata' for timerange) or a positive numeric value
#' indicating the time step to use for interpolating values.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param plot Logical. If TRUE, plots output instead of returning it. 
#' See \code{\link{ctStanKalmanPlot}} for the possible arguments.
#' @param ... additional arguments to pass to \code{\link{ctStanKalmanPlot}}.
#' @return Returns a list containing matrix objects etaprior, etaupd, etasmooth, y, yprior, 
#' yupd, ysmooth, prederror, time, loglik,  with values for each time point in each row. 
#' eta refers to latent states and y to manifest indicators - y itself is thus just 
#' the input data. 
#' Covariance matrices etapriorcov, etaupdcov, etasmoothcov, ypriorcov, yupdcov, ysmoothcov,  
#' are returned in a row * column * time array. 
#' If plot=TRUE, nothing is returned but a plot is generated.
#' @examples
#' #Basic
#' ctStanKalman(ctstantestfit, timerange=c(0,60), timestep=.5, plot=TRUE)
#' 
#' #Multiple subjects, y and yprior, showing plot arguments
#' ctStanKalman(ctstantestfit, timerange=c(0,60), timestep=.1, plot=TRUE,
#'   subjects=2:3, 
#'   kalmanvec=c('y','yprior'),
#'   errorvec=c(NA,'ypriorcov'), #'auto' would also have achieved this
#'   ltyvec="auto",
#'   colvec='auto', 
#'   lwdvec='auto', 
#'   subsetindices=2, #Only plotting 2nd dimension of y and yprior
#'   pchvec='auto', typevec='auto',grid=TRUE,legend=TRUE,
#'   plotcontrol=list(xlim=c(0,55),main='Observations and priors'),
#'   polygoncontrol=list(density=20))
#' @export

ctStanKalman<-function(ctstanfitobj, datalong=NULL, timerange='asdata', timestep='asdata',
  subjects=1, plot=FALSE,...){
  
  out<-list()
  if(timerange[1] != 'asdata' & timestep[1] == 'asdata') stop('If timerange is not asdata, a timestep must be specified!')
  
  
  if(is.null(datalong)) { #get relevant data
    time<-ctstanfitobj$data$time
    datalong<-cbind(ctstanfitobj$data$subject,time,ctstanfitobj$data$Y)
    if(ctstanfitobj$ctstanmodel$n.TDpred > 0) datalong <- cbind(datalong,ctstanfitobj$data$tdpreds)
    
    colnames(datalong)<-c('subject','time',
      ctstanfitobj$ctstanmodel$manifestNames,
      ctstanfitobj$ctstanmodel$TDpredNames)
  }
  if(!all(subjects %in% datalong[,'subject'])) stop('Invalid subjects specified!')
  
  diffusionindices<-ctstanfitobj$data$diffusionindices
  
  for(subjecti in subjects){
    
    #setup subjects data, interpolating and extending as necessary
    sdat=datalong[datalong[,'subject'] == subjecti,,drop=FALSE]
    if(timestep != 'asdata' || timerange[1] != 'asdata') {
      if(timerange[1]=='asdata') stimerange <- range(sdat[,'time']) else {
        stimerange <- timerange
        if(timerange[1] > min(sdat[,'time']) || timerange[2] < max(sdat[,'time']) ) stop('Specified timerange must contain all subjects time ranges!')
      }
      snewtimes <- seq(stimerange[1],stimerange[2],timestep)
      snewdat <- array(NA,dim=c(length(snewtimes),dim(sdat)[-1]),dimnames=dimnames(sdat)) 
      snewdat[,'time'] <- snewtimes
      snewdat[,ctstanfitobj$ctstanmodel$TDpredNames] <- 0
      sdat <- rbind(sdat,snewdat)
      sdat<-sdat[!duplicated(sdat[,'time']),]
      sdat <- sdat[order(sdat[,'time']),]
    }
    
    #get parameter matrices
    model<-ctStanContinuousPars(ctstanfitobj, subjects=subjecti)
    
    #get kalman estimates
    out[[paste('subject',subjecti)]]<-ctKalman(kpars=model,
      datalong=sdat,
      manifestNames=ctstanfitobj$ctstanmodel$manifestNames,
      latentNames=ctstanfitobj$ctstanmodel$latentNames,
      TDpredNames=ctstanfitobj$ctstanmodel$TDpredNames,
      timecol='time',
      diffusionindices=diffusionindices)
  }
  
  if(plot) {
    ctStanKalmanPlot(x=out,subjects=subjects,...)
  } else return(out)
}


#' ctStanKalmanPlot
#' 
#' Plots Kalman filter output from ctStanKalman.
#'
#' @param x Output from \code{\link{ctStanKalman}}. In general it is easier to call 
#' \code{\link{ctStanKalman}} directly with the \code{plot=TRUE} argument, which calls this function.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param kalmanvec string vector of names of any elements of the output you wish to plot, 
#' the defaults of 'y' and 'yprior' plot the original data, 'y', 
#' and the prior from the Kalman filter for y. Replacing 'y' by 'eta' will 
#' plot latent variables instead (though 'eta' alone does not exist) and replacing 'prior' 
#' with 'upd' or 'smooth' respectively plotting updated (conditional on all data up to current time point)
#' or smoothed (conditional on all data) estimates.
#' @param errorvec vector of names of covariance elements to use for uncertainty indication 
#' around the kalmanvec items. 'auto' includes the latent covariance when plotting
#' latent states, and total covariance when plotting expectations of observed states.
#' @param errormultiply Numeric denoting the multiplication factor of the std deviation of errorvec objects. 
#' Defaults to 1.96, for 95\% credible intervals.
#' @param ltyvec vector of line types, varying over dimensions of the kalmanvec object.
#' @param colvec color vector, varying either over subject if multiple subjects, or otherwise over 
#' the dimensions of the kalmanvec object.
#' @param lwdvec vector of line widths, varying over the kalmanvec objects. 
#' @param subsetindices Either NULL, or vector of integers to use for subsetting the (columns) of kalmanvec objects.
#' @param pchvec vector of symbol types, varying over the dimensions of the kalmanvec object.
#' @param typevec vector of plot types, varying over the kalmanvec objects. 'auto' plots lines for
#' any  'prior', 'upd', or 'smooth' objects, and points otherwise.
#' @param grid Logical. Plot a grid?
#' @param add Logical. Create a new plot or update existing plot?
#' @param plotcontrol List of graphical arguments (see \code{\link{par}}), 
#' though lty,col,lwd,x,y, will all be ignored.
#' @param legend Logical, whether to include a legend if plotting.
#' @param legendcontrol List of arguments to the \code{\link{legend}} function.
#' @param polygoncontrol List of arguments to the \code{\link{polygon}} function for filling the uncertainty region.
#' @param polygonalpha Numeric for the opacity of the uncertainty region.
#' @return Nothing. Generates plots.
#' @export
#' @examples
#' ### Get output from ctStanKalman
#' x<-ctStanKalman(ctstantestfit,subjects=2)
#' 
#' ### Plot with ctStanKalmanPlot
#' ctStanKalmanPlot(x, subjects=2)
#' 
#' ###Single step procedure:
#' ctStanKalman(ctstantestfit,subjects=2,plot=TRUE)
ctStanKalmanPlot<-function(x, subjects, kalmanvec=c('y','yprior'),
  errorvec='auto', errormultiply=2,
  ltyvec="auto",colvec='auto', lwdvec='auto', 
  subsetindices=NULL,pchvec='auto', typevec='auto',grid=TRUE,add=FALSE, 
  plotcontrol=list(ylab='Value',xlab='Time'),
  polygoncontrol=list(border=NA),polygonalpha=.1,
  legend=TRUE, legendcontrol=list(x='topright',bg='white')){
  
  out<-x
  
  if(is.null(plotcontrol$xlim)) plotcontrol$xlim <- range(sapply(out,function(x) x$time))
  
  
  if(is.null(plotcontrol$ylim)) {
    plotcontrol$ylim <- range(unlist(lapply(out,function(x) {
    if(!is.null(x)){
    ret<-c()
    
    for(kveci in kalmanvec){
      ret<-c(ret,x[[kveci]][,
        if(is.null(subsetindices)) 1:dim(x[[kveci]])[2] else subsetindices]
        )
    }
    return(ret)}})),na.rm=TRUE)
    #extend range somewhat...
   
    plotcontrol$ylim[1] = plotcontrol$ylim[1] - (plotcontrol$ylim[2] - plotcontrol$ylim[1])/5
    plotcontrol$ylim[2] = plotcontrol$ylim[2] + (plotcontrol$ylim[2] - plotcontrol$ylim[1])/5
  }
  

  if(length(subjects) > 1 & colvec[1] =='auto') colvec = rainbow(length(subjects))
  if(length(subjects) == 1 & colvec[1] =='auto') colvec = rainbow(length(kalmanvec))
  
  if(rl(lwdvec[1]=='auto')) lwdvec=rep(2,length(kalmanvec))
  
  if(is.null(plotcontrol$ylab)) plotcontrol$ylab='Value'
  if(is.null(plotcontrol$xlab)) plotcontrol$xlab='Time'
  
  if(rl(typevec[1]=='auto')) typevec=c('p','l')[grepl("prior|upd|smooth|eta",kalmanvec)+1]
  
  if(rl(errorvec[1]=='auto')) {
    errorvec=rep(NA,length(kalmanvec))
    errorvec[grepl("prior|upd|smooth|eta",kalmanvec)]<-paste0(
      kalmanvec[grepl("prior|upd|smooth|eta",kalmanvec)],'cov')
}

  legendtext<-c()
  legendcol <- c()
  legendlty<-c()
  legendpch<-c()
  
  if(length(subjects) > 1 && length(unique(colvec))>1) { #include subject color in legend if necessary
      legendtext<-c(legendtext,paste('Subject', subjects))
      legendcol <- c(legendcol,colvec)
      legendlty <-c(legendlty,rep(0,length(subjects)))
      legendpch <-c(legendpch,rep(NA,length(subjects)))
    }
  
  
  for(si in 1:length(subjects)){#subjects
    subjecti = subjects[si]
    subiname=paste('subject',subjecti)
    plist<-plotcontrol
    if(length(subjects) > 1) {
      plist$col = colvec[si] #set colour based on subject if multiple subjects
    }
    
    for(kveci in 1:length(kalmanvec)){ #kalman output types
      kvecdims=1:dim(out[[subiname]][[kalmanvec[kveci]]])[-1]
      if(any(subsetindices > max(kvecdims))) stop('subsetindices contains a value greater than relevant dimensions of object in kalmanvec!')
      if(!is.null(subsetindices)) kvecdims=kvecdims[subsetindices]
      if(rl(ltyvec[1]=='auto')) ltyvec <- 1:length(kvecdims)
      
      if(rl(pchvec[1] =='auto')) pchvec = 1:(length(kvecdims))
      
      # if((length(unique(pchvec[typevec!='l']))>1) || 
      #     (length(subjects) == 1 && length(unique(colvec)) > 1)) { #if changing pch, then legend needs to show pch elements along with lty and maybe colour
      #   legendtext<-c(legendtext,paste(kalmanvec[kveci],
      #     colnames(out[[subiname]][[kalmanvec[kveci]]])[kdimi]))
      #   legendlty <- c(legendlty,ifelse(plist$type=='p',0,ltyvec[dimi]))
      #   legendpch <- c(legendpch,ifelse(plist$type=='l',NA,plist$pch))
      #   if(length(subjects) == 1) legendcol = c(legendcol,plist$col) else legendcol=c(legendcol,'black')
      # }
      
      for(dimi in 1:length(kvecdims)){ #dimensions of kalman matrix
        kdimi <- kvecdims[dimi]
        plist$x=out[[subiname]]$time
        plist$y=out[[subiname]][[kalmanvec[kveci]]][,kdimi] 
        plist$lwd=lwdvec[kveci]
        plist$lty=ltyvec[dimi] 
        plist$pch=pchvec[dimi]
        plist$type=typevec[kveci]
        if(length(subjects)==1) plist$col=colvec[dimi]
        
        
        if(subjecti == subjects[1] & kveci==1 && dimi == 1 && !add) {
          do.call(graphics::plot.default,plist) 
          if(grid) grid()
        } else do.call(graphics::points.default,plist) 
        
        if(!is.na(errorvec[kveci])){
          if(is.null(out[[subiname]][[errorvec[kveci]]])) stop('Invalid errorvec specified!')
          backwardstimesindex=order(plist$x,decreasing=TRUE)

          # if(is.null(polygoncontrol$angle)) 
            # polygoncontrol$angle=stats::runif(1,0,359)
            polygonargs<-polygoncontrol
            polygonargs$x=c(plist$x,plist$x[backwardstimesindex])
            polygonargs$y=c(plist$y + errormultiply * sqrt(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,]), 
              (plist$y - errormultiply * sqrt(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,]))[backwardstimesindex])
            polygonargs$col=grDevices::adjustcolor(plist$col,alpha.f=polygonalpha)
            do.call(graphics::polygon,polygonargs)
          }
        
        #if changing lty then legend needs lty types
        if(length(unique(ltyvec))>1 && subjecti == subjects[1]) {
          legendtext<-c(legendtext,paste(kalmanvec[kveci],
            colnames(out[[subiname]][[kalmanvec[kveci]]])[kdimi]))
          legendlty <- c(legendlty,ifelse(plist$type=='p',0,ltyvec[dimi]))
          legendpch <- c(legendpch,ifelse(plist$type=='l',NA,pchvec[dimi]))
          if(length(subjects) == 1) legendcol = c(legendcol,plist$col) else legendcol=c(legendcol,'black')
        }
      }
    }
  }
  
  if(legend && length(legendtext)>0){
    legendcontrol$legend<-legendtext
    legendcontrol$col<-legendcol
    legendcontrol$text.col <- legendcol
    legendcontrol$pch <- legendpch
    legendcontrol$lty <- legendlty  
    do.call(graphics::legend,legendcontrol)
  }
}



