#' Converting data to WOE
#'
#' \code{woe_trans} is for converting dat to woe.
#' The \code{woe_trans_all} function is a simpler wrapper for \code{woe_trans}.
#' @param dat A data.frame with independent variables.
#' @param target The name of target variable. Default is NULL.
#' @param x_list A list of x variables.
#' @param x  The name of an independent variable.
#' @param ex_cols Names of excluded variables. Regular expressions can also be used to match variable names. Default is NULL.
#' @param breaks_list  A list contains breaks of variables. it is generated by code{\link{get_breaks_all}},code{\link{get_breaks}}
#' @param bins_table A table contians woe of each bin of variables, it is generated by code{\link{get_bins_table_all}},code{\link{get_bins_table}}
#' @param note   Logical, outputs info. Default is TRUE.
#' @param parallel Logical, parallel computing. Default is FALSE.
#' @param woe_name Logical. Add "_woe" at the end of the variable name.
#' @param save_data Logical, save results in locally specified folder. Default is TRUE
#' @param file_name The name for periodically saved woe file. Default is "dat_woe".
#' @param dir_path The path for periodically saved woe file Default is "./data"
#' @param ...  Additional parameters.
#' @return A list of breaks for each variables.
#' @seealso \code{\link{get_tree_breaks}}, \code{\link{cut_equal}}, \code{\link{select_best_class}}, \code{\link{select_best_breaks}}
#' @examples
#' sub = cv_split(UCICreditCard, k = 30)[[1]]
#' dat = UCICreditCard[sub,]
#' dat = re_name(dat, "default.payment.next.month", "target")
#' dat = cleaning_data(dat, target = "target", obs_id = "ID", occur_time = "apply_date",
#' miss_values =  list("", -1))
#' 
#' train_test <- train_test_split(dat, split_type = "OOT", prop = 0.7,
#'                                 occur_time = "apply_date")
#' dat_train = train_test$train
#' dat_test = train_test$test
#' #get breaks of all predictive variables
#' x_list = c("PAY_0", "LIMIT_BAL", "PAY_AMT5", "EDUCATION", "PAY_3", "PAY_2")
#' breaks_list <- get_breaks_all(dat = dat_train, target = "target",
#'                               x_list = x_list, occur_time = "apply_date", ex_cols = "ID", 
#' save_data = FALSE, note  = FALSE)
#' #woe transform
#' train_woe = woe_trans_all(dat = dat_train,
#'                           target = "target",
#'                           breaks_list = breaks_list,
#'                           woe_name = FALSE)
#' test_woe = woe_trans_all(dat = dat_test,
#'                        target = "target",
#'                          breaks_list = breaks_list,
#'                          note = FALSE)
#'
#' @export


woe_trans_all <- function(dat, x_list = NULL, ex_cols = NULL, bins_table = NULL,
                          target = NULL, breaks_list = NULL, note = FALSE,
                          save_data = FALSE, parallel = FALSE,woe_name = FALSE,
                          file_name = NULL, dir_path = tempdir(), ...) {

    if (note) {
        cat(paste("[NOTE]  converting all variables to woe...."), "\n")
    }

    opt = options(stringsAsFactors = FALSE) #
    if (is.null(x_list)) {
        if (!is.null(bins_table)) {
            x_list = unique(bins_table[which(as.character(bins_table[, "Feature"]) != "Total"), "Feature"])
        } else {
            x_list = get_names(dat = dat,
                               types = c('factor', 'character', 'numeric', 'integer', 'double'),
                               ex_cols = c(target,ex_cols), get_ex = FALSE)
        }
    }
    ex_vars = get_names(dat = dat, types = c('factor', 'character', 'numeric', 'integer', 'double'),
                        ex_cols = x_list, get_ex = FALSE)
    dat_woe = loop_function(func = woe_trans, x_list = x_list,
                            args = list(dat = dat, bins_table = bins_table,
                                        target = target, breaks_list = breaks_list,
                                        woe_name = woe_name),
                            bind = "cbind", parallel = parallel)
    dat = cbind(dat[ex_vars], dat_woe)
    if (save_data) {
        dir_path = ifelse(is.null(dir_path) || !is.character(dir_path) || !grepl('.|/|:', dir_path),
                      tempdir(), dir_path)
        if (!dir.exists(dir_path)) dir.create(dir_path)
        if (!is.character(file_name)) file_name = NULL
        save_dt(dat, file_name = ifelse(is.null(file_name), "dat.woe", paste(file_name, "dat.woe", sep = ".")), dir_path = dir_path, note = note)
    }
    options(opt) # reset
    return(dat)
}

#' @rdname woe_trans_all
#' @export

woe_trans <- function(dat, x, bins_table = NULL, target = NULL, breaks_list = NULL, woe_name = TRUE) {
    # bins_table
    if (is.null(bins_table)) {
        if (!is.null(breaks_list)) {
            bins_table =  get_bins_table(dat = dat, x = x,
                                         target = target, breaks_list = breaks_list,
                                         note = FALSE)
        } else {
            stop("bins_table & breaks_list are both missing.\n")
        }
    }
    bins_tbl = bins_table[which(as.character(bins_table[, "Feature"]) == names(dat[x])),
                          c("Feature", "cuts", "bins", "woe")]
    if (woe_name) {
        woe_names = paste(names(dat[x]), "woe", sep = "_")
    } else {
        woe_names = names(dat[x])
    }
    if (length(bins_tbl) > 0 && all(as.character(bins_tbl[, "Feature"]) != "Total")) {
        bins = split_bins(dat = dat, x = x, breaks = bins_tbl[, c("cuts")], bins_no = TRUE)
        for (i in 1:length(unique(bins))) {
            dat[as.character(bins) == as.character(bins_tbl[i, "bins"]), woe_names] = bins_tbl[i, "woe"]
        }
        dat[,woe_names] = as.numeric(dat[,woe_names])
    }
    return(dat[woe_names])
}
