% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smqr.R
\name{conquer}
\alias{conquer}
\title{Convolution-Type Smoothed Quantile Regression}
\usage{
conquer(
  X,
  Y,
  tau = 0.5,
  kernel = c("Gaussian", "uniform", "parabolic", "triangular"),
  h = 0,
  standardize = TRUE,
  tol = 1e-04,
  iteMax = 5000,
  ci = FALSE,
  alpha = 0.05,
  B = 1000
)
}
\arguments{
\item{X}{A \eqn{n} by \eqn{p} design matrix. Each row is a vector of observation with \eqn{p} covariates. Number of observations \eqn{n} must be greater than number of covariates \eqn{p}.}

\item{Y}{An \eqn{n}-dimensional response vector.}

\item{tau}{(\strong{optional}) The desired quantile level. Default is 0.5. Value must be between 0 and 1.}

\item{kernel}{(\strong{optional})  A character string specifying the choice of kernel function. Default is "Gaussian". Other choices are "Gaussian", "uniform", "parabolic" or "triangular".}

\item{h}{(\strong{optional}) The bandwidth parameter for kernel smoothing. Default is \eqn{max{((log(n) + p) / n)^0.4, 0.05}}. The default will be used if the input value is less than 0.05.}

\item{standardize}{(\strong{optional}) A logical flag. Default is TRUE. If \code{standardize = TRUE}, then the design matrix will be standardized such that each column has mean zero and standard deviation one.}

\item{tol}{(\strong{optional}) Tolerance level of the gradient descent algorithm. The gradient descent algorithm terminates when the maximal entry of the gradient is less than \code{tol}. Default is 1e-05.}

\item{iteMax}{(\strong{optional}) Maximum number of iterations. Default is 5000.}

\item{ci}{(\strong{optional}) A logical flag. Default is FALSE. If \code{ci = TRUE}, then three types of confidence intervals (percentile, pivotal and normal) will be constructed via multiplier bootstrap.}

\item{alpha}{(\strong{optional}) The nominal level for (1-\eqn{alpha})-confidence intervals. Default is 0.05. The input value must be in \eqn{(0, 1)}.}

\item{B}{(\strong{optional}) The size of bootstrap samples. Default is 1000.}
}
\value{
An object containing the following items will be returned:
\describe{
\item{\code{coeff}}{A \eqn{(p + 1)}-vector of estimated quantile regression coefficients, including the intercept.}
\item{\code{ite}}{The number of iterations of the gradient descent algorithm for convergence.}
\item{\code{residual}}{The residuals of the quantile regression fit.}
\item{\code{bandwidth}}{The value of smoothing bandwidth.}
\item{\code{tau}}{The desired quantile level.}
\item{\code{kernel}}{The choice of kernel function.}
\item{\code{n}}{The sample size.}
\item{\code{p}}{The dimension of the covariates.}
\item{\code{perCI}}{The percentile confidence intervals for regression coefficients. Not available if \code{ci = FALSE}.}
\item{\code{pivCI}}{The pivotal confidence intervals for regression coefficients. Not available if \code{ci = FALSE}}
\item{\code{normCI}}{The normal-based confidence intervals for regression coefficients. Not available if \code{ci = FALSE}}
}
}
\description{
Fit a smoothed quantile regression via convolution-type smoothing method. The solution is computed using gradient descent with Barzilai-Borwein step size. Constructs (1-\eqn{alpha}) confidence intervals with multiplier bootstrap.
}
\examples{
n = 500; p = 10
beta = rep(1, p)
X = matrix(rnorm(n * p), n, p)
Y = 1 + X \%*\% beta + rt(n, 2)

## Smoothed quantile regression with Gaussian kernel
fit.Gauss = conquer(X, Y, tau = 0.5, kernel = "Gaussian")
beta.hat.Gauss = fit.Gauss$coeff

## Smoothe quantile regression with uniform kernel
fit.unif = conquer(X, Y, tau = 0.5, kernel = "uniform")
beta.hat.unif = fit.unif$coeff

## Construct three types of confidence intervals via multiplier bootstrap
fit = conquer(X, Y, tau = 0.5, kernel = "Gaussian", ci = TRUE)
ci.per = fit$perCI
ci.piv = fit$pivCI
ci.norm = fit$normCI
}
\references{
Barzilai, J. and Borwein, J. M. (1988). Two-point step size gradient methods. IMA J. Numer. Anal. 8 141–148.

Fernandes, M., Guerre, E. and Horta, E. (2019). Smoothing quantile regressions. J. Bus. Econ. Statist., in press.

He, X., Pan, X., Tan, K. M., and Zhou, W.-X. (2020). Smoothed quantile regression: Fast computation and bootstrap inference. Preprint.

Koenker, R. and Bassett, G. (1978). Regression quantiles. Econometrica 46 33-50.
}
\author{
Xuming He <xmhe@umich.edu>, Xiaoou Pan <xip024@ucsd.edu>, Kean Ming Tan <keanming@umich.edu>, and Wen-Xin Zhou <wez243@ucsd.edu>
}
