% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crate.R
\name{crate}
\alias{crate}
\title{Crate a function to share with another process}
\usage{
crate(
  .fn,
  ...,
  .parent_env = baseenv(),
  .error_arg = ".fn",
  .error_call = environment()
)
}
\arguments{
\item{.fn}{A fresh formula or function. "Fresh" here means that
they should be declared in the call to \code{crate()}. See examples if
you need to crate a function that is already defined. Formulas
are converted to purrr-like lambda functions using
\code{\link[rlang:as_function]{rlang::as_function()}}.}

\item{...}{Named arguments to declare in the environment of \code{.fn}.}

\item{.parent_env}{The default of \code{baseenv()} ensures that the evaluation
environment of the crate is isolated from the search path. Specifying
another environment such as the global environment allows this condition to
be relaxed (but at the expense of no longer being able to rely on a local
run giving the same results as one in a different process).}

\item{.error_arg}{An argument name as a string. This argument
will be mentioned in error messages as the input that is at the
origin of a problem.}

\item{.error_call}{The execution environment of a currently
running function, e.g. \code{caller_env()}. The function will be
mentioned in error messages as the source of the error. See the
\code{call} argument of \code{\link[rlang:abort]{abort()}} for more information.}
}
\description{
\code{crate()} creates functions in a self-contained environment
(technically, a child of the base environment). This has two
advantages:
\itemize{
\item They can easily be executed in another process.
\item Their effects are reproducible. You can run them locally with the
same results as on a different process.
}

Creating self-contained functions requires some care, see section
below.
}
\section{Creating self-contained functions}{

\itemize{
\item They should call package functions with an explicit \code{::}
namespace. This includes packages in the default search path with
the exception of the base package. For instance \code{var()} from the
stats package must be called with its namespace prefix:
\code{stats::var(x)}.
\item They should declare any data they depend on. You can declare data
by supplying additional arguments or by unquoting objects with \verb{!!}.
}
}

\examples{
# You can create functions using the ordinary notation:
crate(function(x) stats::var(x))

# Or the formula notation:
crate(~ stats::var(.x))

# Declare data by supplying named arguments. You can test you have
# declared all necessary data by calling your crated function:
na_rm <- TRUE
fn <- crate(~ stats::var(.x, na.rm = na_rm))
try(fn(1:10))

# For small data it is handy to unquote instead. Unquoting inlines
# objects inside the function. This is less verbose if your
# function depends on many small objects:
fn <- crate(~ stats::var(.x, na.rm = !!na_rm))
fn(1:10)

# One downside is that the individual sizes of unquoted objects
# won't be shown in the crate printout:
fn


# The function or formula you pass to crate() should defined inside
# the crate() call, i.e. you can't pass an already defined
# function:
fn <- function(x) toupper(x)
try(crate(fn))

# If you really need to crate an existing function, you can
# explicitly set its environment to the crate environment with the
# set_env() function from rlang:
crate(rlang::set_env(fn))
}
