% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_spatial_autocor.R
\name{cv_spatial_autocor}
\alias{cv_spatial_autocor}
\title{Measure spatial autocorrelation in spatial response data or predictor raster files}
\usage{
cv_spatial_autocor(
  r,
  x,
  column = NULL,
  num_sample = 5000L,
  deg_to_metre = 111325,
  plot = TRUE,
  progress = TRUE,
  ...
)
}
\arguments{
\item{r}{a terra SpatRaster object. If provided (and \code{x} is missing), it will be used for to calculate range.}

\item{x}{a simple features (sf) or SpatialPoints object of spatial sample data (e.g., species binary or continuous date).}

\item{column}{character; indicating the name of the column in which response variable (e.g. species data as a binary
response i.e. 0s and 1s) is stored for calculating spatial autocorrelation range. This supports multiple column names.}

\item{num_sample}{integer; the number of sample points of each raster layer to fit variogram models. It is 5000 by default,
however it can be increased by user to represent their region well (relevant to the extent and resolution of rasters).}

\item{deg_to_metre}{integer. The conversion rate of degrees to metres.}

\item{plot}{logical; whether to plot the results.}

\item{progress}{logical; whether to shows a progress bar.}

\item{...}{additional option for \code{\link{cv_plot}}}
}
\value{
An object of class S3. A list object including:
    \itemize{
    \item{range - the suggested range (i.e. size), which is the median of all calculated ranges in case of 'r'.}
    \item{range_table - a table of input covariates names and their autocorrelation range}
    \item{plots - the output plot (the plot is shown by default)}
    \item{num_sample - number sample of 'r' used for analysis}
    \item{variograms - fitted variograms for all layers}
    }
}
\description{
This function provides a quantitative basis for choosing block size. The spatial autocorrelation in either the
spatial sample points or all continuous predictor variables available as raster layers is assessed and reported.
The response (as defined be \code{column}) in spatial sample points can be binary such as species distribution data,
or continuous response like soil organic carbon. The function estimates spatial autocorrelation \emph{ranges} of all input
raster layers or the response data. This is the range over which observations are independent and is determined by
constructing the empirical variogram, a fundamental geostatistical tool for measuring spatial autocorrelation.
The empirical variogram models the structure of spatial autocorrelation by measuring variability between all possible
pairs of points (O'Sullivan and Unwin, 2010). Results are plotted. See the details section for further information.
}
\details{
The input raster layers should be continuous for computing the variograms and estimating the range of spatial
autocorrelation. The input rasters should also have a specified coordinate reference system. However, if the reference
system is not specified, the function attempts to guess it based on the extent of the map. It assumes an un-projected
reference system for layers with extent lying between -180 and 180.

Variograms are calculated based on the distances between pairs of points, so un-projected rasters (in degrees) will
not give an accurate result (especially over large latitudinal extents). For un-projected rasters, \emph{the great circle distance}
(rather than Euclidean distance) is used to calculate the spatial distances between pairs of points. To
enable more accurate estimate, it is recommended to transform un-projected maps (geographic coordinate
system / latitude-longitude) to a projected metric reference system (e.g. UTM or Lambert) where it is possible.
See \code{\link[automap]{autofitVariogram}} from \pkg{automap} and \code{\link[gstat]{variogram}} from \pkg{gstat} packages
for further information.
}
\examples{
\donttest{
library(blockCV)

# import presence-absence species data
points <- read.csv(system.file("extdata/", "species.csv", package = "blockCV"))
# make an sf object from data.frame
pa_data <- sf::st_as_sf(points, coords = c("x", "y"), crs = 7845)

# load raster data
path <- system.file("extdata/au/", package = "blockCV")
files <- list.files(path, full.names = TRUE)
covars <- terra::rast(files)

# spatial autocorrelation of a binary/continuous response
sac1 <- cv_spatial_autocor(x = pa_data,
                           column = "occ", # binary or continuous data
                           plot = TRUE)


# spatial autocorrelation of continuous raster files
sac2 <- cv_spatial_autocor(r = covars,
                           num_sample = 5000,
                           plot = TRUE)

# show the result
summary(sac2)
}
}
\references{
O'Sullivan, D., Unwin, D.J., (2010). Geographic Information Analysis, 2nd ed. John Wiley & Sons.

Roberts et al., (2017). Cross-validation strategies for data with temporal, spatial, hierarchical,
or phylogenetic structure. Ecography. 40: 913-929.
}
\seealso{
\code{\link{cv_block_size}}
}
