#' Create the email message body
#'
#' The `compose_email()` function allows us to easily create an email message.
#' We can incorporate character vectors into the message body, the header, or
#' the footer.
#'
#' @param header,body,footer The three layout sections for an email message
#'   (ordered from top to bottom). Markdown text can be supplied to each of
#'   these by using the [md()] text helper function. Alternatively, we can
#'   supply a set of `block_*()` calls enclosed within the [blocks()] function
#'   to take advantage of precomposed HTML blocks.
#' @param title The title of the email message. This is not the subject but the
#'   HTML title text which may appear in limited circumstances.
#' @param template An email template function to use. The default is
#'   [blastula_template()].
#' @param ... Additional arguments to pass to the `template` function. If you're
#'   using the default template, you can use `font_family` to control the base
#'   font, and `content_width` to control the width of the main content; see
#'   [blastula_template()]. By default, the `content_width` is set to `1000px`.
#'   Using widths less than `600px` is generally not advised but, if necessary,
#'   be sure to test such HTML emails with a wide range of email clients before
#'   sending to the intended recipients. The Outlook mail client (Windows,
#'   Desktop) does not respect `content_width`.
#'
#' @examples
#' # Create a simple email message using
#' # Markdown-formatted text in the `body`
#' # and `footer` sections with the `md()`
#' # text helper function
#' email <-
#'   compose_email(
#'     body = md(
#' "
#' ## Hello!
#'
#' This is an email message that was generated by the blastula package.
#'
#' We can use **Markdown** formatting with the `md()` function.
#'
#' Cheers,
#'
#' The blastula team
#' "),
#'   footer = md(
#' "
#' sent via the [blastula](https://rstudio.github.io/blastula) R package
#' ")
#' )
#'
#' # The email message can always be
#' # previewed by calling the object
#' if (interactive()) email
#'
#' @return An `email_message` object.
#' @export
compose_email <- function(
    body = NULL,
    header = NULL,
    footer = NULL,
    title = NULL,
    ...,
    template = blastula_template
) {

  body <- process_markdown(body)
  header <- process_markdown(header)
  footer <- process_markdown(footer)

  # Define the title text for the email;
  # use an empty string if not supplied
  title <- title %||% ""

  # Generate the email message body
  body <-
    template(
      title = title,
      html_header = header,
      html_body = body,
      html_footer = footer,
      ...
    ) %>%
    # In case the template used md() internally
    process_markdown() %>%
    as.character()

  email_message <- cid_images(html = body)
  email_message
}
