% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/snmatch.R
\name{snmatch}
\alias{snmatch}
\title{Fit a skew-normal distribution to a target density.}
\usage{
snmatch(x, y, p = c(0.025, 0.5, 0.975))
}
\arguments{
\item{x}{A numeric vector on the domain of the target density.}

\item{y}{The y-coordinates of the target density on grid x.}

\item{p}{Vector of probabilities at which to compute quantiles of the
skew-normal fit.}
}
\value{
A list with the following components:

\item{location}{Estimated location parameter.}

\item{scale}{Estimated scale parameter.}

\item{shape}{Estimated shape parameter.}

\item{snfit}{Fitted values of the skew-normal density
  computed on an equally spaced grid between min(x)
  and max(x).}

\item{quant}{Vector of quantiles of the skew-normal fit
  computed on the input vector of probabilities p.}

\item{xgrid}{Equidistant grid on which the skew-normal fitted
  density is computed.}
}
\description{
The routine fits a skew-normal univariate distribution to a target density.
Parameters of the resulting skew-normal fit are estimated by the method
of moments.
}
\details{
The skew-normal density is parameterized by a location parameter
\eqn{\mu}, a scale parameter \eqn{\omega} and a shape parameter
\eqn{\rho} that regulates skewness. The probability density function at any
x on the real line is:

\deqn{p(x) = (2/\omega) \phi((x-\mu)/\omega) \psi(\rho (x-\mu)/\omega),}

where \eqn{\phi()} and \eqn{\psi()} denote the standard Gaussian density and
cumulative distribution function respectively (see Azzalini 2018).
The first moment and second and third central moments of the target density
are computed based on the x, y coordinates using the trapezoidal rule
and matched against the theoretical moments of a skew-normal distribution.
The solution to this system of equations is the method of moment estimate of
the location, scale and shape parameters of a skew-normal density.
}
\examples{
# Pdf of skew-normal density
sn.target <- function(x, location, scale, shape){
               val <- 2 * stats::dnorm(x, mean = location, sd = scale) *
               pnorm(shape * (x - location) / scale)
               return(val)
              }

# Extract x and y coordinates from target
x.grid <- seq(-2, 6, length = 200)
y.grid <- sapply(x.grid, sn.target, location = 0, scale = 2, shape = 3)

# Computation of the fit and graphical illustration
fit <- snmatch(x.grid, y.grid)
domx <- seq(-2, 6, length = 1000)
plot(domx, sapply(domx, sn.target, location = 0, scale = 2, shape = 3),
     type = "l", ylab = "f(x)", xlab = "x", lwd= 2)
lines(fit$xgrid, fit$snfit, type="l", col = "red", lwd = 2, lty = 2)
legend("topright", lty = c(1,2), col = c("black", "red"), lwd = c(2, 2),
       c("Target","SN fit"), bty="n")

# Extract estimated parameters
fit$location # Estimated location parameter
fit$scale    # Estimated scale parameter
fit$shape    # Estimated shape parameter

}
\references{
Azzalini, A. (2018). The Skew-Normal and Related families.
\emph{Cambridge University Press}.
}
\author{
Oswaldo Gressani \email{oswaldo_gressani@hotmail.fr}.
}
