% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integrate_to_ppi.R
\name{integrate_to_ppi}
\alias{integrate_to_ppi}
\title{Calculate a plan position indicator (\code{ppi}) of vertically integrated density adjusted for range effects}
\usage{
integrate_to_ppi(
  pvol,
  vp,
  nx = 100,
  ny = 100,
  xlim,
  ylim,
  zlim = c(0, 4000),
  res,
  quantity = "eta",
  param = "DBZH",
  raster = NA,
  lat,
  lon,
  antenna,
  beam_angle = 1,
  crs,
  param_ppi = c("VIR", "VID", "R", "overlap", "eta_sum", "eta_sum_expected"),
  k = 4/3,
  re = 6378,
  rp = 6357
)
}
\arguments{
\item{pvol}{a polar volume of class pvol}

\item{vp}{a vertical profile of class vp}

\item{nx}{number of raster pixels in the x (longitude) dimension}

\item{ny}{number of raster pixels in the y (latitude) dimension}

\item{xlim}{x (longitude) range}

\item{ylim}{y (latitude) range}

\item{zlim}{altitude range in meter, given as a numeric vector of length two.}

\item{res}{numeric vector of length 1 or 2 to set the resolution of the raster (see \link[raster]{res}).
If this argument is used, arguments \code{nx} and \code{ny} are ignored. Unit is identical to \code{xlim} and \code{ylim}.}

\item{quantity}{profile quantity on which to base range corrections, 'eta' or 'dens'.}

\item{param}{reflectivity factor scan parameter on which to base range corrections.
Typically the same parameter from which animal densities are estimated for object \code{vp}.
One of 'DBZH','DBZV','DBZ','TH','TV'.}

\item{raster}{(optional) RasterLayer with a CRS. When specified this raster topology is used for the output, and nx, ny, res
arguments are ignored.}

\item{lat}{Geodetic latitude of the radar in degrees. If missing taken from \code{pvol}.}

\item{lon}{Geodetic latitude of the radar in degrees. If missing taken from \code{pvol}.}

\item{antenna}{radar antenna height. If missing taken from \code{vp}}

\item{beam_angle}{numeric. Beam opening angle in degrees, typically the
angle between the half-power (-3 dB) points of the main lobe}

\item{crs}{character or object of class CRS. PROJ.4 type description of a Coordinate Reference System (map projection).
When 'NA' (default), an azimuthal equidistant projection with origin at the radar location is used.
To use a WSG84 (lat,lon) projection, use crs="+proj=longlat +datum=WGS84"}

\item{param_ppi}{one or multiple of 'VIR', 'VID', 'R', 'overlap', 'eta_sum', 'eta_sum_expected'}

\item{k}{Standard refraction coefficient.}

\item{re}{Earth equatorial radius in km.}

\item{rp}{Earth polar radius in km.}
}
\value{
An object of class '\link[=summary.ppi]{ppi}'.
}
\description{
This function estimates a spatial image (PPI object) of vertically integrated
density (\code{VID}) based on all elevation scans of the radar, while
accounting for the changing overlap between the radar beams as a function of
range. The resulting PPI is a vertical integration over the layer of
biological scatterers based on all available elevation scans, corrected for
range effects due to partial beam overlap with the layer of biological echoes
(overshooting) at larger distances from the radar. The methodology is
described in detail in Kranstauber et al. (2020).
}
\details{
The function requires
\itemize{
\item a polar volume, containing one or multiple scans (\code{pvol})
\item a vertical profile (of birds) calculated for that same polar volume (\code{vp})
\item a grid defined on the earth's surface, on which we will calculate the range corrected image
(defined by \code{raster}, or a combination of \code{nx},\code{ny},\code{res} arguments).
}
The pixel locations on the ground are easily translated into a corresponding azimuth and range of
the various scans (see function \link{beam_range}).

For each scan within the polar volume, the function calculates:
\enumerate{
\item the vertical radiation profile for each ground surface pixel for that particular scan,
using \link{beam_profile}.
\item the reflectivity expected for each ground surface pixel (\eqn{\eta_{expected}}),
given the vertical profile (of biological scatterers) and the part of the profile radiated
by the beam. This \eqn{\eta_{expected}} is simply the average of
(linear) \code{eta} in the profile, weighted by the vertical radiation profile.
\item the observed eta at each pixel \eqn{\eta_{observed}},
which is converted form \code{DBZH} using function \link{dbz_to_eta},
with \code{DBZH} the reflectivity factor measured at the pixel's distance from the radar.
}

For each pixel on the ground, we thus retrieve a set of \eqn{\eta_{expected}}
and a set of \eqn{\eta_{observed}}. From those we can calculate a spatial adjustment factor
\code{R} as:

\deqn{R=\sum{\eta_{observed}}/\sum{\eta_{expected}}},
with the sum running over scans.

To arrive at the final PPI image, the function calculates
\itemize{
\item the vertically integrated density (\code{vid}) and vertically integrated
reflectivity (\code{vir}) for the profile,
using the function \link{integrate_profile}.
\item the spatial range-corrected PPI for \code{VID}, defined as the adjustment
factor image (\code{R}), multiplied by the \code{vid}
calculated for the profile
\item the spatial range-corrected PPI for \code{VIR}, defined as the
adjustment factor \code{R}, multiplied by the \code{vir} calculated for the profile.
}

If one of \code{lat} or \code{lon} is missing, the extent of the PPI is taken equal to
the extent of the data in the first scan of the polar volume.

As an additional parameter, overlap between vertical profile and vertical radiation
profile is calculated using \link{beam_profile}
and stored as quantity \code{overlap}.

scans at 90 degree beam elevation (birdbath scans) are ignored.
}
\examples{
# locate example polar volume file:
pvolfile <- system.file("extdata", "volume.h5", package = "bioRad")

# load polar volume
example_pvol <- read_pvolfile(pvolfile)

# load the corresponding vertical profile for this polar volume
data(example_vp)

# calculate the range-corrected ppi on a 50x50 pixel raster
my_ppi <- integrate_to_ppi(example_pvol, example_vp, nx = 50, ny = 50)

# plot the vertically integrated reflectivity (VIR) using a 0-2000 cm^2/km^2 color scale:
plot(my_ppi, zlim = c(0, 2000))

\dontrun{
# calculate the range-corrected ppi on finer 2000m x 2000m pixel raster:
my_ppi <- integrate_to_ppi(example_pvol, example_vp, res = 2000)

# plot the vertically integrated density (VID) using a 0-200 birds/km^2 color scale:
plot(my_ppi, param = "VID", zlim = c(0, 200))

# to overlay ppi objects on a background map, first
# download a basemap, and map the ppi:
bm <- download_basemap(my_ppi)
map(my_ppi, bm)

# the ppi can also be projected on a user-defined raster, as follows:
# first define the raster:
template_raster <- raster::raster(raster::extent(12, 13, 56, 57), crs = sp::CRS("+proj=longlat"))

# project the ppi on the defined raster:
my_ppi <- integrate_to_ppi(example_pvol, example_vp, raster = template_raster)

# extract the raster data from the ppi object:
raster::brick(my_ppi$data)

# calculate the range-corrected ppi on an even finer 500m x 500m pixel raster,
# cropping the area up to 50000 meter from the radar.
my_ppi <- integrate_to_ppi(example_pvol, example_vp,
  res = 500,
  xlim = c(-50000, 50000), ylim = c(-50000, 50000)
)
plot(my_ppi, param = "VID", zlim = c(0, 200))
}
}
\references{
\itemize{
\item Kranstauber B, Bouten W, Leijnse H, Wijers B, Verlinden L,
Shamoun-Baranes J, Dokter AM (2020) High-Resolution Spatial Distribution of
Bird Movements Estimated from a Weather Radar Network. Remote Sensing 12 (4), 635.
\url{https://doi.org/10.3390/rs12040635}
\item Buler JJ & Diehl RH (2009) Quantifying bird density during migratory
stopover using weather surveillance radar. IEEE Transactions on Geoscience
and Remote Sensing 47: 2741-2751.
\url{https://doi.org/10.1109/TGRS.2009.2014463}
}
}
